#!/usr/bin/env/R
####### ACSN enrichment analysis #######
####### Single sample analysis / single cohort analysis #######

#' Gene set enrichment analysis
#' 
#' Compute and represent gene set enrichment from your data based on pre-saved maps from ACSN or user imported maps. 
#' The gene set enrichment can be run with hypergeometric test or Fisher exact test, 
#' and can use multiple corrections. Visualization of data can be done either by barplots or heatmaps.
#' 
#' @param Genes Character vector of genes that should be tested for enrichment
#' @param maps list of maps generated by format_from_gmt. Names of element of list will be used to track modules.
#' Default: tests on all acsn maps.
#' @param correction_multitest either FALSE, "bonferroni", "holm", "hochberg", "hommel", "BH", "fdr" (identical to BH), or "BY"
#' @param statistical_test one of "fisher", "hypergeom"
#' @param min_module_size will remove from the analysis all modules which are (strictly) smaller than threshold
#' @param universe Universe on which the statistical analysis should be performed. 
#' Can be either "HUGO","ACSN","map_defined", or a character vector of genes.
#' @param threshold : maximal p-value (corrected if correction is enabled) that will be displayed
#' @param alternative One of "greater", "less", "both" or "two.sided"
#' Greater will check for enrichment, less will check for depletion, and both will look for 
#' both and will keep track of the side,
#' while two-sided (only for fisher test) checks if there is a difference .
#' @examples enrichment(genes_test,min_module_size = 10, 
#'    threshold = 0.05,
#'    maps = list(cellcycle = ACSNMineR::ACSN_maps$CellCycle),
#'    universe = "ACSN")
#' @export
#' @importFrom stats fisher.test p.adjust phyper
#' @importFrom utils read.csv
enrichment<-function(Genes=NULL,
                     maps = ACSNMineR::ACSN_maps, 
                     correction_multitest = "BH",
                     statistical_test = "fisher",
                     min_module_size = 5,
                     universe = "map_defined",
                     threshold = 0.05,
                     alternative = "greater"){
  
  ### Checking maps
  if(is.data.frame(maps) | is.matrix(maps)){
    maps<-list(maps)
  }
  else if(!is.list(maps)){
    stop("maps should be a dataframe or a list of dataframes. Exiting")
  }
  
  if(universe == "ACSN"){
    universe_was_ACSN<-TRUE
  }
  else{
    universe_was_ACSN<-FALSE
    
  }
  
  ### Checking that gene list is unique
  Genes<-unique(Genes)
  Genes_size<-length(Genes)    
  if(!(alternative %in% c("greater","less","both","two.sided"))){
    stop('enrichment variable should be one of:"greater","less","both","two.sided" ')
  }
  ### Checking that parameters are correct
  if(sum(statistical_test %in% c("fisher", "hypergeom"))==0){
    stop("statistical_test should be one of 'fisher' or 'hypergeom'")
  }
  if(alternative == "two.sided" & statistical_test != "fisher"){
    warning("two-sided test only relevant for fisher test. \n Will perform both enrichment and under-representation tests.")
    alternative<-"both"
  }
  result<-data.frame()
  
  ### If universe is ACSN: extract genes from ACSN and define it as universe
  if(length(universe == 1)){
    if(universe == "ACSN"){
      genesACSN<-unique(unlist(lapply(X = ACSNMineR::ACSN_maps,FUN = function(z){
        return(as.character(unique(z[,-(1:2)])))
      })))
      universe<-genesACSN[genesACSN!=""]
      size<-length(genesACSN)
    }
    else if(universe == "HUGO"){
      ###Total size of approved symbols, from http://www.genenames.org/cgi-bin/statistics, as of October 8th 2015
      size = 39480
    }
    
    else if( universe == "map_defined"){
      genesmap<-character()
      iterator<-0
      if(is.list(maps)){
        genesmap<-unique(as.character(unlist(lapply(X = maps,FUN = function(z){
          return(as.character(z[,-(1:2)]))
        }))))
      }
      else{
        genesmaps<-unique(as.character(maps[,-(1:2)]))
      }
      genesmap<-as.character(genesmap[genesmap!=""])
      is_in_ACSN<-Genes %in% genesmap
      S<-sum(!is_in_ACSN)
      if(S > 0){ ### removing genes from list, that are not from ACSN
        warning(paste(S,"genes are not in ACSN modules and will be excluded from analysis"))
      }
      Genes<-Genes[is_in_ACSN]
      size <- length(genesmap)
      universe<-genesmap
      
    }
    else{
      stop("Invalid universe input: must be 'HUGO','ACSN','map_defined', or a gene list")
    }
    
    ### Length of universe can be changed if "ACSN"
    
  }
  if(length(universe )> 1){
    ### Change maps so that they only have gene names from universe and remove modules which are too small
    i<-0
    genesACSN<-character()
    iterator<-0
    for(lt in maps){
      iterator<-iterator+1
      ### extract modules of size >= module_size and get size of ACSN reduced by universe and module size
      validmap<-data.frame()
      
      if(dim(lt)[1]>1){ ### map is a dataframe
        for(k in 1:dim(lt)[1]){
          spare_genes<-as.character(lt[k,-c(1,2)])
          spare_genes[!(spare_genes %in% universe)]<- ""
          test<-spare_genes != ""
          S<-sum(test)
          if( S>= min_module_size){
            genesACSN<-unique(c(genesACSN,unique(spare_genes[test])))
            validmap<-rbind(validmap,cbind(lt[k,1],S,t(spare_genes)))          
          }
        }
      }
      else{ #map is a single line
        if(length(lt)>2){
          spare_genes<-as.character(sapply(3:length(lt),FUN = function(z) as.character(lt[[z]])))
          test<-spare_genes != ""
          S<-sum(test)
          if( S>= min_module_size){
            genesACSN<-unique(c(genesACSN,unique(spare_genes[test])))
            validmap<-rbind(validmap,cbind(lt[1],S,t(spare_genes)))
            validmap[,2]<-as.integer(as.character(validmap[,2]))
          }
        }
        else{
          stop("Empty map, exiting")
        }
      }
      if(!universe_was_ACSN){
        size <- length(genesACSN)
      }
      maps[[iterator]]<-validmap
    }
    is_in_ACSN<-Genes %in% genesACSN
    S<-sum(!is_in_ACSN)
    if(S > 0){ ### removing genes from list, that are not from ACSN
      warning(paste(S,"genes are not in universe and will be excluded from analysis"))
    }
    Genes<-Genes[is_in_ACSN]
    
  }
  if(length(Genes)==0){
    stop("No genes in universe.")
    result<-data.frame()
  }
  
  #### Begin calculation of p-values
  
  ### get from list how many are in each sub-compartment
  result<-data.frame()
  ### what would be the expected?
  tracker<-0
  map_names<-names(maps)
  if(is.null(map_names)){
    if(length(maps)>1){
      message("Maps should be defined in this way: maps<-list(map1 = ... , map2 = ...)")
      message("In the absence of names, will use letters")
    }
    map_names<-paste("map",1:length(maps),sep = "_")
  }
  for(map in maps){
    tracker <- tracker + 1
    modules<-map[,1:2]
    
    ### Calculate p-value for map as a whole if map is not ACSN_master
    if(universe_was_ACSN & map_names[tracker]!="ACSN_master"){
      compute_module_p.value<-TRUE
    }
    else if(!is.data.frame(maps) & length(maps)>1 & map_names[tracker]!="ACSN_master"){
      compute_module_p.value<-TRUE
    }
    else{
      compute_module_p.value<-FALSE
    }
    if( compute_module_p.value ){ ###
      mapgenes<-unique(as.character(unlist(map[,-c(1:2)])))
      
      mapsize<-length(mapgenes)
      genes_in_mapgenes<-(Genes %in% mapgenes)      
      num<-sum( genes_in_mapgenes)
      
      if(alternative != "both"){        
        
        if(statistical_test == "fisher"){
          p.values<-c(p.val.calc(num,
                                 mapsize-num,
                                 Genes_size-num,
                                 size - mapsize,
                                 "fisher",
                                 alternative
          ),
          alternative)
        }
        else{
          p.values<-c(p.val.calc( num,
                                  mapsize,
                                  size - mapsize,
                                  length(Genes),
                                  "hypergeom",
                                  alternative
          ),
          alternative
          ) 
        }
      }
      else{ ### if both computations
        if(statistical_test == "fisher"){
          p.values<-cbind(c(p.val.calc(num,
                                       mapsize-num,
                                       Genes_size-num,
                                       size - mapsize,
                                       "fisher",
                                       "greater"
          ),
          "greater"),
          c(p.val.calc(num,
                       mapsize-num,
                       Genes_size-num,
                       size - mapsize,
                       "fisher",
                       "less"
          ),
          "less")
          )
        }
        else{
          p.values<-rbind(c(p.val.calc( num,
                                        mapsize,
                                        size - mapsize,
                                        length(Genes),
                                        "hypergeom",
                                        'greater'
          ),
          'greater')
          ,c(p.val.calc( num,
                         mapsize,
                         size - mapsize,
                         length(Genes),
                         "hypergeom",
                         'less'
          ),
          'less'
          )
          )
        }
      }

      spare<-cbind(map_names[tracker],
                   mapsize,
                   paste(Genes[genes_in_mapgenes],collapse = " "),
                   num,
                   t(p.values))
      colnames(spare)<-c("module","module_size","genes_in_module",
                         "nb_genes_in_module","p.value","test")


      result<-rbind(result,spare)
      result$module_size<-as.integer(as.character(result$module_size))
    } #### End of module p-value
    
    if(length(maps)>1){
      modules[,1]<-paste(map_names[tracker],modules[,1],sep=":")
      map[,1]<-modules[,1]
    }
    #### Calculation for modules
    if(statistical_test == "fisher"){
      if(alternative != "both"){
        p.values<-apply(map,MARGIN = 1, FUN = function(z){
          short_z<-z[z!=""][-c(1,2)] ### remove empty slots, module name and length
          num<-cnum(z[2])
          test<-Genes %in% short_z
          Gene_set<-paste(Genes[test],collapse = " ")
          Genes_in_module<-sum(test)
          return(t(c(Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                         num-Genes_in_module,
                                                         Genes_size - Genes_in_module,
                                                         size - num,
                                                         statistical_test,
                                                         alternative),alternative))
          )
        }
        )
      }
      else{
        p.values<-apply(map,MARGIN = 1, FUN = function(z){
          short_z<-z[z!=""][-c(1,2)] ### remove empty slots, module name and length
          num<-cnum(z[2])
          test<-Genes %in% short_z
          Gene_set<-paste(Genes[test],collapse = " ")
          Genes_in_module<-sum(test)
          return(cbind(t(c(z[1],z[2],Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                                         num-Genes_in_module,
                                                                         Genes_size - Genes_in_module,
                                                                         size - num,
                                                                         statistical_test,
                                                                         "greater"),"enrichment")),
                       t(c(z[1],z[2],Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                                         num-Genes_in_module,
                                                                         Genes_size - Genes_in_module,
                                                                         size - num,
                                                                         statistical_test,
                                                                         "less"),"depletion"))))
        }
        )
      }
    }else if(statistical_test == "hypergeom"){
      if(alternative != "both"){
        p.values<-apply(map,MARGIN = 1, FUN = function(z){
          short_z<-z[z!=""][-c(1,2)] ### remove empty slots, module name and length
          num<-cnum(z[2])
          test<-Genes %in% short_z
          Genes_in_module<-sum(test)
          Gene_set<-paste(Genes[test],collapse = " ")
          if(Genes_in_module > 0){ ### Correction for depletion: phyper tests for P[X<=x]
            Genes_in_module<-Genes_in_module-1 
          }
          
          return(c(Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                       num,
                                                       size - num,
                                                       length(Genes),
                                                       statistical_test,
                                                       alternative),
                   alternative))
          
        })
      }
      else{ 
        p.values<-apply(map,MARGIN = 1, FUN = function(z){
          short_z<-z[z!=""][-c(1,2)] ### remove empty slots, module name and length
          num<-cnum(z[2])
          test<-Genes %in% short_z
          Genes_in_module<-sum(test)
          Gene_set<-paste(Genes[test],collapse = " ")
          if(Genes_in_module > 0){ ### Correction for depletion: phyper tests for P[X<=x]
            Genes_in_module<-Genes_in_module-1 
          }
          
          return(rbind(c(z[1],z[2],Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                                       num,
                                                                       size - num,
                                                                       length(Genes),
                                                                       statistical_test,
                                                                       "greater"),
                         "enrichment")),
                 c(z[1],z[2],Gene_set,Genes_in_module,p.val.calc(Genes_in_module,
                                                                 num,
                                                                 size - num,
                                                                 length(Genes),
                                                                 statistical_test,
                                                                 "less"),
                   "depletion"))
          
        })
      }
    }
    if(alternative!="both"){
      spare<-cbind(modules,t(p.values))
    }
    else{
      spare<-rbind(t(p.values)[,1:6],t(p.values)[,7:12])
    }
    colnames(spare)<-c("module","module_size","genes_in_module",
                       "nb_genes_in_module","p.value","test")
    result<-rbind(result,spare)
  }
  result$p.value<-cnum(result$p.value)
  result$universe_size<-size
  result$genes_in_universe<-Genes_size
  if(is.logical(correction_multitest)){
    result<-result[cnum(result$p.value) <= threshold,]
    if(correction_multitest){
      warning("If multiple correction is wanted, please use one of the listed parameters.")
    }
  }
  else{
    if(correction_multitest %in% c("bonferroni", "holm", "hochberg", "hommel", "BH", "fdr", "BY")){
      result$p.value.corrected<-p.adjust(cnum(result$p.value),method = correction_multitest)
      result<-result[result$p.value.corrected <= threshold,]
    }
    else{
      result<-result[result$p.value <= threshold,]
      warning('correction multitest should be one of the following: "bonferroni", "holm", "hochberg", "hommel", "BH", "fdr", "BY"')
    }
  }
  
  ###adding missing info
  if(dim(result)[1]==0){
    warning("No modules under threshold")
    return(NA)
  }
  result$universe_size<-size
  result$nb_genes_in_universe<-length(Genes)
  p.val.names<-colnames(result)[grepl(pattern = "p.value",x = colnames(result))]
  
  ### re-ordering
  result<-result[,c("module","module_size","nb_genes_in_module",
                    "genes_in_module","universe_size",
                    "nb_genes_in_universe",
                    p.val.names,
                    "test")]
  return(result)
}

####### Multiplesample analysis / multiple cohorts analysis #######

#' Automated gene set analysis for multiple sets
#' 
#' 
#' @param Genes_by_sample List of character vectors. Each list element name should be a sample name, 
#' and each character vector the set of genes to test for the sample.
#' @param maps list of maps generated by format_from_gmt. Default: tests on all acsn maps
#' @param correction_multitest either FALSE, "bonferroni", "holm", "hochberg", "hommel", "BH", "fdr" (identical to BH), or "BY"
#' @param statistical_test one of "fisher", "hypergeom"
#' @param min_module_size will remove from the analysis all modules which are (strictly) smaller than threshold
#' @param universe Universe on which the statistical analysis should be performed. Can be either "HUGO","ACSN" 
#' ,"map_defined", or a character vector of genes.
#' @param threshold maximal p-value (corrected if correction is enabled) that will be displayed
#' @param cohort_threshold if TRUE modules will be kept in all samples if at least one sample 
#' has p-value lower than threshold, otherwise the threshold is applied for each sample independently.
#' @param alternative One of "greater", "less", "both", or "two.sided" (only for fisher test).
#' Greater will check for enrichment, less will check for depletion, and both will look for both.
#' @examples multisample_enrichment(Genes_by_sample = list(set1 = genes_test[-1],set2=genes_test[-2]),
#' maps = list(cellcycle = ACSNMineR::ACSN_maps$CellCycle),
#' min_module_size = 10,
#' universe = "ACSN")
#' @export

multisample_enrichment<-function(Genes_by_sample=NULL,
                                 maps = ACSNMineR::ACSN_maps, 
                                 correction_multitest = "BH",
                                 statistical_test = "fisher",
                                 min_module_size = 5,
                                 universe = "map_defined",
                                 threshold = 0.05,
                                 cohort_threshold = TRUE,
                                 alternative = "greater"){
  
  if(cohort_threshold){
    result<-lapply(X = Genes_by_sample ,
                   FUN = function(z){
                     enrichment(Genes = z,
                                maps = maps,
                                correction_multitest = correction_multitest,
                                statistical_test = statistical_test,
                                min_module_size = min_module_size,
                                universe = universe,
                                threshold =  1)
                   })
    kept_modules<-character() 
    for(sample in result){
      if(is.logical(correction_multitest)){ 
        kept_modules<-unique(c(kept_modules,sample[sample$p.value<=threshold,1]))
      }
      else{
        kept_modules<-unique(c(kept_modules,sample[sample$p.value.corrected<=threshold,1]))
      }
    }
    for(i in 1:length(result)){
      result[[i]]<-result[[i]][result[[i]][,1] %in% kept_modules,]
    }
  }
  else{
    result<-lapply(X = Genes_by_sample ,
                   FUN = function(z){
                     enrichment(Genes = z,
                                maps = maps,
                                correction_multitest = correction_multitest,
                                statistical_test = statistical_test,
                                min_module_size = min_module_size,
                                universe = universe,
                                threshold =  threshold)
                   })
    names(result)<-names(Genes_by_sample)
  }
  return(result)
  
}
#' Calculate p-value
#' 
#' @param x : first value
#' @param y : second value
#' @param z : third value
#' @param a : fourth value
#' @param stat_test : statistical test to be used
#' @param alt : alternative: one of two-sided, greater, less or both
#' 
p.val.calc<-function(x,y,z,a,stat_test,alt){
  if(stat_test=="fisher"){
    return(fisher.test(matrix(c(x,y,z,a),nrow = 2),
                       alt = alt)$p.value)
  }
  else{
    if(alt =="greater"){
      return(phyper(x, y,z,a,lower.tail = FALSE))
    }
    else{
      return(phyper(x, y,z,a,lower.tail = TRUE))
    }
    
  }
}

#### Graphic representation of results ####
#' Graphic representation of enrichment
#'@param enrichment Data frame or list of dataframes with p-values or corrected p-values (whenever available) and module names for representation.
#'The name of the dataframe will be used as sample name.
#'@param plot Any of "heatmap" or "bar"
#'@param scale Any of "log" or "identity"
#'@param low Color to be used in heatmap mode corresponding to lowest value
#'@param high Color to be used in heatmap mode corresponding to highest value
#'@param nrow Number of rows of the grid for display in bar mode.
#'@param sample_name  used only is enrichment is a dataframe
#'@param na.value color for the missing values in the heatmap
#'@examples represent_enrichment(enrichment = list(SampleA = enrichment_test[1:10,], 
#'SampleB = enrichment_test[3:10,]), plot = "heatmap", scale = "log")
#'@import ggplot2 
#'@importFrom gridExtra grid.arrange
#'@export
represent_enrichment<-function(enrichment, plot = "heatmap" , scale = "log", 
                               low = "steelblue" , high ="white",
                               nrow = 1,sample_name = "Sample",
                               na.value = "grey"){
  
  
  if(is.data.frame(enrichment)){
    if("p.value.corrected" %in% colnames(enrichment)){
      enrichment$p.values<-enrichment$p.value.corrected
    }
    else if("p.value" %in% colnames(enrichment)){
      enrichment$p.values<-enrichment$p.value
    }
    else{
      warning("dataframe has no column 'p.value' or 'p.value.corrected'. Exiting!")
      return(NA)
    }
    enrichment$sample_name<-sample_name 
    enrichment$p.values<-cnum(enrichment$p.values)
    if(plot == "heatmap"){
      
      q<-ggplot2::ggplot(enrichment,
                         ggplot2::aes_string(x= "sample_name",
                                             y = "module", 
                                             fill = "p.values"))+ggplot2::xlab("")+ ggplot2::ylab("Modules") + ggplot2::geom_tile()
      if(scale == "log"){
        q<- q + ggplot2::scale_fill_gradient("p-values",low = low , high = high, na.value = na.value, trans = "log")
      }
      else{
        q<-q+ ggplot2::scale_fill_gradient("p-values",low = low , high = high, na.value = na.value)
      }
      
    }
    else{
      q<-ggplot2::qplot(x = enrichment$module, y = enrichment$p.values,
                        xlab = "Modules", ylab = "p-values")
      if(scale == "log"){
        q<- q + ggplot2::scale_y_continuous(trans = "log")
      }
    }    
    q<-q+ggplot2::theme_minimal()+ggplot2::theme(axis.text.x = element_text(angle = 90, hjust = 0), axis.ticks = ggplot2::element_blank())
    q<-q+ggplot2::geom_bar(stat = "identity")
  }
  else if(is.list(enrichment)){
    sample_names<-names(enrichment)
    if(is.null(sample_names)){
      warning("Your list has no names... Will use numbers instead")
      sample_names<-1:length(enrichment)
    }
    if(plot == "heatmap"){
      dataset<-data.frame()
      
      ### check if there is a need to add NAs for modules which are not present in all datasets
      all_equal<-TRUE
      modules<-as.character(enrichment[[1]]$module)
      for(sample in enrichment){
        if(!setequal(modules,as.character(sample$module))){
          all_equal<-FALSE
          modules<-unique(c(modules,as.character(sample$module)))
        }
      }
      tracker<-0
      if(all_equal){
        
        for(sample in enrichment){ ### Modules are present in all samples
          tracker<-tracker+1
          if("p.value.corrected" %in% colnames(sample)){
            dataset<-rbind(dataset, cbind(modules,sample_names[tracker],cnum(sample$p.value.corrected)))
          }
          else if("p.value" %in% colnames(sample)){
            dataset<-rbind(dataset, cbind(modules,sample_names[tracker],cnum(sample$p.value)))
          }
          else{
            warning(paste("Element", sample_names[tracker],"has no p.value column"))
            return(NA)
          }
        }
        colnames(dataset)<-c("module","sample_name","p.values")
      }
      else{ ###need to fill with NAs
        ### Not functional yet
        
        for(sample in enrichment){
          tracker<-tracker + 1
          
          test<-modules %in% as.character(sample$module)
          restricted_modules<-as.character(modules[test])
          position_modules<-as.numeric(sapply(X = restricted_modules,FUN = function(z){
            which(sample$module ==z)
          }))
          
          if(sum(!test)>0){ ### complement only when necessary
            complement<-as.character(modules[!test])
            if("p.value.corrected" %in% colnames(sample)){ ### 
              
              spare_dataset<-rbind(cbind(module = restricted_modules, sample_name = sample_names[tracker], 
                                         p.values = cnum(sample$p.value.corrected[position_modules])),
                                   cbind(module = complement, 
                                         sample_name = sample_names[tracker],
                                         p.values = NA))
              
            }
            else if("p.value" %in% colnames(sample)){ ### 
              spare_dataset<-rbind(cbind(restricted_modules, sample_names[tracker], cnum(sample$p.value[position_modules])),
                                   cbind(complement, sample_names[tracker],NA))
              
            } else{
              warning(paste("Element", sample_names[tracker],"has no p.value column"))
              return(NA)
            }
            colnames(spare_dataset)<-c("module","sample_name","p.values")
            dataset<-rbind(dataset, spare_dataset)
          }
          else{ ### rbind dataframe with values
            if("p.value.corrected" %in% colnames(sample)){ ### 
              
              spare_dataset<-cbind(module = restricted_modules, sample_name = sample_names[tracker], 
                                   p.values = cnum(sample$p.value.corrected[position_modules]))
              
            }
            else if("p.value" %in% colnames(sample)){ ### 
              spare_dataset<-cbind(restricted_modules, sample_names[tracker], cnum(sample$p.value[position_modules]))
            } else{
              warning(paste("Element", sample_names[tracker],"has no p.value column"))
              return(NA)
            }
            colnames(spare_dataset)<-c("module","sample_name","p.values")
            dataset<-rbind(dataset, spare_dataset)
            
          }
        }
      }
      dataset$p.values<-cnum(dataset$p.values)
      ### Plot heatmap
      q<-ggplot2::ggplot(dataset,
                         aes_string(x= "sample_name",
                                    y = "module", 
                                    fill = "p.values"))+ggplot2::xlab("")+ ggplot2::ylab("Modules") + ggplot2::geom_tile()
      if(scale == "log"){
        q<- q + ggplot2::scale_fill_gradient("p-values",low = low , high = high, na.value = na.value, trans = "log")
      }
      else{
        q<-q+ ggplot2::scale_fill_gradient("p-values",low = low , high = high, na.value = na.value)
      }
      q<-q+ggplot2::theme_minimal()+ggplot2::theme(axis.text.x = element_text(angle = 90, hjust = 0), axis.ticks = ggplot2::element_blank())
    }
    else{ ### barplot with grid
      names_sample<-names(enrichment)
      plot<-list()
      for(s in 1:length(enrichment)){
        plot[[s]]<-represent_enrichment(enrichment[[s]], plot = "bar" , 
                                        scale = scale, 
                                        sample_name = names_sample[s])
        
      }
      #       if(length(plot)%%nrow ==0){
      #         ncol <- length(enrichment)/nrow
      #       }
      #       else{
      #         ncol <- floor(length(enrichment)/nrow)+1
      #       }
      return(do.call(gridExtra::grid.arrange, c(plot, nrow=nrow)))
    }
    
  }
  else{
    warning("Wrong input format for enrichment!")
    return(NA)
    
  }
  return(q)
  
}

#' Convert to numeric
#' 
#' @param x A vector of numbers which is not in numeric format
cnum<-function(x){
  return(as.numeric(as.character(x)))
}


#' Import data from gmt files
#' Convert gmt file to dataframe that can be used for anaysis
#'@param path Path to the gmt file to be imported
#'@examples file<-system.file("extdata", "cellcycle_short.gmt", package = "ACSNMineR")
#'format_from_gmt(file)
#'@export

format_from_gmt<-function(path = ""){
  
  Lines<-readLines(path,warn = FALSE)
  
  
  ### Filter out non-genes
  if(length(Lines)==1){ ### testing if gmt is single line
    gmt<-unlist(strsplit(x = Lines,split = "\t"))
    short_gmt<-gmt[-c(1,2)]
    pos<-grepl(pattern = "\\*",x = short_gmt)
    result<-c(gmt[c(1,2)],gmt[-c(1,2)][!pos])
    result[2]<-length(result)-2
    result<-as.data.frame(t(result))
    
    
  }
  else{
    max_length<-max(sapply(X=Lines,FUN = function(z){
      z2<-gsub("\t","",z)
      return(nchar(z)-nchar(z2))
    }))  
    gmt<-read.csv(path,header = FALSE, 
                  sep = "\t",fill = TRUE,
                  col.names = paste("V",1:max_length,sep="")
    )
    
    result<-t(apply(gmt,1,FUN = function(z){
      pos<-grepl(pattern = "\\*",x = z)
      res<-z
      res[pos]<-""
      return(res)
    }))
    result[,2]<-apply(result[,-(1:2)], 1, FUN = function(z) sum(z!=""))
    
  }
  
  return(result)
}


#' Atlas of Cancer Signalling Networks
#' 
#' A dataset containing the six maps of ACSN: apoptosis, cell cycle, DNA reparation, EMT motility, survival, and the master map
#' @format A list of dataframes
#' \describe{
#'  \item{Apoptosis}{Map of apoptosis}
#'  \item{CellCycle}{Map of the cell cycle}
#'  \item{DNA_repair}{Map of DNA repair}
#' }
#' @source \url{https://acsn.curie.fr/downloads.html}
"ACSN_maps"

#' Set of genes to test map
#' 
#' Genes of high importance in oncogenesis
#' @format A character vector
"genes_test"

#' Result from enrichment test of "genes_test" on the ACSN maps
#' 
#' Parameters: bonferroni correction, min module size = 5
#' @format data.frame
#' \describe{
#'  \item{module}{Name of module}
#'  \item{genes_in_module}{Genes from genes_test in module}
#'  \item{p.value}{Uncorrected p-value}
#'  \item{p.value.corrected}{p-value corrected for multiple testing by Bonferroni correction}
#'  }
"enrichment_test"