% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auto_ardl.R
\name{auto_ardl}
\alias{auto_ardl}
\title{Automatic ARDL model selection}
\usage{
auto_ardl(
  formula,
  data,
  max_order,
  fixed_order = -1,
  starting_order = NULL,
  selection = "AIC",
  selection_minmax = c("min", "max"),
  grid = FALSE,
  search_type = c("horizontal", "vertical"),
  start = NULL,
  end = NULL,
  ...
)
}
\arguments{
\item{formula}{A "formula" describing the linear model. Details for model
specification are given under 'Details' in the help file of the
\code{\link{ardl}} function.}

\item{data}{A time series object (e.g., "ts", "zoo" or "zooreg") or a data
frame containing the variables in the model. In the case of a data frame,
it is coerced into a \code{\link[stats]{ts}} object with \code{start = 1},
\code{end = nrow(data)} and \code{frequency = 1}. If not found in data, the
variables are NOT taken from any environment.}

\item{max_order}{It sets the maximum order for each variable where the search
is taking place. A numeric vector of the same length as the total number of
variables (excluding the fixed ones, see 'Details' in the help file of the
\code{\link{ardl}} function). It should only contain positive integers. An
integer could be provided if the maximum order for all variables is the
same.}

\item{fixed_order}{It allows setting a fixed order for some variables. The
algorithm will not search for any other order than this. A numeric vector
of the same length as the total number of variables (excluding the fixed
ones). It should contain positive integers or 0 to set as a constraint. A
-1 should be provided for any variable that should not be constrained.
\code{fixed_order} overrides the corresponding \code{max_order} and
\code{starting_order}.}

\item{starting_order}{Specifies the order for each variable from which each
  search will start. It is a numeric vector of the same length as the total
  number of variables (excluding the fixed ones). It should contain positive
  integers or 0 or only one integer could be provided if the starting order
  for all variables is the same. Default is set to NULL. If unspecified
  (\code{NULL}) and \code{grid = FALSE}, then all possible \eqn{ARDL(p)}
  models are calculated (constraints are taken into account), where \eqn{p}
  is the minimum value in \code{max_order}. Note that where
  \code{starting_order} is provided, its first element will be the minimum
value of \eqn{p} that the searching algorithm will consider (think of it like
a 'minimum p order' restriction) (see 'Searching algorithm' below). If
\code{grid = TRUE}, only the first argument (\eqn{p}) will have an effect.}

\item{selection}{A character string specifying the selection criterion
according to which the candidate models will be ranked. Default is
\code{\link[stats]{AIC}}. Any other selection criterion can be used (a user
specified or a function from another package) as long as it can be applied
as \code{selection(model)}. The preferred model is the one with the smaller
value of the selection criterion. If the selection criterion works the
other way around (the bigger the better), \code{selection_minmax = "max"}
should also be supplied (see 'Examples' below).}

\item{selection_minmax}{A character string that indicates whether the
criterion in \code{selection} is supposed to be minimized (default) or
maximized.}

\item{grid}{If \code{FALSE} (default), the stepwise searching regression
algorithm will search for the best model by adding and subtracting terms
corresponding to different ARDL orders. If \code{TRUE}, the whole set of
all possible ARDL models (accounting for constraints) will be evaluated.
Note that this method can be very time-consuming in case that
\code{max_order} is big and there are many independent variables that
create a very big number of possible combinations.}

\item{search_type}{A character string describing the search type. If
"horizontal" (default), the searching algorithm increases or decreases by 1
the order of each variable in each iteration. When the order of the last
variable has been accessed, it begins again from the first variable until
it converges. If "vertical", the searching algorithm increases or decreases
by 1 the order of a variable until it converges. Then it continues the same
for the next variable. The two options result to very similar top orders.
The default ("horizontal"), sometimes is a little more accurate, but the
"vertical" is almost 2 times faster. Not applicable if \code{grid = TRUE}.}

\item{start}{Start of the time period which should be used for fitting the
model.}

\item{end}{End of the time period which should be used for fitting the model.}

\item{...}{Additional arguments to be passed to the low level regression
fitting functions.}
}
\value{
\code{auto_ardl} returns a list which contains:
  \item{\code{best_model}}{An object of class \code{c("dynlm", "lm", "ardl")}}
  \item{\code{best_order}}{A numeric vector with the order of the best model selected}
  \item{\code{top_orders}}{A data.frame with the orders of the top 20 models}
}
\description{
It searches for the best ARDL order specification, according to the selected
criterion, taking into account the constraints provided.
}
\section{Searching algorithm}{
 The algorithm performs the optimization process
  starting from multiple starting points concerning the autoregressive order
  \eqn{p}. The searching algorithm will perform a complete search, each time
  starting from a different starting order. These orders are presented in the
  tables below, for \code{grid = FALSE} and different values of
  \code{starting_order}.

  \code{starting_order = NULL}:
  \tabular{ccccccc}{
  ARDL(p) \tab -> \tab p \tab q1 \tab q2 \tab ... \tab qk\cr
  ARDL(1) \tab -> \tab 1 \tab 1 \tab 1 \tab ... \tab 1\cr
  ARDL(2) \tab -> \tab 2 \tab 2 \tab 2 \tab ... \tab 2\cr
  : \tab -> \tab : \tab : \tab : \tab : \tab :\cr
  ARDL(P) \tab -> \tab P \tab P \tab P \tab ... \tab P
  }

  \code{starting_order = c(3, 0, 1, 2)}:
  \tabular{cccc}{
  p \tab q1 \tab q2 \tab q3\cr
  3 \tab 0 \tab 1 \tab 2\cr
  4 \tab 0 \tab 1 \tab 2\cr
  : \tab : \tab : \tab :\cr
  P \tab 0 \tab 1 \tab 2
  }
}

\examples{
data(denmark)

## Find the best ARDL order --------------------------------------------

# Up to 5 for the autoregressive order (p) and 4 for the rest (q1, q2, q3)

# Using the defaults search_type = "horizontal", grid = FALSE and selection = "AIC"
# ("Not run" indications only for testing purposes)
\dontrun{
model1 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                    max_order = c(5,4,4,4))
model1$top_orders

## Same, with search_type = "vertical" -------------------------------

model1_h <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                      max_order = c(5,4,4,4), search_type = "vertical")
model1_h$top_orders

## Find the global optimum ARDL order ----------------------------------

# It may take more than 10 seconds
model_grid <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), grid = TRUE)

## Different selection criteria ----------------------------------------

# Using BIC as selection criterion instead of AIC
model1_b <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                      max_order = c(5,4,4,4), selection = "BIC")
model1_b$top_orders

# Using other criteria like adjusted R squared (the bigger the better)
adjr2 <- function(x) { summary(x)$adj.r.squared }
model1_adjr2 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                           max_order = c(5,4,4,4), selection = "adjr2",
                           selection_minmax = "max")
model1_adjr2$top_orders

# Using functions from other packages as selection criteria
if (requireNamespace("qpcR", quietly = TRUE)) {

library(qpcR)
model1_aicc <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                          max_order = c(5,4,4,4), selection = "AICc")
model1_aicc$top_orders
adjr2 <- function(x){ Rsq.ad(x) }
model1_adjr2 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                           max_order = c(5,4,4,4), selection = "adjr2",
                           selection_minmax = "max")
model1_adjr2$top_orders

## DIfferent starting order --------------------------------------------

# The searching algorithm will start from the following starting orders:
# p q1 q2 q3
# 1 1  3  2
# 2 1  3  2
# 3 1  3  2
# 4 1  3  2
# 5 1  3  2

model1_so <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), starting_order = c(1,1,3,2))

# Starting from p=3 (don't search for p=1 and p=2)
# Starting orders:
# p q1 q2 q3
# 3 1  3  2
# 4 1  3  2
# 5 1  3  2

model1_so_3 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), starting_order = c(3,1,3,2))

# If starting_order = NULL, the starting orders for each iteration will be:
# p q1 q2 q3
# 1 1  1  1
# 2 2  2  2
# 3 3  3  3
# 4 4  4  4
# 5 5  5  5
}

## Add constraints -----------------------------------------------------

# Restrict only the order of IBO to be 2
model1_ibo2 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), fixed_order = c(-1,-1,2,-1))
model1_ibo2$top_orders

# Restrict the order of LRM to be 3 and the order of IBO to be 2
model1_lrm3_ibo2 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), fixed_order = c(3,-1,2,-1))
model1_lrm3_ibo2$top_orders

## Set the starting date for the regression (data starts at "1974 Q1") -

# Set regression starting date to "1976 Q1"
model1_76q1 <- auto_ardl(LRM ~ LRY + IBO + IDE, data = denmark,
                        max_order = c(5,4,4,4), start = "1976 Q1")
start(model1_76q1$best_model)
}
}
\seealso{
\code{\link{ardl}}
}
\author{
Kleanthis Natsiopoulos, \email{klnatsio@gmail.com}
}
\keyword{models}
\keyword{optimize}
\keyword{ts}
