% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clust_cp.R
\name{clust_cp}
\alias{clust_cp}
\title{Clustering time dependent obsevations with common change points.}
\usage{
clust_cp(
  data,
  n_iterations,
  n_burnin = 0,
  params = list(),
  print_progress = TRUE,
  user_seed = 1234,
  kernel
)
}
\arguments{
\item{data}{a vector or a matrix. If a vector the algorithm for
univariate time series is used. If a matrix, where rows are the observations
and columns are the times, then the algorithm for multivariate time series is used.}

\item{n_iterations}{number of MCMC iterations.}

\item{n_burnin}{number of iterations that must be excluded when computing the posterior estimate.}

\item{params}{a list of parameters:

If the time series is univariate the following must be specified:

\itemize{
\item \code{q} probability of a split in the split-merge proposal and acceleration step.
\item \code{B} number of orders for the normalization constant.
\item \code{L} number of split-merge steps for the proposal step.
\item \code{alpha_SM} \eqn{\alpha} for the split-merge proposal and acceleration step.
\item \code{gamma},\code{a},\code{b},\code{c} parameters of the integrated likelihood.
}

If the time series is multivariate the following must be specified:

\itemize{
\item \code{q} probability of a split in the split-merge proposal and acceleration step.
\item \code{B} number of orders for the normalization constant.
\item \code{L} number of split-merge steps for the proposal step.
\item \code{gamma},\code{k_0},\code{nu_0},\code{phi_0},\code{m_0} parameters of the integrated likelihood.
}

If data are survival functions:

\itemize{
\item \code{q} probability of a split in the split-merge proposal and acceleration step.
\item \code{B} number of orders for the normalization constant.
\item \code{L} number of split-merge steps for the proposal step.
\item \code{alpha_SM} \eqn{\alpha} for the split-merge proposal and acceleration step.
\item \code{M} number of Monte Carlo iterations when computing the likelihood of the survival function.
\item \code{gamma} recovery rate fixed constant for each population at each time.
\item \code{alpha} \eqn{\alpha} for the acceptance ration in the split-merge procedure.
\item \code{dt},\code{a0},\code{b0},\code{c0},\code{d0} parameters for the computation of the integrated likelihood of the survival functions.
\item \code{MH_var} variance for the Metropolis-Hastings estimation of the proportion of infected at time 0.
\item \code{S0},\code{R0} parameters for the SDE solver.
\item \code{p} prior average number of change points for each order.
}}

\item{print_progress}{If TRUE (default) print the progress bar.}

\item{user_seed}{seed for random distribution generation.}

\item{kernel}{can be "ts" if data are time series or "epi" if data are survival functions.}
}
\value{
A \code{ClustCpObj} class object containing

\itemize{
\item \code{$data} vector or matrix with the data.
\item \code{$n_iterations} number of iterations.
\item \code{$n_burnin} number of burn-in iterations.
\item \code{$clust} a matrix where each row corresponds to the output cluster of the corresponding iteration.
\item \code{$orders} a multidimensional array where each slice is a matrix and represent an iteration. The row of each matrix correspond the order associated to the corresponding cluster.
\item \code{$time} computational time.
\item \code{$lkl} a matrix where each row is the likelihood of each observation computed at the corresponding iteration.
\item \code{$norm_vec} a vector containing the normalisation constant computed at the beginning of the algorithm.
\item \code{$rho} a vector with the final estimate of the proportion of infected individuals at time 0.
\item \code{$kernel_ts} if TRUE data are time series.
\item \code{$kernel_epi} if TRUE data are survival function.
\item \code{$univariate_ts} TRUE if data is an univariate time series, FALSE if it is a multivariate time series.
}
}
\description{
The \code{clust_cp} function cluster observations with common change points. Data can be time series or survival functions.
}
\examples{

\donttest{
## Univariate time series

data_mat <- matrix(NA, nrow = 5, ncol = 100)

data_mat[1,] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_mat[2,] <- as.numeric(c(rnorm(50,0,0.125), rnorm(50,1,0.225)))
data_mat[3,] <- as.numeric(c(rnorm(50,0,0.175), rnorm(50,1,0.280)))
data_mat[4,] <- as.numeric(c(rnorm(25,0,0.135), rnorm(75,1,0.225)))
data_mat[5,] <- as.numeric(c(rnorm(25,0,0.155), rnorm(75,1,0.280)))

out <- clust_cp(data = data_mat, n_iterations = 5000, n_burnin = 1000,
                params = list(L = 1, B = 1000, gamma = 0.5), kernel = "ts")

print(out)

## Multivariate time series


data_array <- array(data = NA, dim = c(3,100,5))

data_array[1,,1] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_array[2,,1] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_array[3,,1] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))

data_array[1,,2] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_array[2,,2] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_array[3,,2] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))

data_array[1,,3] <- as.numeric(c(rnorm(50,0,0.175), rnorm(50,1,0.280)))
data_array[2,,3] <- as.numeric(c(rnorm(50,0,0.175), rnorm(50,1,0.280)))
data_array[3,,3] <- as.numeric(c(rnorm(50,0,0.175), rnorm(50,1,0.280)))

data_array[1,,4] <- as.numeric(c(rnorm(25,0,0.135), rnorm(75,1,0.225)))
data_array[2,,4] <- as.numeric(c(rnorm(25,0,0.135), rnorm(75,1,0.225)))
data_array[3,,4] <- as.numeric(c(rnorm(25,0,0.135), rnorm(75,1,0.225)))

data_array[1,,5] <- as.numeric(c(rnorm(25,0,0.155), rnorm(75,1,0.280)))
data_array[2,,5] <- as.numeric(c(rnorm(25,0,0.155), rnorm(75,1,0.280)))
data_array[3,,5] <- as.numeric(c(rnorm(25,0,0.155), rnorm(75,1,0.280)))

out <- clust_cp(data = data_array, n_iterations = 3000, n_burnin = 1000,
                params = list(B = 1000, L = 1, gamma = 0.5, k_0 = 0.25,
                              nu_0 = 5, phi_0 = diag(0.1,3,3),
                              m_0 = rep(0,3)), kernel = "ts")

print(out)

## Epidemiological data



data_mat <- matrix(NA, nrow = 5, ncol = 50)

betas <- list(c(rep(0.45, 25),rep(0.14,25)),
              c(rep(0.55, 25),rep(0.11,25)),
              c(rep(0.50, 25),rep(0.12,25)),
              c(rep(0.52, 10),rep(0.15,40)),
              c(rep(0.53, 10),rep(0.13,40)))

inf_times <- list()

for(i in 1:5){

  inf_times[[i]] <- sim_epi_data(10000, 10, 50, betas[[i]], 1/8)

  vec <- rep(0,50)
  names(vec) <- as.character(1:50)

  for(j in 1:50){
    if(as.character(j) \%in\% names(table(floor(inf_times[[i]])))){
      vec[j] = table(floor(inf_times[[i]]))[which(names(table(floor(inf_times[[i]]))) == j)]
    }
  }
  data_mat[i,] <- vec
}

out <- clust_cp(data = data_mat, n_iterations = 100, n_burnin = 10,
                params = list(M = 100, L = 1, B = 1000), kernel = "epi")

print(out)
}

}
\references{
Corradin, R., Danese, L., KhudaBukhsh, W. R., & Ongaro, A. (2024). \emph{Model-based clustering of time-dependent observations with common structural changes}. arXiv preprint arXiv:2410.09552.
}
