% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bibit.R
\name{bibit3}
\alias{bibit3}
\title{The BiBit Algorithm with Noise Allowance guided by Provided Patterns.}
\usage{
bibit3(matrix = NULL, minr = 1, minc = 2, noise = 0,
  pattern_matrix = NULL, subpattern = TRUE, extend_columns = TRUE,
  pattern_combinations = FALSE, arff_row_col = NULL)
}
\arguments{
\item{matrix}{The binary input matrix.}

\item{minr}{The minimum number of rows of the Biclusters. (Note that in contrast to \code{\link{bibit}} and \code{\link{bibit2}}, this can be be set to 1 since we are looking for additional rows to the provided pattern.)}

\item{minc}{The minimum number of columns of the Biclusters.}

\item{noise}{Noise parameter which determines the amount of zero's allowed in the bicluster (i.e. in the extra added rows to the starting row pair).
\itemize{
\item \code{noise=0}: No noise allowed. This gives the same result as using the \code{\link{bibit}} function.
\item \code{0<noise<1}: The \code{noise} parameter will be a noise percentage. The number of allowed 0's in a (extra) row in the bicluster will depend on the column size of the bicluster. 
More specifically \code{zeros_allowed = ceiling(noise * columnsize)}. For example for \code{noise=0.10} and a bicluster column size of \code{5}, the number of allowed 0's would be \code{1}.
\item \code{noise>=1}: The \code{noise} parameter will be the number of allowed 0's in a (extra) row in the bicluster independent from the column size of the bicluster. In this noise option, the noise parameter should be an integer.
}}

\item{pattern_matrix}{Matrix (Number of Patterns x Number of Data Columns) containing the patterns of interest.}

\item{subpattern}{Boolean value if sub patterns are of interest as well (default=TRUE).}

\item{extend_columns}{Boolean value if columns of Biclusters should also be extended for additional results (default=TRUE). See Details Section for more info.}

\item{pattern_combinations}{Boolean value if the pairwise combinations of patterns (the intersecting 1's) should also used as starting points (default=FALSE).}

\item{arff_row_col}{Same argument as in \code{\link{bibit}} and \code{\link{bibit2}}. However you can only provide 1 pattern by using this option. For \code{bibit3} to work, the pattern has to be added 2 times on top of the matrix (= identical first 2 rows).}
}
\value{
A S3 list object, \code{"bibit3"} in which each element (apart from the last one) corresponds with a provided pattern or combination thereof. \cr
Each element is a list containing:
\describe{
\item{\code{Number}: }{Number of Initially found BC's by applying BiBit with the provided pattern.} 
\item{\code{Number_Extended}: }{Number of additional discovered BC's by extending the columns.}
\item{\code{FullPattern}: }{Biclust S4 Class Object containing the Bicluster with the Full Pattern.}
\item{\code{SubPattern}: }{Biclust S4 Class Object containing the Biclusters showing parts of the pattern.}
\item{\code{Extended}: }{Biclust S4 Class Object containing the additional Biclusters after extending the biclusters (column wise) of the full and sub patterns}
\item{\code{info}: }{Contains \code{Time_Min} element which includes the elapsed time of parts and the full analysis.}
}
The last element in the list is a matrix containing all the investigated patterns.
}
\description{
Same function as \code{\link{bibit2}} but only aims to discover biclusters containing the (sub) pattern of provided patterns or their combinations.
}
\details{
The goal of the \code{\link{bibit3}} function is to provide one or multiple patterns in order to only find those biclusters exhibiting those patterns.
Multiple patterns can be given in matrix format, \code{pattern_matrix}, and their pairwise combinations can automatically be added to this matrix by setting \code{pattern_combinations=TRUE}.
All discovered biclusters are still subject to the provided \code{noise} level.

Three types of Biclusters can be discovered:
\describe{
\item{\emph{Full Pattern: }}{Bicluster which overlaps completely (within allowed noise levels) with the provided pattern. The column size of this bicluster is always equal to the number of 1's in the pattern.}
\item{\emph{Sub Pattern: }}{Biclusters which overlap with a part of the provided pattern within allowed noise levels. Will only be given if \code{subpattern=TRUE} (default). Setting this option to \code{FALSE} decreases computation time.}
\item{\emph{Extended: }}{Using the resulting biclusters from the full and sub patterns, other columns will be attempted to be added to the biclusters while keeping the noise as low as possible (the number of rows in the BC stays constant). Naturally the articially added pattern rows will not be taken into account with the noise levels as they are 0 in each other column.
\cr The question which is attempted to be answered here is \emph{`Do the rows, which overlap partly or fully with the given pattern, have other similarities outside the given pattern?`}}
} 

\emph{How?}
\cr The BiBit algorithm is applied to a data matrix that contains 2 identical artificial rows at the top which contain the given pattern. 
The default algorithm is then slightly altered to only start from this articial row pair (=Full Pattern) or from 1 artificial row and 1 other row (=Sub Pattern).

\emph{Note 1 - Large Data:}
\cr The \code{arff_row_col} can still be provided in case of large data matrices, but the \code{.arff} file should already contain the pattern of interest in the first two rows. Consequently not more than 1 pattern at a time can be investigated with a single call of \code{bibit3}.

\emph{Note 2 - Viewing Results:}
\cr A \code{print} and \code{summary} method has been implemented for the output object of \code{bibit3}. It gives an overview of the amount of discovered biclusters and their dimensions
\cr Additionally, the \code{\link{bibit3_patternBC}} function can extract a Bicluster and add the artificial pattern rows to investigate the results.
}
\examples{
\dontrun{ 
set.seed(1)
data <- matrix(sample(c(0,1),100*100,replace=TRUE,prob=c(0.9,0.1)),nrow=100,ncol=100)
data[1:10,1:10] <- 1 # BC1
data[11:20,11:20] <- 1 # BC2
data[21:30,21:30] <- 1 # BC3
colsel <- sample(1:ncol(data),ncol(data))
data <- data[sample(1:nrow(data),nrow(data)),colsel]

pattern_matrix <- matrix(0,nrow=3,ncol=100)
pattern_matrix[1,1:7] <- 1
pattern_matrix[2,11:15] <- 1
pattern_matrix[3,13:20] <- 1

pattern_matrix <- pattern_matrix[,colsel]


out <- bibit3(matrix=data,minr=2,minc=2,noise=0.1,pattern_matrix=pattern_matrix,
              subpattern=TRUE,extend_columns=TRUE,pattern_combinations=TRUE)
out  # OR print(out) OR summary(out)


bibit3_patternBC(result=out,matrix=data,pattern=c(1),type=c("full","sub","ext"),BC=c(1,2))
}
}
\author{
Ewoud De Troyer
}
\references{
Domingo S. Rodriguez-Baena, Antonia J. Perez-Pulido and Jesus S. Aguilar-Ruiz (2011), "A biclustering algorithm for extracting bit-patterns from binary datasets", \emph{Bioinformatics}
}

