% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_experiment.R
\name{run_experiment}
\alias{run_experiment}
\title{Run a full experiment}
\usage{
run_experiment(Instance.list, Algorithm.list, power, d, sig.level = 0.05,
  alternative = "two.sided", test.type = "t.test", se.max, dif,
  method = "param", nstart = 20, nmax = 1000, seed = NULL,
  boot.R = 999, force.balanced = FALSE)
}
\arguments{
\item{Instance.list}{list object containing the definitions of the
\emph{available} instances. this list may (or may not) be exhausted in the
experiment. To estimate the number of required instances,
see \code{\link[=calc_instances]{calc_instances()}}. For more detail on the definition of each
instance, see \code{\link[=calc_nreps2]{calc_nreps2()}}.}

\item{Algorithm.list}{list object containing the definitions of the
algorithms to be compared. See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{power}{(desired) test power. See \code{\link[=calc_instances]{calc_instances()}} for details.}

\item{d}{minimally relevant effect size (MRES), expressed as a standardized
effect size, i.e., "deviation from H0" / "standard deviation".
See \code{\link[=calc_instances]{calc_instances()}} for details.}

\item{sig.level}{significance level (alpha) for the experiment.
See \code{\link[=calc_instances]{calc_instances()}} for details.}

\item{alternative}{type of alternative hypothesis ("two.sided" or
"one.sided"). See \code{\link[=calc_instances]{calc_instances()}} for details.}

\item{test.type}{type of test ("t.test", "wilcoxon", "binomial").
See \code{\link[=calc_instances]{calc_instances()}} for details.}

\item{se.max}{desired upper limit for the standard error of the estimated
difference between the two algorithms on each instance.
See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{dif}{type of difference to be used on each instance. Accepts "perc"
(for percent differences) or "simple" (for simple differences).
See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{method}{method to use for estimating the standard errors. Accepts
"param" (for parametric) or "boot" (for bootstrap).
See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{nstart}{initial number of algorithm runs for each algorithm in each
instance. See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{nmax}{maximum total allowed sample size in each instance
See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{seed}{seed for the random number generator}

\item{boot.R}{number of bootstrap resamples. See \code{\link[=calc_nreps2]{calc_nreps2()}} for details.}

\item{force.balanced}{logical flag to force the use of balanced sampling for
the algorithms on each instance}
}
\value{
a list object containing the full input configuration plus the
following fields:
\itemize{
\item \code{data.raw} - data frame containing all observations generated
\item \code{data.summary} - data frame summarizing the experiment.
\item \code{N} - number of instances sampled
\item \code{N.star} - number of instances required
\item \code{instances.sampled} - names of the instances sampled
\item \code{Underpowered} - flag: TRUE if N < N.star
}
}
\description{
Design and run a full experiment - calculate the required number of
instances, run the algorithms on each problem instance using the iterative
approach based on optimal sample size ratios, and return the results of the
experiment. This routing builds upon \code{\link[=calc_instances]{calc_instances()}} and \code{\link[=calc_nreps2]{calc_nreps2()}},
so refer to the documentation of these two functions for details.
}
\section{Instance List}{

Parameter \code{Instance.list} must contain a list of instance objects, where
each field is itself a list, as defined in Section _Instances and Algorithms
of the documentation of _\code{\link[=calc_nreps2]{calc_nreps2()}}. In summary, each element of
\code{Instance.list} is an \code{instance}, i.e., a named list containing all relevant
parameters that define the problem instance. This list must contain at least
the field \code{instance$FUN}, with the name of the problem instance function,
that is, a routine that calculates y = f(x). If the instance requires
additional parameters, these must also be provided as named fields.
An additional field, "instance$alias", can be used to
provide the instance with a unique identifier (e.g., when using an
instance generator).
}

\section{Algorithms}{

Parameter \code{Algorithm.list} must contain a list of instance objects, where
each field is itself a list, as defined in Section \emph{Instances and Algorithms}
of the documentation of \code{\link[=calc_nreps2]{calc_nreps2()}}. In summary, each element of
\code{Algorithm.list} is an \code{algorithm}, i.e., a named list containing all
relevant parameters that define the algorithm.

An \code{algorithm} must contain a \code{algorithm$FUN} field (the name
of the function that calls the algorithm) and any other elements/parameters
that \code{algorithm$FUN} requires (e.g., stop criteria, operator names and
parameters, etc.). An additional field, \code{algorithm$alias}, can be used to
provide the algorithm with a unique identifier (e.g., when comparing two
different configurations of the same algorithm).

The function defined by the routine \code{algorithm$FUN} must have the
following structure: supposing that the list in \code{algorithm} has
fields \code{algorithm$FUN = myalgo} and
\code{algorithm$par1 = "a", algorithm$par2 = 5}, then:

\preformatted{
         myalgo <- function(par1, par2, instance, ...){
               # do stuff
               # ...
               return(results)
         }
   }

That is, it must be able to run if called as:

\preformatted{
         # remove '$FUN' and '$alias' from list of arguments
         # and include the problem definition as field 'instance'
         myargs          <- algorithm[names(algorithm) != "FUN"]
         myargs          <- myargs[names(myargs) != "alias"]
         myargs$instance <- instance

         # call function
         do.call(algorithm$FUN,
                 args = myargs)
   }

The \code{algorithm$FUN} routine must return a list object containing (at
least) the performance value of the final solution obtained after a given
run, in a field named \code{value} (e.g., \code{result$value}) .
}

\section{Initial Number of Observations}{

In the \emph{general case} the initial number of observations / algorithm /
instance (\code{nstart}) should be relatively high. For the parametric case
we recommend ~15 if outliers are not expected, ~50 (at least) if that
assumption cannot be made. For the bootstrap approch we recommend using at
least 15 or 20. However, if some distributional assumptions can be
made - particularly low skewness of the population of algorithm results on
the test instances), then \code{nstart} can in principle be as small as 5 (if the
output of the algorithm were known to be normal, it could be 1).

In general, higher sample sizes are the price to pay for abandoning
distributional assumptions. Use lower values of \code{nstart} with caution.
}

\section{Types of Differences}{

Parameter \code{dif} informs the type of difference in performance to be used
for the estimation (mu1 and mu2 represent the mean performance of each
algorithm on the problem instance):
\itemize{
\item If \code{dif == "perc"} it estimates (mu2 - mu1) / mu1.
\item If \code{dif == "simple"} it estimates mu2 - mu1.
}
}

\section{Sample Sizes for Nonparametric Methods}{

If the parameter \code{test.type} is set to either \code{Wilcoxon} or \code{Binomial}, this
routine approximates the number of instances using the ARE of these tests
in relation to the paired t.test, using the formulas:
\itemize{
\item \code{n.wilcox = n.ttest / 0.86 = 1.163 * n.ttest}
\item \code{n.binom = n.ttest / 0.637 = 1.570 * n.ttest}
}
}

\examples{
# Example using dummy algorithms and instances. See ?dummyalgo for details.
# In this case all instances are the same, so we expect all cases to return
# a percent difference of approx. phi.j = 1.0 and sample sizes of
# approx. n1 = 31, n2 = 87
algorithm1 <- list(FUN = "dummyalgo", alias = "algo1",
                   distribution.fun = "rnorm",
                   distribution.pars = list(mean = 10, sd = 1))
algorithm2 <- list(FUN = "dummyalgo", alias = "algo2",
                   distribution.fun = "rnorm",
                   distribution.pars = list(mean = 20, sd = 4))
algolist <- list(algorithm1, algorithm2)
instlist <- vector(100, mode = "list")
for (i in 1:100) instlist[[i]] <- list(FUN = "dummyinstance",
                                       alias = paste0("Inst. ", i))

my.results <- run_experiment(Instance.list = instlist,
                             Algorithm.list = algolist,
                             power = 0.8,
                             d = 1,
                             sig.level = 0.01,
                             se.max = 0.05,
                             dif = "perc",
                             nmax   = 200,
                             seed   = 1234)

# Take a look at the summary table
my.results$data.summary

# To perform inference on the results:
t.test(my.results$data.summary$phi.j, conf.level = 0.95)

# Test assumption of normality (of the data)
shapiro.test(my.results$data.summary$phi.j)
}
\references{
\itemize{
\item F. Campelo, F. Takahashi:
Sample size estimation for power and accuracy in the experimental
comparison of algorithms (submitted, 2017).
\item P. Mathews.
Sample size calculations: Practical methods for engineers and scientists.
Mathews Malnar and Bailey, 2010.
\item A.C. Davison, D.V. Hinkley:
Bootstrap methods and their application. Cambridge University Press (1997)
\item E.C. Fieller:
Some problems in interval estimation. Journal of the Royal Statistical
Society. Series B (Methodological) 16(2), 175–185 (1954)
\item V. Franz:
Ratios: A short guide to confidence limits and proper use (2007).
https://arxiv.org/pdf/0710.2024v1.pdf
\item D.C. Montgomery, C.G. Runger:
Applied Statistics and Probability for Engineers, 6th ed. Wiley (2013)
}
}
\author{
Felipe Campelo (\email{fcampelo@ufmg.br})
}
