# Copyright 2022 DARWIN EU (C)
#
# This file is part of DrugUtilisation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

#' This function is used to summarise the dose table over multiple cohorts.
#'
#' @param cohort Cohort with drug use variables and strata
#' @param cdm cdm_reference generated by CDMConnector
#' @param strata Stratification list
#' @param drugUseEstimates Estimates that we want for the columns
#' @param minCellCount Below this number counts will be supressed
#'
#' @return A summary of the drug use stratified by cohort_name and strata_name
#'
#' @export
#'
#' @examples
#' \donttest{
#' library(DrugUtilisation)
#' library(PatientProfiles)
#' library(CodelistGenerator)
#'
#' cdm <- mockDrugUtilisation()
#' cdm <- generateDrugUtilisationCohortSet(
#'   cdm, "dus_cohort", getDrugIngredientCodes(cdm, "acetaminophen")
#' )
#' cdm[["dus_cohort"]] <- cdm[["dus_cohort"]] %>%
#'   addDrugUse(cdm, 1125315)
#' result <- summariseDrugUse(cdm[["dus_cohort"]], cdm)
#' print(result)
#'
#' cdm[["dus_cohort"]] <- cdm[["dus_cohort"]] %>%
#'   addSex(cdm) %>%
#'   addAge(cdm, ageGroup = list("<40" = c(0, 30), ">40" = c(40, 150)))
#'
#' summariseDrugUse(
#'   cdm[["dus_cohort"]], cdm, strata = list(
#'    "age_group" = "age_group", "sex" = "sex",
#'    "age_group and sex" = c("age_group", "sex")
#'   )
#' )
#' }
#'
summariseDrugUse<- function(cohort,
                            cdm = lifecycle::deprecated(),
                            strata = list(),
                            drugUseEstimates = c(
                              "min", "q05", "q25", "median", "q75", "q95",
                              "max", "mean", "sd", "missing"
                            ),
                            minCellCount = lifecycle::deprecated()) {
  if (lifecycle::is_present(cdm)) {
    lifecycle::deprecate_warn(when = "0.5.0", what = "summariseDrugUse(cdm = )")
  }
  if (lifecycle::is_present(minCellCount)) {
    lifecycle::deprecate_warn(when = "0.5.0", what = "summariseDrugUse(minCellCount = )")
  }
  cdm <- omopgenerics::cdmReference(cohort)
  # check inputs
  checkInputs(
    cohort = cohort, cdm = cdm, strata = strata,
    drugUseEstimates = drugUseEstimates
  )

  # update cohort_names
  cohort <- cohort %>% PatientProfiles::addCohortName() %>% dplyr::collect()

  # summarise drug use columns
  result <- PatientProfiles::summariseResult(
    table = cohort, group = list("cohort_name" = "cohort_name"),
    strata = strata, variables = drugUseColumns(cohort),
    functions = drugUseEstimates
  ) %>%
    dplyr::mutate(
      cdm_name = dplyr::coalesce(omopgenerics::cdmName(cdm), as.character(NA)),
      result_type = "summarised_drug_use",
      package_name = "DrugUtilisation",
      package_version = as.character(utils::packageVersion("DrugUtilisation"))
    ) |>
    omopgenerics::newSummarisedResult() |>
    omopgenerics::suppress(minCellCount = 5)

  return(result)
}

#' Obtain automatically the drug use columns
#'
#' @param cohort A cohort
#'
#' @return Name of the drug use columns
#'
#' @noRd
#'
drugUseColumns <- function(cohort) {
  cohort %>%
    dplyr::select(
      dplyr::any_of(c(
        "number_exposures", "duration", "cumulative_quantity", "number_eras",
        "initial_quantity", "impute_daily_dose_percentage", "impute_duration_percentage"
      )),
      dplyr::starts_with("initial_daily_dose"),
      dplyr::starts_with("cumulative_dose")
    ) %>%
    colnames()
}
