% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group_design.R
\name{group_design}
\alias{group_design}
\title{Create Group-Level Design Matrices}
\usage{
group_design(formula, data, subject_design, contrasts = NULL)
}
\arguments{
\item{formula}{A list of formulas specifying the relationship between subject-level parameters
and group-level predictors. Each formula should have a subject-level parameter on the left-hand side
and group-level predictors on the right-hand side.}

\item{data}{The same data as used in the subject-level design. Must include a 'subjects' column.}

\item{subject_design}{An emc.design object containing the subject-level design.}

\item{contrasts}{Optional list of contrast matrices to be used for categorical predictors.}
}
\value{
A list of design matrices, one for each parameter specified in the formula. The intercept is
automatically included as the group-level mean and is omitted from the design matrices.
}
\description{
Creates design matrices for group-level parameters based on subject-level design and formulas.
This function is used for hierarchical modeling to specify how subject-level parameters
vary across groups or conditions.
}
\details{
Here it is important to consider the interpretation of the group-level mean. This allows
one to add covariates/group-level factors to the model. However, mu, the group-level mean, is still
included for all parameters. Mu represents the intercept in the design matrix, this intercept is always
added to the group-level model. Therefore, to keep the interpretation of mu as the group-level mean,
it is important to ensure that the design matrix has a mean of zero. If not, this function will throw a
warning. For some unbalanced designs, this is unavoidable and the warning can be ignored.
}
\examples{
# Create subject-level design 
subj_design <- design(data = forstmann, model = DDM,
                      formula = list(v ~ S, a ~ E, t0 ~ 1),
                      contrasts = list(S = contr.helmert))
# Add some age covariate and roughly demeans
# Demeaning is important to ensure that the interpretation of the group-level intercept
# is the mean of the group (i.e., 'mu' still represents the group-level mean)
forstmann$age <- as.numeric(forstmann$subjects) -mean(as.numeric(forstmann$subjects))
# Create fake group column
forstmann$group <- ifelse(forstmann$subjects \%in\%
              unique(forstmann$subjects)[seq(1, 19, 2)], "A", "B")

# Create group-level design matrices
group_des <- group_design(
  formula = list(v_S1 ~ age + group, a ~ age),
  data = forstmann,
  subject_design = subj_design,
  contrasts = list(group = contr.bayes)
)
# Then you can make the emc object with
emc <- make_emc(forstmann, subj_design, compress = FALSE, group_design = group_des)
}
