% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/Laplace.R
\name{Laplace}
\alias{Laplace}
\alias{dLaplace}
\alias{eLaplace}
\alias{lLaplace}
\alias{pLaplace}
\alias{qLaplace}
\alias{rLaplace}
\title{The Laplace Distribution.}
\usage{
dLaplace(x, mu = 0, b = 1, params = list(mu, b), ...)

pLaplace(q, mu = 0, b = 1, params = list(mu, b), ...)

qLaplace(p, mu = 0, b = 1, params = list(mu, b), ...)

rLaplace(n, mu = 0, b = 1, params = list(mu, b), ...)

eLaplace(X, w, method = c("analytic.MLE", "numerical.MLE"), ...)

lLaplace(x, w = 1, mu = 0, b = 1, params = list(mu, b), logL = TRUE,
  ...)
}
\arguments{
\item{x,q}{A vector of quantiles.}

\item{mu}{Location parameter.}

\item{b}{Scale parameter.}

\item{params}{A list that includes all named parameters}

\item{...}{Additional parameters.}

\item{p}{A vector of probabilities.}

\item{n}{Number of observations.}

\item{X}{Sample observations.}

\item{w}{Optional vector of sample weights.}

\item{method}{Parameter estimation method.}

\item{logL}{logical; if TRUE, lLaplace gives the log-likelihood, otherwise the likelihood is given.}
}
\value{
dLaplace gives the density, pLaplace the distribution function,
qLaplace the quantile function, rLaplace generates random deviates, and
eLaplace estimates the distribution parameters. lLaplace provides the log-likelihood function, sLaplace the score function,
and iLaplace the observed information matrix.
}
\description{
Density, distribution, quantile, random number generation
and parameter estimation functions for the Laplace distribution with \code{location} parameter \eqn{\mu} and
\code{scale} parameter \eqn{b}. Parameter estimation can for the Laplace distribution can be carried out numerically
or analytically but may only be based on an unweighted i.i.d. sample.
}
\details{
The \code{dLaplace()}, \code{pLaplace()}, \code{qLaplace()},and \code{rLaplace()} functions allow for the parameters to be declared not only as
individual numerical values, but also as a list so parameter estimation can be carried out. \cr
\cr
The Laplace distribution with parameters \code{location} = \eqn{\mu} and \code{scale}=\eqn{b} has probability density
function
\deqn{f(x) = (1/2b) exp(-|x-\mu|/b)}
where \eqn{-\infty < x < \infty} and \eqn{b > 0}. The cumulative distribution function for \code{pLaplace} is defined
by Johnson et.al (p.166). \cr
\cr
Parameter estimation can be carried out analytically via maximum likelihood estimation, see Johnson et.al (p.172). Where the population
mean, \eqn{\mu}, is estimated using the sample median and \eqn{b} by the mean of \eqn{|x-b|}.\cr
\cr
Johnson et.al (p.172) also provides the log-likelihood function for the Laplace distribution
\deqn{l(\mu, b | x) = -n ln(2b) - b^{-1} \sum |xi-\mu|.}
}
\note{
The estimation of the population mean is done using the median of the sample. Unweighted
samples are not yet catered for in the eLaplace() function.
}
\examples{
# Parameter estimation for a distribution with known shape parameters
X <- rLaplace(n=500, mu=1, b=2)
est.par <- eLaplace(X, method="analytic.MLE"); est.par
plot(est.par)

#  Fitted density curve and histogram
den.x <- seq(min(X),max(X),length=100)
den.y <- dLaplace(den.x, location = est.par$location, scale= est.par$scale)
hist(X, breaks=10, probability=TRUE, ylim = c(0,1.1*max(den.y)))
lines(den.x, den.y, col="blue")
lines(density(X), lty=2)

# Extracting location or scale parameters
est.par[attributes(est.par)$par.type=="location"]
est.par[attributes(est.par)$par.type=="scale"]

# Parameter estimation for a distribution with unknown shape parameters
# Example from Best et.al (2008). Original source of flood data from Gumbel & Mustafi.
# Parameter estimates as given by Best et.al mu=10.13 and  b=3.36
flood <- c(1.96, 1.96, 3.60, 3.80, 4.79, 5.66, 5.76, 5.78, 6.27, 6.30, 6.76, 7.65, 7.84, 7.99,
           8.51, 9.18, 10.13, 10.24, 10.25, 10.43, 11.45, 11.48, 11.75, 11.81, 12.34, 12.78, 13.06,
           13.29, 13.98, 14.18, 14.40, 16.22, 17.06)
est.par <- eLaplace(flood, method="numerical.MLE"); est.par
plot(est.par)

#log-likelihood function
lLaplace(flood,param=est.par)

# Evaluating the precision by the Hessian matrix
H <- attributes(est.par)$nll.hessian
var <- solve(H)
se <- sqrt(diag(var));se
}
\author{
A. Jonathan R. Godfrey and Haizhen Wu. \cr
Updates and bug fixes by Sarah Pirikahu
}
\references{
Johnson, N. L., Kotz, S. and Balakrishnan, N. (1995) Continuous Univariate Distributions,
 volume 2, chapter 24, Wiley, New York.\cr
 \cr
 Best, D.J., Rayner, J.C.W. and Thas O. (2008) Comparison of some tests of fit for the Laplace distribution,
 Computational Statistics and Data Analysis, Vol. 52, pp.5338-5343.\cr
 \cr
 Gumbel, E.J., Mustafi, C.K., 1967. Some analytical properties of bivariate extremal distributions.
 J. Am. Stat. Assoc. 62, 569-588
}
\seealso{
\pkg{\link{ExtDist}} for other standard distributions.
}

