\name{select.colnames.lsd}
\alias{select.colnames.lsd}
\title{
Select a subset of a LSD results matrix (by column/variable names)
}
\description{
This function select a subset of a LSD results matrix (as produced by \code{\link{read.raw.lsd}}) by the column (variable) names, considering only the name part of the column labels.
}
\usage{
select.colnames.lsd( dataSet, col.names = NULL, instance = 0,
                     check.names = TRUE, posit = NULL,
                     posit.match = c( "fixed", "glob", "regex" ) )
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{dataSet}{
matrix produced by the invocation of \code{\link{read.raw.lsd}}, \code{\link{read.single.lsd}}, \code{\link{read.multi.lsd}} or \code{\link{read.list.lsd}} (a single matrix a time) functions.
}
  \item{col.names}{
a vector of optional names for the variables. The default is to read all variables. The names must to be in LSD/C++ format, without dots (".") in the name. Any dot (and trailing characters) will be automatically removed.
}
  \item{instance}{
integer: the instance of the variable to be read, for variables that exist in more than one object. This number is based on the relative position (column) of the variable in the results file. The default (0) is to read all instances.
}
  \item{check.names}{
logical. If \code{TRUE} the names of the variables are checked to ensure that they are syntactically valid variable names. If necessary they are adjusted to ensure that there are no duplicates.
}
  \item{posit}{
a string, a vector of strings or an integer vector describing the LSD object position of the variable(s) to select. If an integer vector, it should define the position of a SINGLE LSD object. If a string or vector of strings, each element should define one or more different LSD objects, so the returning matrix will contain variables from more than one object. By setting \code{posit.match}, globbing (wildcard), and regular expressions can be used to select multiple objects at once; in this case, all matching objects are returned. This option only operates if \code{dataSet} was generated by \code{read.raw.lsd} WITHOUT argument \code{clean.names = TRUE}.
}
  \item{posit.match}{
a string defining how the \code{posit} argument, if provided, should be matched against the LSD object positions. If equal to \code{"fixed"}, the default, only exact matching is done. \code{"glob"} allows using simple wildcard characters ('*' and '?') in \code{posit} for matching. If \code{posit.match="regex"} interpret \code{posit} as POSIX 1003.2 extended regular expression(s). See \code{\link[=regex]{regular expressions}} for details of the different types of regular expressions. Options can be abbreviated.
}
}
\details{
Selection restriction arguments can be provided as needed; when not specified, all available cases are selected.

The selection of specific \code{posit} object positions require full detail on \code{dataSet} column names, as produced by \code{read.raw.lsd} and \code{clean.names = TRUE} is NOT used. Other \code{read.XXXX.lsd} functions do NOT produce the required detail on the data matrices to do object position selection. If such datasets are used to feed this function and \code{posit} is set, the return value will be NULL. In this case, consider using \code{select.colattrs.lsd}, or specifying \code{posit} when calling \code{read.XXXX.lsd} functions.

When \code{posit} is supplied together with other attribute filters, the selection process is done in two steps. Firstly, the column names set by otter attribute filters are selected. Secondly, the instances defined by \code{posit} are selected from the first selection set.

See also the \code{read.XXXX.lsd} functions which may select just specific \code{col.names} columns, \code{instance} instances, or \code{posit} positions when loading LSD results. If only a single set of columns/instance/positions is required, this may be more efficient than using this function.
}
\value{
Returns a single matrix containing the selected variables' time series contained in the original data set.
}
\author{Marcelo C. Pereira}
\note{
The variable/column names must be valid R or LSD column names.
}
\seealso{
\code{\link{list.files.lsd}()},
\code{\link{select.colattrs.lsd}()},
\code{\link{read.raw.lsd}()}
}
\examples{
# get the list of file names of example LSD results
files <- list.files.lsd( system.file( "extdata", package = "LSDinterface" ) )

# read all instances of all variables in first file
bigTable <- read.raw.lsd( files[ 1 ] )
print( bigTable[ 1 : 10, 1 : 7 ] )

# extract all instances of a set of variables named '_A1p' and '_growth1'
abTable <- select.colnames.lsd( bigTable, c( "_A1p", "_growth1" ) )
print( abTable[ 11 : 15, ] )

# extract specific instances of a set of variables named '_A1p' and '_growth1'
abFirst2 <- select.colnames.lsd( bigTable, c( "_A1p", "_growth1" ),
                                 posit = c( "1_2", "1_5" ) )
print( abFirst2[ 50 : 60, ] )

# extract all second-level object instances of all variables
aSec <- select.colnames.lsd( bigTable, posit = "*_*", posit.match = "glob" )
print( aSec[ 1 : 10, ] )

# extract just top-level object instances variables
aTop <- select.colnames.lsd( bigTable, posit = "^[0-9]+$", posit.match = "regex" )
print( aTop[ 1 : 10, ] )
}
\keyword{ datasets }
\keyword{ database }
