\name{analyse_IRSAR.RF}
\alias{analyse_IRSAR.RF}
\title{Analyse IRSAR RF measurements}
\description{Function to analyse IRSAR RF measurements on K-feldspar samples, performed 
using the protocol according to Erfurt et al. (2003)}
\usage{analyse_IRSAR.RF(object, sequence.structure = c("NATURAL", "REGENERATED"), 
    method = "FIT", fit.range.min, fit.range.max, fit.trace = FALSE, 
    fit.MC.runs = 10, slide.outlier.rm = FALSE, slide.trend.corr = FALSE, 
    output.plot = TRUE, xlab.unit = "s", legend.pos, ...)}
\arguments{
  \item{object}{\code{\linkS4class{RLum.Analysis}} (\bold{required}): 
input object containing data for protocol analysis}
  \item{sequence.structure}{\code{\link{vector}} \link{character} (with default): specifies the general 
sequence structure. Allowed steps are \code{NATURAL}, \code{REGENERATED}.
In addition any other character is allowed in the sequence structure; 
such curves will be ignored. }
  \item{method}{\code{\link{character}} (with default): setting method applied for the data analysis.
Possible options are \code{"FIT"} or \code{"SLIDE"}.}
  \item{fit.range.min}{\code{\link{integer}} (optional): set the minimum channel range for signal fitting and sliding.   
Usually the entire data set is used for curve fitting, but there might be 
reasons to limit the channels used for fitting.
Note: This option also limits the values used for natural signal calculation.}
  \item{fit.range.max}{\code{\link{integer}} (optional): set maximum channel range for signal fitting and sliding. 
Usually the entire data set is used for curve fitting, but there might be 
reasons to limit the channels used for fitting.}
  \item{fit.trace}{\code{\link{logical}} (with default): trace fitting (for debugging use)}
  \item{fit.MC.runs}{\code{\link{numeric}} (with default): set number of Monte Carlo runs for start 
parameter estimation. Note: Higher values will significantly increase 
the calculation time.   }
  \item{slide.outlier.rm}{\code{\link{logical}} (with default): enable or disable outlier removal. 
Outliers are removed from the natural signal curve only.}
  \item{slide.trend.corr}{\code{\link{logical}} (with default): enable or disable trend correction. 
If \code{TRUE}, the sliding is applied to a previously trend corrected data set.}
  \item{output.plot}{\code{\link{logical}} (with default): plot output (\code{TRUE} or \code{FALSE})}
  \item{xlab.unit}{\code{\link{character}} (with default): set unit for x-axis}
  \item{legend.pos}{\code{\link{character}} (with default): useful keywords are \code{bottomright}, 
\code{bottom}, \code{bottomleft}, \code{left}, \code{topleft}, 
\code{top}, \code{topright}, \code{right} and \code{center}. 
For further details see \code{\link{legend}.}}
  \item{\dots}{further arguments that will be passed to the plot output. 
Currently supported arguments are \code{main}, \code{xlab}, \code{ylab}.}
}
\details{The function performs an IRSAR analysis described for K-feldspar samples by 
Erfurt et al. (2003) assuming a negligible sensitivity change of the RF signal.\cr
\bold{General Sequence Structure} (according to Erfurt et al. (2003))
\enumerate{
\item Measuring IR-RF intensity of the natural dose for a few seconds (\eqn{D_{natural}})     
\item Bleach the samples under solar conditions for at least 30 min without changing 
the geometry 
\item Waiting for at least one hour 
\item Regeneration of the IR-RF signal to at least the natural level
\item Fitting data with a stretched exponential function
\item Calculate the the palaeodose \eqn{D_{e}} using the parameters from the fitting
}
\bold{Function Used For The Fitting} (according to Erfurt et al. (2003))\cr
\deqn{\phi(D) = \phi_{0}-\Delta\phi(1-exp(-\lambda*D))^\beta}
with \eqn{\phi(D)} the dose dependent IR-RF flux, \eqn{\phi_{0}} the inital IR-RF flux, 
\eqn{\Delta\phi} the dose dependent change of the IR-RF flux, \eqn{\lambda} 
the exponential parameter, \eqn{D} the dose and \eqn{\beta} the dispersive 
factor.\cr\cr
To obtain the palaeodose \eqn{D_{e}} the function is changed to:\cr
\deqn{D_{e} = ln(-(\phi(D) - \phi_{0})/(-\lambda*\phi)^{1/\beta}+1)/-\lambda}\cr
The fitting is done using the \code{port} algorithm of the \code{\link{nls}} function.\cr

Two methods are supported to obtain the De:\cr

\bold{\code{method = "FIT"}}\cr

The principle is described above and follows the orignal suggestions from Erfurt et al., 2003.\cr

\bold{\code{method = "SLIDE"}}\cr

For this method the natural curve is slided along the x-axis until congruence with the 
regenerated curve is reached. Instead of fitting this allows to work with the original data 
without the need of any physical model. 
This approach was introduced for RF curves by Buylaert et al., 2012 and 
Lapp et al., 2012.

Here the sliding is done by searching for the minimum of the residual squares. 

\deqn{min(\Sigma(RF.reg_{k.i} - RF.nat_{k.i})^2)}
for 
\deqn{k = {t.0+i,...,t.max+i}}


\bold{Correction for outliers} (\code{slide.outlier.rm = TRUE})\cr

By using \code{method = "SLIDE"} and setting the argument \code{slide.outlier.rm = TRUE}
an automatic outlier removal can be applied to the natural curve. Outliers may be observed also
on the regenerative curve, but here the impact of single outliers on the curve adjustment (sliding)
is considered as negligible. \cr
The applied outlier removal algorithm consists of three steps:\cr

(a) Input data are smoothed using the function \code{\link{rollmedian}}. Value \code{k} for 
the rolling window is fixed to 11. Therefore, the natural curve needs to comprise at least of 33 
values, otherwise outlier removal is rejected. \cr

(b) To subsequently remove outliers, code blocks from the function \code{\link{apply_CosmicRayRemoval}} 
were recycled, therefore in general the outlier correction works as described by Pych (2003). 
In contrast, here no sigma clipping before constructing the histograms is applied.\cr

(c) Outliers are marked in the data set and visualised in the graphical output. The subsequent
adjustement of both curves (natural and regenerative) is done without outliers, whereas the 
sliding itself is done with the entire data set.\cr


\bold{Trend correction} (\code{slide.trend.corr = TRUE})\cr

This option allows for correcting any linear trend in the natural curve in comparison to the 
regenerative curve. The trend correction is based on regression analysis of the residuals from 
the slided curve. The corrected De is obtained by sliding the trend corrected values (again)
along the regenerative data curve. This correction is driven by the idea that the 
rediduals from the regenerative and the natural curve should be free of any trend, as long as 
they are comparable. \cr

\bold{Error estimation}

For \bold{\code{method = "FIT"}} the asymmetric error range is taken from the standard deviation
of the natural signal.\cr

For \bold{\code{method = "SLIDE"}} so far no error estimation is implemented. Instead, to asses
the error of the De several aliquots should be measured and the error obtained from the 
De distribution.}
\value{A plot (optional) and an \code{\linkS4class{RLum.Results}} object is returned 
containing the following elements: \cr
\item{De.values}{\code{\link{data.frame}} containing De-values with error 
(gray dashed lines in the plot) and further parameters. Corrected De values are only 
provided for the method \code{"SLIDE"}, provided the trend correction is applied.}
\item{fit}{\link{nls} \code{nlsModel} object}\cr
\bold{Note:} The output (\code{De.values}) should be accessed using the 
function \code{\link{get_RLum.Results}}}
\references{Buylaert, J.P., Jain, M., Murray, A.S., Thomsen, K.J., Lapp, T., 2012. 
IR-RF dating of sand-sized K-feldspar extracts: A test of accuracy. 
Radiation Measurements 44 (5-6), 560-565. doi: 10.1016/j.radmeas.2012.06.021

Erfurt, G., Krbetschek, M.R., 2003. IRSAR - A single-aliquot regenerative-dose 
dating protocol applied to the infrared radiofluorescence (IR-RF) of coarse- grain 
K-feldspar. Ancient TL 21, 35-42.

Erfurt, G., 2003. Infrared luminescence of Pb+ centres in potassium-rich feldspars. 
physica status solidi (a) 200, 429-438.

Erfurt, G., Krbetschek, M.R., 2003. Studies on the physics of the infrared 
radioluminescence of potassium feldspar and on the methodology of its application 
to sediment dating. Radiation Measurements 37, 505-510.

Erfurt, G., Krbetschek, M.R., Bortolot, V.J., Preusser, F., 2003. 
A fully automated multi-spectral radioluminescence reading system for 
geochronometry and dosimetry. Nuclear Instruments and Methods in Physics Research 
Section B: Beam Interactions with Materials and Atoms 207, 487-499.

Lapp, T., Jain, M., Thomsen, K.J., Murray, A.S., Buylaert, J.P., 2012. New luminescence measurement
facilities in retrospective dosimetry. Radiation Measurements 47, 803-808. 
doi:10.1016/j.radmeas.2012.02.006

Pych, W., 2003. A Fast Algorithm for Cosmic-Ray Removal from Single Images.
Astrophysics 116, 148-153.
\url{http://arxiv.org/pdf/astro-ph/0311290.pdf?origin=publication_detail}

Trautmann, T., 2000. A study of radioluminescence kinetics of natural feldspar 
dosimeters: experiments and simulations. Journal of Physics D: Applied Physics 33, 2304-2310.

Trautmann, T., Krbetschek, M.R., Dietrich, A., Stolz, W., 1998. 
Investigations of feldspar radioluminescence: potential for a new dating technique. 
Radiation Measurements 29, 421-425.

Trautmann, T., Krbetschek, M.R., Dietrich, A., Stolz, W., 1999. Feldspar 
radioluminescence: a new dating method and its physical background. 
Journal of Luminescence 85, 45-58.

Trautmann, T., Krbetschek, M.R., Stolz, W., 2000. A systematic study of the 
radioluminescence properties of single feldspar grains. 
Radiation Measurements 32, 685-690.}
\author{Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne (France) \cr
R Luminescence Package Team}
\note{This function assumes that there is no sensitivity change during the measurements (natural
vs. regenerated signal), which is in contrast to the findings from Buylaert et al. (2012).}


\seealso{\code{\linkS4class{RLum.Analysis}}, \code{\linkS4class{RLum.Results}},
\code{\link{get_RLum.Results}}, \code{\link{nls}}}
\examples{
##load data
data(ExampleData.RLum.Analysis, envir = environment())

##perform analysis
temp <- analyse_IRSAR.RF(object = IRSAR.RF.Data) 

}

\keyword{datagen}
\section{Function version}{0.2.1 (2014-12-18 17:15:20)}
