% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse_FadingMeasurement.R
\name{analyse_FadingMeasurement}
\alias{analyse_FadingMeasurement}
\title{Analyse fading measurements and returns the fading rate per decade (g-value)}
\usage{
analyse_FadingMeasurement(object, structure = c("Lx", "Tx"), signal.integral,
  background.integral, t_star = "half", n.MC = 100, verbose = TRUE,
  plot = TRUE, plot.single = FALSE, ...)
}
\arguments{
\item{object}{\code{\linkS4class{RLum.Analysis}} (\bold{required}): input object with the
measurement data. Alternatively, a \code{\link{list}} containing \code{\linkS4class{RLum.Analysis}}
objects or a \code{\link{data.frame}} with three columns
(x = LxTx, y = LxTx error, z = time since irradiation) can be provided.
Can also be a wide table, i.e. a \code{\link{data.frame}} with a number of colums divisible by 3
and where each triplet has the before mentioned column structure.}

\item{structure}{\code{\link{character}} (with default): sets the structure of the measurement
data. Allowed are \code{'Lx'} or \code{c('Lx','Tx')}. Other input is ignored}

\item{signal.integral}{\code{\link{vector}} (\bold{required}): vector with the
limits for the signal integral. Not required if a \code{data.frame} with LxTx values are
provided.}

\item{background.integral}{\code{\link{vector}} (\bold{required}): vector with the
bounds for the background integral. Not required if a \code{data.frame} with LxTx values are
provided.}

\item{t_star}{\code{\link{character}} (with default): method for calculating the time elasped
since irradiaton. Options are: \code{'half'}, which is \eqn{t_star := t_1 + (t_2 - t_1)/2} (Auclair et al., 2003)
and \code{'end'}, which takes the time between irradiation and the measurement step. Default is \code{'half'}}

\item{n.MC}{\code{\link{integer}} (with default): number for Monte Carlo runs for the error
estimation}

\item{verbose}{\code{\link{logical}} (with default): enables/disables verbose mode}

\item{plot}{\code{\link{logical}} (with default): enables/disables plot output}

\item{plot.single}{\code{\link{logical}} (with default): enables/disables single plot
mode, i.e. one plot window per plot. Alternatively a vector specifying the plot to be drawn, e.g.,
\code{plot.single = c(3,4)} draws only the last two plots}

\item{\dots}{(optional) further arguments that can be passed to internally used functions (see details)}
}
\value{
An \code{\linkS4class{RLum.Results}} object is returned:

Slot: \bold{@data}\cr

\tabular{lll}{
\bold{OBJECT} \tab \code{TYPE} \tab \code{COMMENT}\cr
\code{fading_results} \tab \code{data.frame} \tab results of the fading measurement in a table \cr
\code{fit} \tab \code{lm} \tab object returned by the used linear fitting function \code{\link[stats]{lm}}\cr
\code{rho_prime} \tab \code{data.frame} \tab results of rho' estimation after Kars et al. 2008 \cr
\code{LxTx_table} \tab \code{data.frame} \tab Lx/Tx table, if curve data had been provided \cr
\code{irr.times} \tab \code{integer} \tab vector with the irradiation times in seconds \cr
}

Slot: \bold{@info}\cr

\tabular{lll}{
\bold{OBJECT} \tab \code{TYPE} \tab \code{COMMENT}\cr
\code{call} \tab \code{call} \tab the original function call\cr

}
}
\description{
The function analysis fading measurements and returns a fading rate including an error estimation.
The function is not limited to standard fading measurements, as can be seen, e.g., Huntley and
Lamothe 2001. Additionally, the density of recombination centres (rho') is estimated after
Kars et al. 2008.
}
\details{
All provided output corresponds to the \eqn{tc} value obtained by this analysis. Additionally
in the output object the g-value normalised to 2-days is provided. The output of this function
can be passed to the function \code{\link{calc_FadingCorr}}.\cr

\bold{Fitting and error estimation}\cr

For the fitting the function \code{\link[stats]{lm}} is used without applying weights. For the
error estimation all input values, except tc, as the precision can be consdiered as sufficiently
high enough with regard to the underlying problem, are sampled assuming a normal distribution
for each value with the value as the mean and the provided uncertainty as standard deviation. \cr

\bold{Density of recombination centres}

The density of recombination centres, expressed by the dimensionless variable rho', is estimated
by fitting equation 5 in Kars et al. 2008 to the data. For the fitting the function
\code{\link[stats]{nls}} is used without applying weights. For the error estimation the same
procedure as for the g-value is applied (see above).
}
\note{
\bold{This function has BETA status and should not be used for publication work!}
}
\section{Function version}{
 0.1.4 (2017-02-04 12:39:20)
}

\examples{

## load example data (sample UNIL/NB123, see ?ExampleData.Fading)
data("ExampleData.Fading", envir = environment())

##(1) get fading measurement data (here a three column data.frame)
fading_data <- ExampleData.Fading$fading.data$IR50

##(2) run analysis
g_value <- analyse_FadingMeasurement(
fading_data,
plot = TRUE,
verbose = TRUE,
n.MC = 10)

##(3) this can be further used in the function
## to correct the age according to Huntley & Lamothe, 2001
results <- calc_FadingCorr(
age.faded = c(100,2),
g_value = g_value,
n.MC = 10)


} 

\section{How to cite}{
Kreutzer, S., Burow, C. (2017). analyse_FadingMeasurement(): Analyse fading measurements and returns the fading rate per decade (g-value). Function version 0.1.4. In: Kreutzer, S., Dietze, M., Burow, C., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J. (2017). Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.7.0. https://CRAN.R-project.org/package=Luminescence
}

\references{
Auclair, M., Lamothe, M., Huot, S., 2003. Measurement of anomalous fading for feldpsar IRSL using
SAR. Radiation Measurements 37, 487-492. doi:10.1016/S1350-4487(03)00018-0

Huntley, D.J., Lamothe, M., 2001. Ubiquity of anomalous fading in K-feldspars and the measurement
and correction for it in optical dating. Canadian Journal of Earth Sciences 38,
1093-1106. doi:10.1139/cjes-38-7-1093

Kars, R.H., Wallinga, J., Cohen, K.M., 2008. A new approach towards anomalous fading correction for feldspar
IRSL dating-tests on samples in field saturation. Radiation Measurements 43, 786-790. doi:10.1016/j.radmeas.2008.01.021
}
\seealso{
\code{\link{calc_OSLLxTxRatio}}, \code{\link{read_BIN2R}}, \code{\link{read_XSYG2R}},
\code{\link{extract_IrradiationTimes}}
}
\author{
Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne (France) \cr
Christoph Burow, University of Cologne (Germany)
\cr R Luminescence Package Team}
\keyword{datagen}
