% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_Activity2Concentration.R
\name{convert_Activity2Concentration}
\alias{convert_Activity2Concentration}
\title{Convert Nuclide Activities to Abundance and Vice Versa}
\usage{
convert_Activity2Concentration(data, input_unit = "activity", verbose = TRUE)
}
\arguments{
\item{data}{\link{data.frame} (\strong{required}):
provide dose rate data (activity or concentration) in three columns.
The first column indicates the nuclide, the 2nd column measured value and
in the 3rd column its error value. Allowed nuclide data are
\code{'U-238'}, \code{'Th-232'} and \code{'K-40'}. See examples for an example.}

\item{input_unit}{\link{character} (\emph{with default}):
specify unit of input data given in the dose rate data frame, choose between
\code{"activity"} (considered as given Bq/kg) and \code{"abundance"} (considered as given in mug/g or mass. \%).
The default value is \code{"activity"}}

\item{verbose}{\link{logical} (\emph{with default}):
enable/disable output to the terminal.}
}
\value{
Returns an \linkS4class{RLum.Results} object with a \link{data.frame} containing
input and newly calculated values. Please not that in the column header µg/g
is written as \code{mug/g} due to the R requirement to maintain packages portable using
ASCII characters only.
}
\description{
The function performs the conversion of the specific activities into
mass abundance and vice versa for the radioelements U, Th, and K to
harmonise the measurement unit with the required data input unit of
potential analytical tools for, e.g. dose rate calculation or related
functions such as \link{use_DRAC}.
}
\details{
The conversion from nuclide activity of a sample to nuclide concentration
is performed using conversion factors that are based on the mass-related
specific activity of the respective nuclide.

Constants used in this function were obtained from \verb{https://physics.nist.gov/cuu/Constants/}
all atomic weights and composition values from
\verb{https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses}
and the nuclide data from \verb{https://www.iaea.org/resources/databases/livechart-of-nuclides-advanced-version}

The factors can be calculated using the equation:

\deqn{
A = N_A \frac{N_{abund}}{N_{mol.mass}}  ln(2) / N.half.life
}

to convert in µg/g we further use:

\deqn{
f = A / 10^6
}

where:
\itemize{
\item \code{N_A} - Avogadro constant in 1/mol
\item \code{A} - specific activity of the nuclide in Bq/kg
\item \code{N.abund} - relative natural abundance of the isotope
\item \code{N.mol.mass} molar mass in kg/mol
\item \code{N.half.life} half-life of the nuclide in s
}

example for calculating the activity of the radionuclide U-238:
\itemize{
\item \code{N_A} = 6.02214076e+23 (1/mol)
\item \code{T_0.5} = 1.41e+17 (s)
\item \code{m_U_238} = 0.23802891 (kg/mol)
\item \code{U_abund} = 0.992745 (unitless)
}

\deqn{A_{U} = N_{A} * U_{abund} / m_{U_238} * ln(2) / T_{1/2} = 2347046} (Bq/kg)

\deqn{f.U = A_{U} / 10^6}
}
\note{
Although written otherwise for historical reasons. Input values must be element values.
For instance, if a value is provided for U-238 the function assumes that this value
represents the sum (activity or abundance) of U-238, U-235 and U-234.
In other words, 1 µg/g of U means that this is the composition of 0.992 parts of U-238,
0.000054 parts of U-234, and 0.00072 parts of U-235.
}
\section{Function version}{
 0.1.2
}

\examples{

##construct data.frame
data <- data.frame(
 NUCLIDES = c("U-238", "Th-232", "K-40"),
 VALUE = c(40,80,100),
 VALUE_ERROR = c(4,8,10),
 stringsAsFactors = FALSE)

##perform analysis
convert_Activity2Concentration(data)

} 

\section{How to cite}{
Fuchs, M.C., 2025. convert_Activity2Concentration(): Convert Nuclide Activities to Abundance and Vice Versa. Function version 0.1.2. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.0.0. https://r-lum.github.io/Luminescence/
}

\references{
Debertin, K., Helmer, R.G., 1988. Gamma- and X-ray Spectrometry with
Semiconductor Detectors, Elsevier Science Publishers, p.283

Wiechen, A., Ruehle, H., Vogl, K., 2013. Bestimmung der massebezogenen
Aktivitaet von Radionukliden. AEQUIVAL/MASSAKT, ISSN 1865-8725,
\url{https://www.bmuv.de/fileadmin/Daten_BMU/Download_PDF/Strahlenschutz/aequival-massakt_v2013-07_bf.pdf}
}
\author{
Margret C. Fuchs, Helmholtz-Institute Freiberg for Resource Technology (Germany)
, RLum Developer Team}
\keyword{IO}
