# === OM specification using SS3 (Methot 2012) stock assessments ====================

#' Reads MLE estimates from Stock Synthesis file structure into an operating model using package r4ss.
#'
#'
#' @description A function that uses the file location of a fitted SS3 model including input files to population the
#' various slots of an operating model with MLE parameter estimates. The function mainly populates the Stock and Fleet portions
#' of the operating model; the user still needs to parameterize most of the observation and implementation portions of the operating model.
#' @param SSdir A folder with Stock Synthesis input and output files in it.
#' @param nsim The number of simulations to take for parameters with uncertainty (for OM@@cpars custom parameters).
#' @param proyears The number of projection years for MSE
#' @param reps The number of stochastic replicates within each simulation in the operating model.
#' @param maxF The maximum allowable F in the operating model.
#' @param seed The random seed for the operating model.
#' @param interval The interval at which management procedures will update the management advice in \link[DLMtool]{runMSE}, e.g., 1 = annual updates.
#' @param Obs The observation model (class Obs). This function only updates the catch and index observation error.
#' @param Imp The implementation model (class Imp). This function does not update implementation parameters.
#' @param import_mov Logical, whether to import movement matrix from the assessment.
#' @param gender An integer that indexes the sex for importing life history parameters (1 = usually female, 2 = usually male, 1:2 = mean across both sexes).
#' @param age_rec Integer for the age of recruitment. The default is 1 for DLMtool operating models. Generally, should not be changed.
#' @param silent Whether to silence messages to the console.
#' @param Name The name of the operating model
#' @param Source Reference to assessment documentation e.g. a url
#' @param Author Who did the assessment
#' @param report Logical, if TRUE, the function will run \link[DLMtool]{runMSE} to generate historical data from the operating model
#' to compare against SS output. A markdown report will be generated.
#' @param filename If \code{report = TRUE}, character string for the name of the markdown and HTML files.
#' @param dir If \code{report = TRUE}, the directory in which the markdown and HTML files will be saved.
#' @param open_file If \code{report = TRUE}, whether the HTML document is opened after it is rendered.
#' @param ... Arguments to pass to \link[r4ss]{SS_output}.
#' @note Currently supports versions of r4ss on CRAN (v.1.24) and Github (v.1.34-38).
#' @details The function generally uses values from the terminal year of the assessment for most life history parameters (maturity, M, etc). This function
#' does detect time-varying growth in the assessment and uses annual length/weight-at-age for historical years.
#' Selectivity is derived from the F-at-age matrix.
#' @return An object of class OM.
#' @author T. Carruthers and Q. Huynh
#' @export
#' @seealso \link{SS2Data}
#' @importFrom stats acf
#' @importFrom reshape2 melt
#' @importFrom dplyr summarise group_by pull left_join
SS2OM <- function(SSdir, nsim = 48, proyears = 50, reps = 1, maxF = 3, seed = 1, interval = 1,
                  Obs = DLMtool::Generic_Obs, Imp = DLMtool::Perfect_Imp,
                  import_mov = TRUE, gender = 1:2, age_rec = 1, silent = FALSE,
                  Name = "OM generated by SS2OM function", Source = "No source provided", Author = "No author provided",
                  report = FALSE, filename = "SS2OM", dir = tempdir(), open_file = TRUE, ...) {
  if(!requireNamespace("r4ss", quietly = TRUE)) {
    stop("Download the r4ss package to use this function. It is recommended to install the Github version with: devtools::install_github(\"r4ss/r4ss\")", call. = FALSE)
  }

  dots <- list(dir = SSdir, ...)
  if(!any(names(dots) == "covar")) dots$covar <- FALSE
  if(!any(names(dots) == "forecast")) dots$forecast <- FALSE
  #if(!any(names(dots) == "ncols")) dots$ncols <- 1e3
  if(!any(names(dots) == "printstats")) dots$printstats <- FALSE
  if(!any(names(dots) == "verbose")) dots$verbose <- FALSE
  if(!any(names(dots) == "warn")) dots$warn <- FALSE

  if(!silent) {
    message(paste("-- Using function SS_output of package r4ss version", packageVersion("r4ss"), "to extract data from SS file structure --"))
    message(paste("Reading directory:", SSdir))
  }
  replist <- try(do.call(r4ss::SS_output, dots), silent = TRUE)
  if(is.character(replist)) stop("r4ss::SS_output function returned an error -\n", replist, call. = FALSE)
  if(!silent) message("-- End of r4ss operations --\n")

  season_as_years <- FALSE
  if(replist$nseasons == 1 && all(replist$seasduration < 1)) {
    if(!silent) message(paste("Season-as-years detected in SS model. There is one season in the year with duration of", replist$seasduration, "year."))
    season_as_years <- TRUE
    nseas <- 1/replist$seasduration
    if(!silent) message("DLMtool operating model is an annual model. Since the SS model is seasonal, we need to aggregate over seasons.\n")
  } else {
    nseas <- replist$nseasons
    if(nseas > 1) {
      if(!silent) message("DLMtool operating model is an annual model. Since the SS model is seasonal, we need to aggregate over seasons.\n")
    }
  }

  # Create OM object
  Stock <- new("Stock")
  Fleet <- new("Fleet")
  OM <- new("OM", Stock = Stock, Fleet = Fleet, Obs = Obs, Imp = Imp)
  OM@nsim <- nsim
  OM@proyears <- proyears
  OM@Name <- Name
  OM@interval <- interval

  OM@Source <- paste0(Source, ". Author: ", Author, ".")

  mainyrs <- replist$startyr:replist$endyr
  OM@nyears <- nyears <- ceiling(length(mainyrs) / ifelse(season_as_years, nseas, 1))

  seas1_yind_full <- expand.grid(nseas = 1:nseas, true_year = 1:nyears) # Group assessment years to true years
  seas1_yind_full$assess_year <- mainyrs
  seas1_yind <- which(seas1_yind_full$nseas == 1) # Assessment years that correspond to first season of a true year

  OM@maxF <- maxF
  OM@reps <- reps
  OM@seed <- seed
  OM@CurrentYr <- ifelse(season_as_years, nyears, replist$endyr)

  # === Stock parameters =======================================================
  if(!silent && replist$nsexes == 2) {
    message("2-sex SS model found.")
    if(length(gender) == 1 && gender == 1) message("Life history parameters for gender code 1 (usually female) will be used for the OM.")
    if(length(gender) == 1 && gender == 2) message("Life history parameters for gender code 2 (usually male) will be used for the OM.")
    if(all(gender == c(1,2))) message("Life history parameters will be averaged from both genders for the OM.")
  }

  # ---- Stock-recruit relationship ----
  # R0 is unfished recruits at age = age_rec
  if(all(match(c("Gender", "Year"), names(replist$natage_annual_1_no_fishery), nomatch = 0))) { # SS 3.24

    N_at_age <- reshape2::melt(replist$natage_annual_1_no_fishery, id.vars = c("Bio_Pattern", "Gender", "Year"),
                               variable.name = "Age", value.name = "N")
    N_at_age <- N_at_age[N_at_age$Year == mainyrs[1] & N_at_age$Age == age_rec * ifelse(season_as_years, nseas, 1), ]
    N_at_age <- N_at_age[vapply(N_at_age$Gender, "%in%", logical(1), table = gender), ]

  } else { # SS 3.30

    N_at_age <- reshape2::melt(replist$natage_annual_1_no_fishery, id.vars = c("Bio_Pattern", "Sex", "Yr"),
                               variable.name = "Age", value.name = "N")
    N_at_age <- N_at_age[N_at_age$Yr == mainyrs[1] & N_at_age$Age == age_rec * ifelse(season_as_years, nseas, 1), ]
    N_at_age <- N_at_age[vapply(N_at_age$Sex, "%in%", logical(1), table = gender), ]

  }

  R0 <- sum(N_at_age$N)

  # In season-as-year model, R0 is the seasonal rate of recruitment, must adjust for annual model
  OM@R0 <- R0 * ifelse(season_as_years, nseas, 1)
  if(!silent) {
    message("R0 = ", round(R0), " (unfished abundance at age ", age_rec , ")")
    if(replist$nsexes == 2 && all(gender == c(1,2))) message("R0 is the sum of abundances of both genders.")
  }

  # Steepness is now deterministic
  if(packageVersion("r4ss") == 1.24) {
    SR_ind <- match(mainyrs, replist$recruit$year)
    SSB <- replist$recruit$spawn_bio[SR_ind]
    SSB0 <- replist$derived_quants[replist$derived_quants$LABEL == "SPB_Virgin", 2]
  } else {
    SR_ind <- match(mainyrs, replist$recruit$Yr)
    SSB <- replist$recruit$SpawnBio[SR_ind]
    SSB0 <- replist$derived_quants[replist$derived_quants$Label == "SSB_Virgin", 2]
  }

  if(replist$SRRtype == 3 || replist$SRRtype == 6) { # Beverton-Holt SR
    SR <- "BH"
    OM@SRrel <- 1L
    hs <- replist$parameters[grepl("steep", rownames(replist$parameters)), ]$Value
    if(!silent) message("Beverton-Holt stock-recruit relationship found with steepness = ", hs, ".\n")
  } else if(replist$SRRtype == 2) {
    SR <- "Ricker"
    OM@SRrel <- 2L
    hs <- replist$parameters[grepl("SR_Ricker", rownames(replist$parameters)), ]$Value
    if(!silent) message("Ricker stock-recruit relationship found with steepness = ", steep$Value, ".\n")
  } else if(replist$SRRtype == 7) {
    SR <- "BH"
    OM@SRrel <- 1L

    s_frac <- replist$parameters$Value[replist$parameters$Label == "SR_surv_Sfrac"]
    Beta <- replist$parameters$Value[replist$parameters$Label == "SR_surv_Beta"]

    s0 <- 1/SpR0
    z0 <- -log(s0)
    z_min <- z0 * (1 - s_frac)

    hs <- 0.2 * exp(z0 * s_frac * (1 - 0.2 ^ Beta))

    if(!silent) {
      message("Survival-based stock-recruit relationship was detected with steepness = ", round(hs, 2), ".")
      message("As an approximation, a Beverton-Holt relationship is used with the same steepness value.")
    }

  } else {
    SR <- OM@SRrel <- 1L

    rec <- replist$recruit$pred_recr[SR_ind] # recruits to age 0
    SpR0 <- SSB0/(R0 * ifelse(season_as_years, nseas, 1))

    hs <- SRopt(1, SSB, rec, SpR0, plot = FALSE, type = ifelse(SR == 1, "BH", "Ricker"))

    if(!silent) {
      message("From r4ss, SRRtype = ", replist$SRRtype)
      message("Steepness value not found. By default, estimating Beverton-Holt steepness from R and SSB estimates.\n")
      message("Steepness = ", hs, ".\n")
    }
  }
  OM@h <- rep(hs, 2)

  # ---- Max-age ----
  growdat <- getGpars(replist)      # Age-specific parameters in endyr
  growdat <- do.call(rbind, growdat)
  if("int_Age" %in% names(growdat)) {
    ages <- unique(growdat$int_Age)
  } else {
    ages <- unique(growdat$Age)
  }

  if(replist$nsexes == 1) {
    growdat$Len_Mat[growdat$Len_Mat < 0] <- 1
    growdat$Age_Mat[growdat$Age_Mat < 0] <- 1
  }

  maxage <- floor(max(ages)/ifelse(season_as_years, nseas, 1))

  seas1_aind_full <- expand.grid(nseas = 1:nseas, true_age = 0:maxage)[1:length(ages), ] # Group assessment ages to true ages
  seas1_aind_full$assess_age <- ages
  seas1_aind <- which(seas1_aind_full$nseas == 1 & seas1_aind_full$true_age >= age_rec) # Age indices that correspond to season 1

  OM@maxage <- maxage
  OM@cpars$plusgroup <- rep(1L, OM@nsim)
  if(!silent) message("Max age is ", maxage, ".")

  # ---- Growth ----
  if("int_Age" %in% names(growdat)) { # SS 3.30?

    if(season_as_years) {
      growdat <- growdat[vapply(growdat$int_Age, "%in%", logical(1), ages[seas1_aind]), ]
    } else growdat <- growdat[growdat$int_Age >= age_rec, ]
    growdat <- growdat[vapply(growdat$Sex, "%in%", logical(1), gender), ]

    Len_age_terminal <- summarise(group_by(growdat, int_Age), LAA = mean(Len_Beg)) %>% pull(2)
    Wt_age_terminal <- summarise(group_by(growdat, int_Age), WAA = mean(Wt_Beg)) %>% pull(2)
    Mat_age_terminal <- summarise(group_by(growdat, int_Age), MAA = mean(Len_Mat[Len_Mat >= 0] * Age_Mat[Age_Mat >= 0])) %>% pull(2)

  } else {

    if(season_as_years) {
      growdat <- growdat[vapply(growdat$Age, "%in%", logical(1), ages[seas1_aind]), ]
    } else growdat <- growdat[growdat$Age >= age_rec, ]
    growdat <- growdat[vapply(growdat$Sex, "%in%", logical(1), gender), ]

    Len_age_terminal <- unlist(summarise(group_by(growdat, Age), LAA = mean(Len_Beg))[, 2])
    Wt_age_terminal <- unlist(summarise(group_by(growdat, Age), WAA = mean(Wt_Beg))[, 2])
    Mat_age_terminal <- unlist(summarise(group_by(growdat, Age), MAA = mean(Len_Mat[Len_Mat >= 0] * Age_Mat[Age_Mat >= 0]))[, 2])
  }


  if(!is.null(replist$growthvaries) && replist$growthvaries) { # For all years except terminal year?
    if(!silent) {
      message("Time-varying growth found.")
      message("Projection period growth assumed to be the same as that in terminal year.")
    }
    if(season_as_years) stop("Can't deal with season_as_years yet when growth is time-varying.")

    Len_age <- reshape2::melt(replist$growthseries, id.vars = c("Morph", "Yr", "Seas", "SubSeas"), variable.name = "Age",
                              value.name = "LAA")
    Len_age <- Len_age[vapply(Len_age$Yr, "%in%", logical(1), table = mainyrs), ] # Subset by year
    Len_age <- Len_age[as.numeric(Len_age$Age)-1 >= age_rec, ] # Subset by age >= age_rec
    Len_age <- Len_age[Len_age$Seas == 1 & Len_age$SubSeas == 1, ] # Subset by season 1

    Len_age <- Len_age[vapply(Len_age$Morph, "%in%", logical(1), table = gender), ] # Subset by gender

    Len_age <- summarise(group_by(Len_age, Yr, Age), LAA = mean(LAA))
    Len_age_timevarying <- reshape2::acast(Len_age, list("Age", "Yr"), value.var = "LAA")

    Len_age <- as.matrix(cbind(Len_age_timevarying, Len_age_terminal))
    Len_age2 <- array(NA, dim = c(maxage, nyears+proyears, nsim))
    Len_age2[, 1:nyears, ] <- Len_age
    Len_age2[, nyears + 1:proyears, ] <- Len_age[, nyears]

    OM@cpars$Len_age <- aperm(Len_age2, c(3, 1, 2))

  } else {
    Len_age2 <- array(NA, dim = c(maxage, nsim, nyears+proyears))
    Len_age2[, , 1:nyears] <- Len_age_terminal
    Len_age2[, , nyears + 1:proyears] <- Len_age2[, , nyears]

    OM@cpars$Len_age <- aperm(Len_age2, c(2, 1, 3))

  }
  if(!silent) message("Length-at-age found.")

  #if(replist$wtatage_switch) {
  #  stop("Found empirical weights at age")
  #  Wt_age_emp <- melt(replist$wtatage, id.vars = c("Yr", "Seas", "Sex", "Bio_Pattern", "BirthSeas", "Fleet"),
  #                     variable.name = "Age", value.name = "WAA")
  #  Wt_age_emp$Yr <- abs(Wt_age_emp$Yr)
  #}
  GP <- replist$Growth_Parameters   # Some growth parameters (presumably in endyr)
  if (!is.null(GP$Platoon)) {
    GP <- GP[GP$Platoon == 1, ]
  }
  muLinf <- mean(GP$Linf[gender], na.rm = TRUE)
  cvLinf <- mean(GP$CVmax[gender], na.rm = TRUE)



  if(cvLinf > 1) cvLinf <- cvLinf/muLinf
  OM@LenCV <- rep(cvLinf, 2)

  OM@Ksd <- OM@Kgrad <- OM@Linfsd <- OM@Linfgrad <- c(0, 0)
  OM@K <- OM@Linf <- OM@t0 <- c(0, 0) # not used - vB pars estimated from Len_age internally

  # Weight at age
  OM@a <- mean(replist$Growth_Parameters$WtLen1[gender], na.rm = TRUE)
  OM@b <- mean(replist$Growth_Parameters$WtLen2[gender], na.rm = TRUE)


  if(exists("Len_age_timevarying")) { # This is a better solution for weight at age when growth is time varying
    OM@cpars$Wt_age <- OM@a * OM@cpars$Len_age ^ OM@b
  } else {
    Wt_age2 <- array(NA, dim = c(maxage, nsim, nyears + proyears))
    Wt_age2[, , 1:nyears] <- Wt_age_terminal
    Wt_age2[, , nyears + 1:proyears] <- Wt_age2[, , nyears]
    OM@cpars$Wt_age <- aperm(Wt_age2, c(2, 1, 3)) # dims = nsim, max_age, nyears+proyears
  }
  if(!silent) message("Weight-at-age found.")

  # ---- Maturity ----
  Mat_age <- array(NA, dim = c(maxage, nsim, nyears+proyears))
  Mat_age[, , 1:nyears] <- Mat_age_terminal/max(Mat_age_terminal)
  Mat_age[, , nyears + 1:proyears] <- Mat_age[, , nyears]
  OM@cpars$Mat_age <- aperm(Mat_age, c(2, 1, 3))  # dims = nsim, max_age, nyears+proyears
  if(!silent) message("Maturity-at-age found.")

  OM@L50 <- OM@L50_95 <- c(0, 0) # calculated internally

  # ---- M-at-Age ----
  if(all(match(c("Gender", "Year"), names(replist$M_at_age), nomatch = 0))) { # SS 3.24

    M_at_age <- reshape2::melt(replist$M_at_age, id.vars = c("Bio_Pattern", "Gender", "Year"), variable.name = "Age",
                               value.name = "M")
    M_at_age <- M_at_age[as.numeric(M_at_age$Age)-1 >= age_rec * ifelse(season_as_years, nseas, 1), ]  # Subset by age >= age_rec
    M_at_age <- M_at_age[vapply(M_at_age$Year, "%in%", logical(1), mainyrs), ]  # Subset by year
    M_at_age <- M_at_age[vapply(M_at_age$Gender, "%in%", logical(1), table = gender), ] # Subset by gender

    if(season_as_years) { # Mean across sub-ages, then sum across seasons
      M_at_age$true_Age <- seas1_aind_full$true_age[match(M_at_age$Age, seas1_aind_full$assess_age)]
      M_at_age <- summarise(group_by(M_at_age, Gender, Year, true_Age), M = mean(M))

      M_at_age$true_Year <- seas1_yind_full$true_year[match(M_at_age$Year, seas1_yind_full$assess_year)]
      M_at_age <- summarise(group_by(M_at_age, Gender, true_Year, true_Age), M = sum(M))

      M_ageArray <- reshape2::acast(M_at_age, list("true_Age", "true_Year"), value.var = "M")
    } else {
      M_at_age <- summarise(group_by(M_at_age, Year, Age), M = mean(M))
      M_ageArray <- reshape2::acast(M_at_age, list("Age", "Year"), value.var = "M")
    }


  } else { # SS 3.30
    M_at_age <- reshape2::melt(replist$M_at_age, id.vars = c("Bio_Pattern", "Sex", "Yr"), variable.name = "Age", value.name = "M")
    M_at_age <- M_at_age[vapply(M_at_age$Yr, "%in%", logical(1), table = mainyrs), ] # Subset by year
    M_at_age <- M_at_age[vapply(M_at_age$Sex, "%in%", logical(1), table = gender), ] # Subset by gender

    M_at_age <- M_at_age[as.numeric(M_at_age$Age)-1 >= age_rec * ifelse(season_as_years, nseas, 1), ]  # Subset by age >= age_rec

    M_at_age <- summarise(group_by(M_at_age, Yr, Age), M = mean(M))
    M_ageArray <- reshape2::acast(M_at_age, list("Age", "Yr"), value.var = "M")
  }

  if(ncol(M_ageArray) < nyears) M_ageArray <- cbind(M_ageArray, M_ageArray[, ncol(M_ageArray)])
  if(all(is.na(M_ageArray[maxage, ]))) M_ageArray[maxage, ] <- M_ageArray[maxage - 1, ]

  projM <- matrix(M_ageArray[, nyears], nrow = maxage, ncol = OM@proyears)
  M_ageArray <- cbind(M_ageArray, projM)
  M_ageArray <- replicate(OM@nsim, M_ageArray)
  OM@cpars$M_ageArray <- aperm(M_ageArray, c(3, 1, 2))

  OM@M <- M_ageArray[, nyears, 1]
  OM@M2 <- OM@M + 1e-6
  OM@Mexp <- OM@Msd <- c(0, 0)  # Time varying M would be in cpars
  if(!silent) message("Natural mortality found.")

  # ---- Depletion ----
  InitF <- replist$parameters$Value[grepl("InitF", replist$parameters$Label)]
  R_offset <- replist$parameters$Value[grepl("SR_R1_offset", replist$parameters$Label)]

  if(any(InitF > 0, na.rm = TRUE) || any(R_offset != 0, na.rm = TRUE)) {
    initD <- SSB[1]/SSB0
    OM@cpars$initD <- rep(initD, nsim)
    if(!silent) message("Initial depletion: OM@cpars$initD = ", round(initD, 2))
  }

  OM@D <- rep(replist$current_depletion, 2)
  if(!silent) message("Current depletion: OM@D = ", round(replist$current_depletion, 2), "\n")

  # ---- Recruitment deviations ----
  if(season_as_years) {
    replist$recruit$true_Yr <- seas1_yind_full$true_year[match(replist$recruit$Yr, seas1_yind_full$assess_year)]
    recruit <- summarise(group_by(replist$recruit, true_Yr), exp_recr = sum(exp_recr), pred_recr = sum(pred_recr)) # Need to sum over season_as_years
    hist_dev <- c(rep(1, maxage - 1), recruit$pred_recr[!is.na(recruit$true_Yr)]/recruit$exp_recr[!is.na(recruit$true_Yr)])
    dev_for_AC <- log(hist_dev)

  } else {

    year_first_rec_dev <- mainyrs[1] - maxage
    rec_ind <- match(year_first_rec_dev:(max(mainyrs) - age_rec * ifelse(season_as_years, nseas, 1)), replist$recruit$Yr)
    hist_dev <- replist$recruit$pred_recr[rec_ind]/replist$recruit$exp_recr[rec_ind] # In normal space
    hist_dev[is.na(hist_dev)] <- 1
    dev_for_AC <- replist$recruit$dev[rec_ind] # In log-space

  }

  dev_for_AC <- dev_for_AC[!is.na(dev_for_AC)]
  if(all(dev_for_AC == 0)) {
    if(!silent) message("Note: no recruitment deviates appear to be estimated.")
    AC <- 0
  } else {
    AC <- acf(dev_for_AC[dev_for_AC != 0], plot = FALSE)$acf[2, 1, 1]
  }
  if(is.na(AC)) AC <- 0
  if(!silent) message("Recruitment autocorrelation for projection period is estimated from historical recruitment deviations. OM@AC = ", round(AC, 3), ".")

  procsd <- replist$sigma_R_in
  procmu <- -0.5 * procsd^2 * (1 - AC/sqrt(1 - AC^2)) # adjusted log normal mean with AC
  Perr_hist <- matrix(hist_dev, nrow = nsim, ncol = maxage + nyears - 1, byrow = TRUE) # Historical recruitment is deterministic
  Perr_proj <- matrix(rnorm(proyears * nsim, rep(procmu, each = nsim),
                            rep(procsd, each = nsim)), nrow = nsim, ncol = proyears) # Sample recruitment for projection

  if(AC != 0) { # Add autocorrelation to projection recruitment
    Perr_proj[, 1] <- AC * Perr_hist[, ncol(Perr_hist)] + Perr_proj[, 1] * sqrt(1 - AC^2)
    for(y in 2:ncol(Perr_proj)) Perr_proj[, y] <- AC * Perr_proj[, y-1] + Perr_proj[, y] * sqrt(1 - AC^2)
  }

  OM@cpars$Perr_y <- cbind(Perr_hist, exp(Perr_proj))
  OM@Perr <- rep(procsd, 2) # Point estimate from assessment MLE
  OM@AC <- rep(AC, 2)
  if(!silent) message("Recruitment deviates found and future deviates sampled.")

  # ---- Movement modelling ----
  OM@Frac_area_1 <- OM@Size_area_1 <- OM@Prob_staying <- rep(0.5, 2)
  if(import_mov && nrow(replist$movement) > 0) {
    movement <- replist$movement[replist$movement$Seas == 1 & replist$movement$Gpattern == 1, ]
    if(nrow(movement) == 0) movement <- replist$movement[replist$movement$Seas == 1 & replist$movement$GP == 1, ]

    nareas <- length(unique(movement$Source_area))
    if(!silent) message(nareas, " area model found. Parameterizing movement matrix.")

    full_movement <- movement[, grepl("age", names(movement)) & names(movement) != "minage" & names(movement) != "maxage"]

    nages <- ncol(full_movement)
    mov <- array(NA, c(nsim, nages, nareas, nareas))

    for(i in 1:nrow(full_movement)) {
      from <- movement$Source_area[i]
      to <- movement$Dest_area[i]

      for(j in 1:ncol(full_movement)) mov[1:nsim, j, from, to] <- full_movement[i, j]
    }
    mov[is.na(mov)] <- 0

    if(season_as_years) mov <- mov[, seas1_aind, , , drop = FALSE]

    OM@cpars$mov <- mov
  }

  # Fleet parameters ===========================================================
  # ---- Vulnerability ----
  if(all(match(c("Gender", "Year"), names(replist$Z_at_age), nomatch = 0))) { # SS 3.24

    Z_at_age <- reshape2::melt(replist$Z_at_age, id.vars = c("Bio_Pattern", "Gender", "Year"), variable.name = "Age", value.name = "Z")
    Z_at_age <- Z_at_age[as.numeric(Z_at_age$Age)-1 >= age_rec * ifelse(season_as_years, nseas, 1), ]  # Subset by age >= age_rec
    Z_at_age <- Z_at_age[vapply(Z_at_age$Year, "%in%", logical(1), table = mainyrs), ]  # Subset by year
    Z_at_age <- Z_at_age[vapply(Z_at_age$Gender, "%in%", logical(1), table = gender), ] # Subset by gender

    if(season_as_years) { # Mean across sub-ages, then sum across seasons
      Z_at_age$true_Age <- seas1_aind_full$true_age[match(Z_at_age$Age, seas1_aind_full$assess_age)]
      Z_at_age <- summarise(group_by(Z_at_age, Gender, Year, true_Age), Z = mean(Z))

      Z_at_age$true_Year <- seas1_yind_full$true_year[match(Z_at_age$Year, seas1_yind_full$assess_year)]
      Z_at_age <- summarise(group_by(Z_at_age, Gender, true_Year, true_Age), Z = sum(Z))

      Z_ageArray <- reshape2::acast(Z_at_age, list("true_Age", "true_Year"), value.var = "Z")
    } else {
      Z_at_age <- summarise(group_by(Z_at_age, Year, Age), Z = mean(Z))
      Z_ageArray <- reshape2::acast(Z_at_age, list("Age", "Year"), value.var = "Z")
    }

  } else { # SS 3.30
    Z_at_age <- reshape2::melt(replist$Z_at_age, id.vars = c("Bio_Pattern", "Sex", "Yr"), variable.name = "Age", value.name = "Z")
    Z_at_age <- Z_at_age[vapply(Z_at_age$Yr, "%in%", logical(1), table = mainyrs), ] # Subset by year
    Z_at_age <- Z_at_age[as.numeric(Z_at_age$Age)-1 >= age_rec, ]                    # Subset by age >= age_rec
    Z_at_age <- Z_at_age[vapply(Z_at_age$Sex, "%in%", logical(1), table = gender), ] # Subset by gender

    Z_at_age <- summarise(group_by(Z_at_age, Yr, Age), Z = mean(Z))
    Z_ageArray <- reshape2::acast(Z_at_age, list("Age", "Yr"), value.var = "Z")

  }

  if(ncol(Z_ageArray) < nyears) Z_ageArray <- cbind(Z_ageArray, Z_ageArray[, ncol(Z_ageArray)])

  F_ageArray <- Z_ageArray - M_ageArray[, 1:nyears, 1]
  if(all(is.na(F_ageArray[maxage, ]))) F_ageArray[maxage, ] <- F_ageArray[maxage - 1, ]
  F_ageArray[F_ageArray < 1e-8] <- 1e-8

  F_norm <- apply(F_ageArray, 2, function(x) x/max(x))

  V <- array(NA, dim = c(maxage, nyears + proyears, nsim))
  V[, 1:nyears, ] <- array(F_norm, dim = c(maxage, nyears, nsim))
  V[, nyears + 1:proyears, ] <- V[, nyears, ]

  Find <- apply(F_ageArray, 2, max, na.rm = TRUE) # get apical F

  OM@cpars$V <- aperm(V, c(3, 1, 2))
  if(!silent) message("Selectivity found.")

  OM@L5 <- OM@LFS <- OM@Vmaxlen <- c(0,0) # calculated internally
  OM@isRel <- "FALSE" # these are real lengths not relative to length at 50% maturity

  # ---- Fishing mortality rate index ----
  OM@cpars$Find <- matrix(Find, nsim, nyears, byrow = TRUE) # is only historical years
  if(!silent) message("Historical F found.")

  OM@Spat_targ <- rep(1, 2)

  if(season_as_years) OM@EffYears <- 1:nyears else OM@EffYears <- mainyrs

  OM@EffLower <- OM@EffUpper <- OM@cpars$Find[1, 1:nyears]
  OM@Esd <- OM@qcv <- c(0, 0)
  OM@qinc <- c(0, 0)

  OM@Period <- OM@Amplitude <- rep(NaN, 2)


  # Observation model parameters ==============================================================================
  CSD <- replist$catch_error
  if(all(is.na(CSD)) && packageVersion("r4ss") == 1.35) CSD <- replist$catch_se
  if(!all(is.na(CSD))) {
    OM@Cobs <- range(sqrt(exp(CSD[CSD > 0]^2) - 1), na.rm = TRUE)
    if(!silent) {
      message("\nRange of error in catch (OM@Cobs) based on range of catch standard deviations: ", paste(CSD, collapse = " "))
    }
  }

  # Index observations -------------------------------------------------------
  Ind <- SS2Data_get_index(replist, mainyrs, season_as_years, nseas, index_season = "mean")

  if(is.null(Ind)) {
    message("No indices found.")
    if(packageVersion("DLMtool") >= 5.4) {
      Data@AddInd <- Data@CV_AddInd <- Data@AddIndV <- array(NA, c(1, 1, 1))
    }
  } else {
    Data <- new("Data")
    message(length(Ind$Iname), " indices of abundance found:")
    message(paste(Ind$Iname, collapse = "\n"))

    if(packageVersion("DLMtool") >= "5.4.4") {

      Data@AddInd <- Ind$AddInd
      Data@CV_AddInd <- sqrt(exp(Ind$SE_AddInd^2) - 1)

      Data@AddIunits <- Ind$AddIunits
      Data@AddIndType <- Ind$AddIndType

      if(season_as_years) {
        AddIndV <- apply(Ind$AddIndV, 1, function(x) {
          xx <- data.frame(assess_age = as.numeric(names(x)), sel = x) %>% left_join(seas1_aind_full[, -1], by = "assess_age")
          xx_agg <- aggregate(xx$sel, by = list(age = xx$true_age), mean, na.rm = TRUE)
          xx_agg$x[xx_agg$age >= age_rec]
        }) %>% t()
      } else {
        AddIndV <- Ind$AddIndV
      }

      Data@AddIndV <- array(AddIndV, c(1, dim(AddIndV)))

      OM@cpars$Data <- Data

      message("Updated Data@AddInd, Data@CV_AddInd, Data@AddIndV.")
    } else {
      message("\n\n *** Update DLMtool to latest version (5.4.4+) in order to add indices to OM (via OM@cpars$Data). *** \n\n")
    }
  }

  if(report) {
    if(!silent) message("\nRunning historical simulations to compare SS output and OM conditioning...\n")
    Hist <- runMSE(OM, Hist = TRUE)

    rmd_file <- file.path(system.file(package = "MSEtool"), "rmarkdown_templates", "SS2OM.Rmd")
    rmd <- readLines(rmd_file)

    write(rmd, file = file.path(dir, paste0(filename, ".rmd")))

    if(!silent) message("Rendering markdown file to HTML: ", file.path(dir, paste0(filename, ".html")))

    rmarkdown::render(file.path(dir, paste0(filename, ".rmd")), "html_document", paste0(filename, ".html"), dir,
                      output_options = list(df_print = "paged"), quiet = TRUE)
    message("Rendering complete.")

    if(open_file) browseURL(file.path(dir, paste0(filename, ".html")))
  }

  return(OM)
}


# #' A function that samples multivariate normal (logspace) variables
# #'
# #' @param xmu The mean (normal space) of the first (x) variable
# #' @param ymu The mean (normal space) of the second (y) variable
# #' @param xcv The coefficient of variation (normal space, log normal sd) of the x variable
# #' @param nsim The number of random draws
# #' @param cor The off-diagonal (symmetrical) correlation among x and y
# #' @param ploty Whether a plot of the sampled variables should be produced
# #' @author T. Carruthers
# #' @export negcorlogspace
#' @importFrom mvtnorm rmvnorm
negcorlogspace<-function(xmu,ymu,xcv=0.1,nsim,cor=-0.9,ploty=FALSE){

  varcov=matrix(c(1,cor,cor,1),nrow=2)
  out<-mvtnorm::rmvnorm(nsim,c(0,0),sigma=varcov)
  out<-out/rep(apply(out,2,sd)/xcv,each=nsim)
  out<-exp(out)
  out<-out/rep(apply(out,2,mean),each=nsim)
  out[,1]<-out[,1]*xmu
  out[,2]<-out[,2]*ymu
  if(ploty)plot(out[,1],out[,2])
  out

}

# #' Simplified a multi-area transition matrix into the best 2 x 2 representation
# #'
# #' @description A Function that takes a larger movement matrix, identifies the most parsimonious representation of 2 non-mixed areas and returns the final unfished movement matrix
# #' @param movtab a table of estimated movements
# #' @author T. Carruthers
# #' @export
# movdistil<-function(movtab){
#
#   nareas<-max(movtab$Source_area,movtab$Dest_area)
#   mov<-array(0,c(nareas,nareas))
#   mov[cbind(movtab$Source_area,movtab$Dest_area)]<-movtab[,ncol(movtab)]
#
#   vec<-rep(1/nareas,nareas)
#   for(i in 1:100)vec<-vec%*%mov
#   endmov<-array(vec,c(nareas,nareas))*mov
#
#   listy<-new('list')
#   for(i in 1:nareas)listy[[i]]<-c(1,2)
#
#   combins<-expand.grid(listy)
#   combins<-combins[!apply(combins,1,sum)%in%c(nareas*1,nareas*2),]
#   nc<-nrow(combins)/2
#   combins<-combins[(nc+1):nrow(combins),]
#
#   base<-cbind(expand.grid(1:nareas,1:nareas),as.vector(endmov))
#
#   emt<-NULL
#   out<-rep(NA,nc)
#
#   for(i in 1:nc){
#
#     vec<-combins[i,]
#     vec<-c(-1,1)[as.numeric(vec)]
#     out[i]<-sum((vec-(vec%*%mov))^2)
#
#   }
#
#   best<-as.numeric(combins[which.min(out),])
#
#   aggdat<-cbind(expand.grid(best,best),as.vector(endmov))
#   agg<-aggregate(aggdat[,3],by=list(aggdat[,1],aggdat[,2]),sum)
#   newmov<-array(NA,c(2,2))
#   newmov[as.matrix(agg[,1:2])]<-agg[,3]
#   newmov/apply(newmov,1,sum)
#
# }

# #' Predict recruitment and return fit to S-R observations
# #'
# #' @description Internal function to \link{optSR}
# #' @param pars an initial guess at model parameters steepness and R0
# #' @param SSB 'observations' of spawning biomass
# #' @param rec 'observations' (model predictions) of recruitment
# #' @param SSBpR spawning stock biomass per recruit at unfished conditions
# #' @param mode should fit (= 1) or recruitment deviations (not 1) be returned
# #' @param plot should a plot of the model fit be produced?#'
# #' @param type what type of stock recruitment curve is being fitted ("BH" = Beverton-Holt or "Ricker")
# #' @author T. Carruthers
# #' @export
getSR <- function(pars, SSB, rec, SSBpR, mode = 1, plot = FALSE, type = c("BH", "Ricker")){
  R0 <- exp(pars[2])
  if(type == "BH") {
    h <- 0.2 + 0.8 * ilogit(pars[1])
    recpred<-((0.8*R0*h*SSB)/(0.2*SSBpR*R0*(1-h)+(h-0.2)*SSB))
  }
  if(type == "Ricker") {
    h <- 0.2 + exp(pars[1])
    recpred <- SSB * (1/SSBpR) * (5*h)^(1.25*(1 - SSB/(R0*SSBpR)))
  }

  if(plot){
    ord <- order(SSB)
    plot(SSB[ord], rec[ord], ylim=c(0, max(rec, R0)), xlim=c(0, max(SSB, R0*SSBpR)), xlab="", ylab="")
    SSB2 <- seq(0, R0*SSBpR, length.out=500)
    if(type == "BH") recpred2 <- ((0.8*R0*h*SSB2)/(0.2*SSBpR*R0*(1-h)+(h-0.2)*SSB2))
    if(type == "Ricker") recpred2 <- SSB2 * (1/SSBpR) * (5*h)^(1.25*(1 - SSB2/(R0*SSBpR)))
    lines(SSB2, recpred2, col='blue')
    abline(v=c(0.2*R0*SSBpR, R0*SSBpR), lty=2, col='red')
    abline(h=c(R0, R0*h), lty=2, col='red')
    legend('topright', legend=c(paste0("h = ", round(h,3)), paste0("ln(R0) = ", round(log(R0),3))), bty='n')
  }

  if(mode==1){
    #return(sum(((recpred-rec)/10000)^2))
    sigmaR <- sqrt(sum((log(rec/recpred))^2)/length(recpred))
    return(-sum(dnorm(log(rec)-log(recpred),0,sigmaR,log=T)))
    #-dnorm(pars[1],0,6,log=T)) # add a vague prior on h = 0.6
    #return(-sum(dnorm(recpred,rec,rec*0.5,log=T)))
  }else{
    return(rec-recpred)
  }
}

# #' Wrapper for estimating stock recruitment parameters from resampled stock-recruitment data
# #'
# #' @param x position to accommodate lapply-type functions
# #' @param SSB 'observations' of spawning biomass
# #' @param rec 'observations' (model predictions) of recruitment
# #' @param SSBpR spawning stock biomass per recruit at unfished conditions
# #' @param pars an initial guess at model parameters steepness and R0
# #' @param frac the fraction of observations for resampling
# #' @param plot should a plot of model fit be produced?
# #' @param type what type of stock recruitment curve is being fitted ("BH" = Beverton-Holt or "Ricker")
# #' @return Estimated value of steepness.
# #' @author T. Carruthers
# #' @export
optSR<-function(x, SSB, rec, SSBpR, pars, frac = 0.5, plot = FALSE, type = c("BH", "Ricker")) {
  type <- match.arg(type)
  samp <- sample(1:length(SSB), size = ceiling(length(SSB) * frac), replace = FALSE)
  opt <- optim(pars, getSR, method = "BFGS", #lower = c(-6, pars[2]/50), upper = c(6, pars[2] * 50),
               SSB = SSB[samp], rec = rec[samp], SSBpR = SSBpR, mode = 1, plot = FALSE, type = type)
  if(plot) getSR(opt$par, SSB, rec, SSBpR, mode = 2, plot = plot, type = type)
  if(type == "BH") h <- 0.2 + 0.8 * ilogit(opt$par[1])
  if(type == "Ricker") h <- 0.2 + exp(opt$par[1])
  return(h)
}

# #' Function that returns a stochastic estimate of steepness given observed stock recruitment data
# #'
# #' @param nsim number of samples of steepness to generate
# #' @param SSB 'observations' of spawning biomass
# #' @param rec 'observations' (model predictions) of recruitment
# #' @param SSBpR spawning stock biomass per recruit at unfished conditions
# #' @param plot should plots of model fit be produced?
# #' @param type what type of stock recruitment curve is being fitted ("BH" = Beverton-Holt or "Ricker")
# #' @return Vector of length nsim with steepness values.
# #' @author T. Carruthers
# #' @export
SRopt <- function(nsim, SSB, rec, SSBpR, plot = FALSE, type = c("BH", "Ricker")) {
  type <- match.arg(type)
  R0temp <- rec[1] # have a guess at R0 for initializing nlm
  pars <- c(0, log(R0temp))
  #SSBpR=SSB[1]/rec[1]
  vapply(1:nsim, optSR, numeric(1), SSB = SSB, rec = rec, SSBpR = SSBpR, pars = pars, frac = 0.8,
         plot = plot, type = type)
}


# #' Extracts growth parameters from a SS3 r4ss replist
# #'
# #' @param replist the list output of the r4ss SS_output function (a list of assessment inputs / outputs)
# #' @param seas The reference season for the growth (not actually sure what this does yet)
# #' @author T. Carruthers
# #' @export getGpars
getGpars <- function(replist, seas = 1) { # This is a rip-off of SSPlotBiology

  if(packageVersion("r4ss") == 1.24) {
    res <- getGpars_r4ss_124(replist, seas)
  } else res <- getGpars_r4ss_134(replist, seas)
  #if(nrow(res) == 0) warning("No growth parameters were retrieved from r4ss output.")
  return(res)
}

getGpars_r4ss_124 <- function(replist, seas = 1) {

  nseasons <- replist$nseasons
  growdat <- replist$endgrowth[replist$endgrowth$Seas == seas, ]
  growdat$CV_Beg <- growdat$SD_Beg/growdat$Len_Beg
  growthCVtype <- replist$growthCVtype
  biology <- replist$biology
  startyr <- replist$startyr
  FecType <- replist$FecType
  FecPar1name <- replist$FecPar1name
  FecPar2name <- replist$FecPar2name
  FecPar1 <- replist$FecPar1
  FecPar2 <- replist$FecPar2
  parameters <- replist$parameters
  nsexes <- replist$nsexes
  mainmorphs <- replist$mainmorphs
  accuage <- replist$accuage
  startyr <- replist$startyr
  endyr <- replist$endyr
  growthvaries <- replist$growthvaries
  growthseries <- replist$growthseries
  ageselex <- replist$ageselex
  MGparmAdj <- replist$MGparmAdj
  wtatage <- replist$wtatage
  Growth_Parameters <- replist$Growth_Parameters
  Grow_std <- replist$derived_quants[grep("Grow_std_", replist$derived_quants$LABEL), ]
  if (nrow(Grow_std) == 0) {
    Grow_std <- NULL
  }  else {
    Grow_std$pattern <- NA
    Grow_std$sex_char <- NA
    Grow_std$sex <- NA
    Grow_std$age <- NA
    for (irow in 1:nrow(Grow_std)) {
      tmp <- strsplit(Grow_std$LABEL[irow], split = "_")[[1]]
      Grow_std$pattern[irow] <- as.numeric(tmp[3])
      Grow_std$sex_char[irow] <- tmp[4]
      Grow_std$age[irow] <- as.numeric(tmp[6])
    }
    Grow_std$sex[Grow_std$sex_char == "Fem"] <- 1
    Grow_std$sex[Grow_std$sex_char == "Mal"] <- 2
  }
  if (!is.null(replist$wtatage_switch)) {
    wtatage_switch <- replist$wtatage_switch
  } else{ stop("SSplotBiology function doesn't match SS_output function. Update one or both functions.")
  }
  if (wtatage_switch)
    cat("Note: this model uses the empirical weight-at-age input.\n",
        "     Therefore many of the parametric biology quantities which are plotted\n",
        "     are not used in the model.\n")
  if (!seas %in% 1:nseasons)
    stop("'seas' input should be within 1:nseasons")

  if (length(mainmorphs) > nsexes) {
    cat("!Error with morph indexing in SSplotBiology function.\n",
        " Code is not set up to handle multiple growth patterns or birth seasons.\n")
  }
  #if (FecType == 1) {
  #  fec_ylab <- "Eggs per kg"
  #  FecX <- biology$Wt_len_F
  #  FecY <- FecPar1 + FecPar2 * FecX
  #}

  growdatF <- growdat[growdat$Gender == 1 & growdat$Morph ==
                        mainmorphs[1], ]
  growdatF$Sd_Size <- growdatF$SD_Beg

  if (growthCVtype == "logSD=f(A)") {
    growdatF$high <- qlnorm(0.975, meanlog = log(growdatF$Len_Beg),
                            sdlog = growdatF$Sd_Size)
    growdatF$low <- qlnorm(0.025, meanlog = log(growdatF$Len_Beg),
                           sdlog = growdatF$Sd_Size)
  }  else {
    growdatF$high <- qnorm(0.975, mean = growdatF$Len_Beg,
                           sd = growdatF$Sd_Size)
    growdatF$low <- qnorm(0.025, mean = growdatF$Len_Beg,
                          sd = growdatF$Sd_Size)
  }
  if (nsexes > 1) {
    growdatM <- growdat[growdat$Gender == 2 & growdat$Morph ==
                          mainmorphs[2], ]
    xm <- growdatM$Age_Beg
    growdatM$Sd_Size <- growdatM$SD_Beg
    if (growthCVtype == "logSD=f(A)") {
      growdatM$high <- qlnorm(0.975, meanlog = log(growdatM$Len_Beg),
                              sdlog = growdatM$Sd_Size)
      growdatM$low <- qlnorm(0.025, meanlog = log(growdatM$Len_Beg),
                             sdlog = growdatM$Sd_Size)
    }    else {
      growdatM$high <- qnorm(0.975, mean = growdatM$Len_Beg,
                             sd = growdatM$Sd_Size)
      growdatM$low <- qnorm(0.025, mean = growdatM$Len_Beg,
                            sd = growdatM$Sd_Size)
    }
  } else growdatM <- NULL

  list(Female = growdatF, Male = growdatM)

}

getGpars_r4ss_134 <- function(replist, seas = 1) {
  nseasons <- replist$nseasons
  growdat <- replist$endgrowth[replist$endgrowth$Seas == seas,
                               ]
  growdat$CV_Beg <- growdat$SD_Beg/growdat$Len_Beg
  growthCVtype <- replist$growthCVtype
  biology <- replist$biology
  startyr <- replist$startyr
  FecType <- replist$FecType
  FecPar1name <- replist$FecPar1name
  FecPar2name <- replist$FecPar2name
  FecPar1 <- replist$FecPar1
  FecPar2 <- replist$FecPar2
  parameters <- replist$parameters
  nsexes <- replist$nsexes
  accuage <- replist$accuage
  startyr <- replist$startyr
  endyr <- replist$endyr
  growthvaries <- replist$growthvaries
  growthseries <- replist$growthseries
  ageselex <- replist$ageselex
  MGparmAdj <- replist$MGparmAdj
  wtatage <- replist$wtatage
  if ("comment" %in% names(wtatage)) {
    wtatage <- wtatage[, -grep("comment", names(wtatage))]
  }
  M_at_age <- replist$M_at_age
  Growth_Parameters <- replist$Growth_Parameters
  #if (is.null(morphs)) {
    morphs <- replist$mainmorphs
  #}
  Grow_std <- replist$derived_quants[grep("Grow_std_", replist$derived_quants$Label), ]
  if (nrow(Grow_std) == 0) {
    Grow_std <- NULL
  } else {
    Grow_std$pattern <- NA
    Grow_std$sex_char <- NA
    Grow_std$sex <- NA
    Grow_std$age <- NA
    for (irow in 1:nrow(Grow_std)) {
      tmp <- strsplit(Grow_std$Label[irow], split = "_")[[1]]
      Grow_std$pattern[irow] <- as.numeric(tmp[3])
      Grow_std$sex_char[irow] <- tmp[4]
      Grow_std$age[irow] <- as.numeric(tmp[6])
    }
    Grow_std$sex[Grow_std$sex_char == "Fem"] <- 1
    Grow_std$sex[Grow_std$sex_char == "Mal"] <- 2
  }
  if (!is.null(replist$wtatage_switch)) {
    wtatage_switch <- replist$wtatage_switch
  } else {
    stop("SSplotBiology function doesn't match SS_output function.",
         "Update one or both functions.")
  }
  #if (wtatage_switch) {
  #  cat("Note: this model uses the empirical weight-at-age input.\n",
  #      "      Plots of many quantities related to growth are skipped.\n")
  #}
  if (!seas %in% 1:nseasons)
    stop("'seas' input should be within 1:nseasons")
  #if (nseasons > 1) {
  #  labels[6] <- gsub("beginning of the year", paste("beginning of season",
  #                                                   seas), labels[6])
  #}

  if (length(morphs) > nsexes) {
    cat("!Error with morph indexing in SSplotBiology function.\n",
        " Code is not set up to handle multiple growth patterns or birth seasons.\n")
  }
  #if (FecType == 1) {
  #  fec_ylab <- "Eggs per kg"
  #  fec_xlab <- labels[8]
  #  FecX <- biology$Wt_len_F
  #  FecY <- FecPar1 + FecPar2 * FecX
  #}
  #if (labels[11] != "Default fecundity label")
  #  fec_ylab <- labels[11]
  growdatF <- growdat[growdat$Sex == 1 & growdat$Morph == morphs[1], ]
  growdatF$Sd_Size <- growdatF$SD_Beg
  if (growthCVtype == "logSD=f(A)") {
    growdatF$high <- qlnorm(0.975, meanlog = log(growdatF$Len_Beg),
                            sdlog = growdatF$Sd_Size)
    growdatF$low <- qlnorm(0.025, meanlog = log(growdatF$Len_Beg),
                           sdlog = growdatF$Sd_Size)
  } else {
    growdatF$high <- qnorm(0.975, mean = growdatF$Len_Beg,
                           sd = growdatF$Sd_Size)
    growdatF$low <- qnorm(0.025, mean = growdatF$Len_Beg,
                          sd = growdatF$Sd_Size)
  }
  if (nsexes > 1) {
    growdatM <- growdat[growdat$Sex == 2 & growdat$Morph == morphs[2], ]
    xm <- growdatM$Age_Beg
    growdatM$Sd_Size <- growdatM$SD_Beg
    if (growthCVtype == "logSD=f(A)") {
      growdatM$high <- qlnorm(0.975, meanlog = log(growdatM$Len_Beg),
                              sdlog = growdatM$Sd_Size)
      growdatM$low <- qlnorm(0.025, meanlog = log(growdatM$Len_Beg),
                             sdlog = growdatM$Sd_Size)
    } else {
      growdatM$high <- qnorm(0.975, mean = growdatM$Len_Beg,
                             sd = growdatM$Sd_Size)
      growdatM$low <- qnorm(0.025, mean = growdatM$Len_Beg,
                            sd = growdatM$Sd_Size)
    }
  } else growdatM <- NULL

  list(Female = growdatF, Male = growdatM)

}



someplot<-function (replist, yrs = "all", Ftgt = NA, ylab = "Summary Fishing Mortality",
          plot = TRUE, print = FALSE, plotdir = "default", verbose = TRUE,
          uncertainty = TRUE, pwidth = 6.5, pheight = 5, punits = "in",
          res = 300, ptsize = 10)
{
  pngfun <- function(file, caption = NA) {
    png(filename = file, width = pwidth, height = pheight,
        units = punits, res = res, pointsize = ptsize)
    plotinfo <- rbind(plotinfo, data.frame(file = file, caption = caption))
    return(plotinfo)
  }
  plotinfo <- NULL
  if (plotdir == "default")
    plotdir <- replist$inputs$dir
  if (yrs[1] == "all") {
    yrs <- replist$startyr:replist$endyr
  }
  Ftot <- replist$derived_quants[match(paste("F_", yrs, sep = ""),
                                       replist$derived_quants$LABEL), ]
  if (all(is.na(Ftot$Value))) {
    warning("Skipping SSplotSummaryF because no real values found in DERIVED_QUANTITIES\n",
            "    Values with labels like F_2012 may not be real.\n")
    return()
  }
  Fmax <- max(c(Ftot$Value, Ftgt + 0.01), na.rm = TRUE)
  if (uncertainty) {
    uppFtot <- Ftot$Value + 1.96 * Ftot$StdDev
    lowFtot <- Ftot$Value - 1.96 * Ftot$StdDev
    Fmax <- max(c(uppFtot, Ftgt + 0.01), na.rm = TRUE)
  }
  plotfun <- function() {
    plot(0, type = "n", , xlab = "Year", ylab = ylab, xlim = range(yrs),
         ylim = c(0, Fmax), cex.lab = 1, cex.axis = 1, cex = 0.7)
    abline(h = 0, col = "grey")
    if (uncertainty)
      segments(as.numeric(substring(Ftot$LABEL, 3, 6)),
               uppFtot, as.numeric(substring(Ftot$LABEL, 3,
                                             6)), lowFtot, col = gray(0.5))
    points(as.numeric(substring(Ftot$LABEL, 3, 6)), Ftot$Value,
           pch = 16, type = "p")
    abline(h = Ftgt, col = "red")
  }
  if (plot)
    plotfun()
  if (print) {
    file <- file.path(plotdir, "ts_summaryF.png")
    caption <- "Summary F (definition of F depends on setting in starter.ss)"
    plotinfo <- pngfun(file = file, caption = caption)
    plotfun()
    dev.off()
    if (!is.null(plotinfo))
      plotinfo$category <- "Timeseries"
  }
  if (verbose)
    cat("Plotting Summary F\n")
  return(invisible(plotinfo))
}

