#' Function to generate a prediction expression for the two-sided Taguchi (T1)
#'   method
#'
#' \code{T1} generates a prediction expression for the two-sided Taguchi (T1)
#'   method. In \code{\link{general_T}}, the data are normalized by subtracting
#'   the mean and without scaling based on \code{unit_space_data}. The sample
#'   data should be divided into 2 datasets in advance. One is for the unit
#'   space and the other is for the signal space.
#'
#' @param unit_space_data Matrix with n rows (samples) and (p + 1) columns
#'                          (variables). The 1 ~ p th columns are independent
#'                          variables and the (p + 1) th column is a dependent
#'                          variable. Underlying data to obtain a representative
#'                          point for the normalization of the
#'                          \code{signal_space_data}. All data should be
#'                          continuous values and should not have missing values.
#' @param signal_space_data Matrix with m rows (samples) and (p + 1) columns
#'                            (variables). The 1 ~ p th columns are independent
#'                            variables and the (p + 1) th column is a dependent
#'                            variable. Underlying data to generate a prediction
#'                            expression. All data should be continuous values
#'                            and should not have missing values.
#' @param includes_transformed_data If \code{TRUE}, then the transformed data
#'                                    are included in a return object.
#'
#' @return A list containing the following components is returned.
#'
#'  \item{beta_hat}{Vector with length q. Estimated proportionality constants
#'                   between each independent variable and the dependent
#'                   variable.}
#'  \item{eta_hat}{Vector with length q. Estimated squared signal-to-noise
#'                  ratios (S/N) coresponding to \code{beta_hat}.}
#'  \item{M_hat}{Vector with length n. The estimated values of the dependent
#'                variable after the data transformation for \code{signal_space_data}.}
#'  \item{overall_prediction_eta}{Numeric. The overall squared signal-to-noise
#'                                 ratio (S/N).}
#'  \item{transforms_independent_data}{Data transformation function generated
#'                                      from \code{generates_transform_functions}
#'                                      based on the \code{unit_space_data}. The
#'                                      function for independent variables takes
#'                                      independent variable data (a matrix of p
#'                                      columns) as an (only) argument and
#'                                      returns the transformed independent
#'                                      variable data.}
#'  \item{transforms_dependent_data}{Data transformation function generated from
#'                                    \code{generates_transform_functions} based
#'                                    on the \code{unit_space_data}. The
#'                                    function for a dependent variable takes
#'                                    dependent variable data (a vector) as an
#'                                    (only) argument and returns the
#'                                    transformed dependent variable data.}
#'  \item{inverses_dependent_data}{Data transformation function generated
#'                                  from \code{generates_transform_functions}
#'                                  based on the \code{unit_space_data}. The
#'                                  function of the takes the transformed
#'                                  dependent variable data (a vector) as an
#'                                  (only) argument and returns the dependent
#'                                  variable data inversed from the transformed
#'                                  dependent variable data.}
#'  \item{m}{The number of samples for \code{signal_space_data}.}
#'  \item{q}{The number of independent variables after the data transformation.
#'            q equals p.}
#'  \item{X}{If \code{includes_transformed_data} is \code{TRUE}, then the
#'            independent variable data after the data transformation for the
#'            \code{signal_space_data} are included.}
#'  \item{M}{If \code{includes_transformed_data} is \code{TRUE}, then the (true)
#'            value of the dependent variable after the data transformation for
#'            the \code{signal_space_data} are included.}
#'
#' @references
#'   Taguchi, G. (2006). Objective Function and Generic Function (12).
#'     \emph{Journal of Quality Engineering Society, 14}(3), 5-9. (In Japanese)
#'
#'   Inou, A., Nagata, Y., Horita, K., & Mori, A. (2012). Prediciton Accuracies
#'     of Improved Taguchi's T Methods Compared to those of Multiple Regresssion
#'     Analysis. \emph{Journal of the Japanese Society for Quality Control,
#'     42}(2), 103-115. (In Japanese)
#'
#'   Kawada, H., & Nagata, Y. (2015). An application of a generalized inverse
#'     regression estimator to Taguchi's T-Method. \emph{Total Quality Science,
#'     1}(1), 12-21.
#'
#' @seealso \code{\link{general_T}},
#'            \code{\link{generates_transformation_functions_T1}}, and
#'            \code{\link{forecasting.T1}}
#'
#' @examples
#' # The value of the dependent variable of the following samples mediates
#' # in the stackloss dataset.
#' stackloss_center <- stackloss[c(9, 10, 11, 20, 21), ]
#'
#' # The following samples are data other than the unit space data and the test
#' # data.
#' stackloss_signal <- stackloss[-c(2, 9, 10, 11, 12, 19, 20, 21), ]
#'
#' model_T1 <- T1(unit_space_data = stackloss_center,
#'                signal_space_data = stackloss_signal,
#'                includes_transformed_data = TRUE)
#'
#' (model_T1$M_hat)
#'
#' @export
T1 <- function(unit_space_data,
               signal_space_data,
               includes_transformed_data = FALSE) {

  model_T1 <- general_T(unit_space_data = unit_space_data,
                        signal_space_data = signal_space_data,
                        generates_transform_functions =
                                          generates_transformation_functions_T1,
                        includes_transformed_data = includes_transformed_data)

  class(model_T1) <- "T1"

  return(model_T1)

}

#' Forecasting method for the T1 method
#'
#' \code{forecasting.T1} (via \code{\link{forecasting}}) estimates the dependent
#'   values based on the T1 model.
#'
#' @param model Object of class "T1" generated by \code{\link{T1}} or
#'                \code{\link{generates_model}}(..., method = "T1").
#' @param newdata Matrix with n rows (samples) and p columns (variables). The
#'                  Data to be estimated. All data should be continuous values
#'                  and should not have missing values.
#' @param includes_transformed_newdata If \code{TRUE}, then the transformed data
#'                                       for \code{newdata} are included in a
#'                                       return object.
#'
#' @return A list containing the following components is returned.
#'
#'  \item{M_hat}{Vector with length n. The estimated values of the dependent
#'                variable after the data transformation.}
#'  \item{y_hat}{Vector with length n. The estimated values after the inverse
#'                transformation from \code{M_hat}.}
#'  \item{model}{Object of class "T1" passed by \code{model}.}
#'  \item{n}{The number of samples for \code{newdata}.}
#'  \item{q}{The number of variables after the data transformation. q equals p.}
#'  \item{X}{If \code{includes_transformed_newdata} is \code{TRUE}, then the
#'            transformed data for \code{newdata} are included.}
#'
#' @references
#'   Taguchi, G. (2006). Objective Function and Generic Function (12).
#'     \emph{Journal of Quality Engineering Society, 14}(3), 5-9. (In Japanese)
#'
#'   Inou, A., Nagata, Y., Horita, K., & Mori, A. (2012). Prediciton Accuracies
#'     of Improved Taguchi's T Methods Compared to those of Multiple Regresssion
#'     Analysis. \emph{Journal of the Japanese Society for Quality Control,
#'     42}(2), 103-115. (In Japanese)
#'
#'   Kawada, H., & Nagata, Y. (2015). An application of a generalized inverse
#'     regression estimator to Taguchi's T-Method. \emph{Total Quality Science,
#'     1}(1), 12-21.
#'
#' @seealso \code{\link{general_forecasting.T}} and \code{\link{T1}}
#'
#' @examples
#' # The value of the dependent variable of the following samples mediates
#' # in the stackloss dataset.
#' stackloss_center <- stackloss[c(9, 10, 11, 20, 21), ]
#'
#' # The following samples are data other than the unit space data and the test
#' # data.
#' stackloss_signal <- stackloss[-c(2, 9, 10, 11, 12, 19, 20, 21), ]
#'
#' model_T1 <- T1(unit_space_data = stackloss_center,
#'                signal_space_data = stackloss_signal,
#'                includes_transformed_data = TRUE)
#'
#' # The following test samples are chosen casually.
#' stackloss_test <- stackloss[c(2, 12, 19), -4]
#'
#' forecasting_T1 <- forecasting(model = model_T1,
#'                               newdata = stackloss_test,
#'                               includes_transformed_newdata = TRUE)
#'
#' (forecasting_T1$y_hat) # Estimated values
#' (stackloss[c(2, 12, 19), 4]) # True values
#'
#' @export
forecasting.T1 <- function(model,
                           newdata,
                           includes_transformed_newdata = FALSE) {

  if (!inherits(model, "T1")) {
    warning("calling forecasting.T1(<fake-T1-object>) ...")
  }

  general_forecasting.T(model = model,
                        newdata = newdata,
                        includes_transformed_newdata =
                                                   includes_transformed_newdata)

}
