% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matchit2nearest.R
\name{method_nearest}
\alias{method_nearest}
\title{Nearest Neighbor Matching}
\arguments{
\item{formula}{a two-sided \link{formula} object containing the treatment and
covariates to be used in creating the distance measure used in the matching.}

\item{data}{a data frame containing the variables named in \code{formula}.
If not found in \code{data}, the variables will be sought in the
environment.}

\item{method}{set here to \code{"nearest"}.}

\item{distance}{the distance measure to be used. See \code{\link{distance}}
for allowable options. Can be supplied as a distance matrix.}

\item{link}{when \code{distance} is specified as a method of estimating
propensity scores, an additional argument controlling the link function used
in estimating the distance measure. See \code{\link{distance}} for allowable
options with each option.}

\item{distance.options}{a named list containing additional arguments
supplied to the function that estimates the distance measure as determined
by the argument to \code{distance}.}

\item{estimand}{a string containing the desired estimand. Allowable options
include \code{"ATT"} and \code{"ATC"}. See Details.}

\item{exact}{for which variables exact matching should take place; two units with different values of an exact matching variable will not be paired.}

\item{mahvars}{for which variables Mahalanobis distance matching should take
place when \code{distance} corresponds to a propensity score (e.g., for
caliper matching or to discard units for common support). If specified, the
distance measure will not be used in matching.}

\item{antiexact}{for which variables anti-exact matching should take place; two units with the same value of an anti-exact matching variable will not be paired.}

\item{discard}{a string containing a method for discarding units outside a
region of common support. Only allowed when \code{distance} corresponds to a
propensity score.}

\item{reestimate}{if \code{discard} is not \code{"none"}, whether to
re-estimate the propensity score in the remaining sample prior to matching.}

\item{s.weights}{the variable containing sampling weights to be incorporated
into propensity score models and balance statistics.}

\item{replace}{whether matching should be done with replacement (i.e., whether control units can be used as matches multiple times). See also the \code{reuse.max} argument below. Default is \code{FALSE} for matching without replacement.}

\item{m.order}{the order that the matching takes place. Allowable options
include \code{"largest"}, where matching takes place in descending order of
distance measures; \code{"smallest"}, where matching takes place in ascending
order of distance measures; \code{"closest"}, where matching takes place in
ascending order of the smallest distance between units; \code{"farthest"}, where matching takes place in
descending order of the smallest distance between units; \code{"random"}, where matching takes place
in a random order; and \code{"data"} where matching takes place based on the
order of units in the data. When \code{m.order = "random"}, results may differ
across different runs of the same code unless a seed is set and specified
with \code{\link[=set.seed]{set.seed()}}. The default of \code{NULL} corresponds to \code{"largest"} when a
propensity score is estimated or supplied as a vector and \code{"data"}
otherwise. See Details for more information.}

\item{caliper}{the width(s) of the caliper(s) used for caliper matching. Two units with a difference on a caliper variable larger than the caliper will not be paired. See Details and Examples.}

\item{std.caliper}{\code{logical}; when calipers are specified, whether they
are in standard deviation units (\code{TRUE}) or raw units (\code{FALSE}).}

\item{ratio}{how many control units should be matched to each treated unit
for k:1 matching. For variable ratio matching, see section "Variable Ratio
Matching" in Details below. When \code{ratio} is greater than 1, all treated units will be attempted to be matched with a control unit before any treated unit is matched with a second control unit, etc. This reduces the possibility that control units will be used up before some treated units receive any matches.}

\item{min.controls, max.controls}{for variable ratio matching, the minimum
and maximum number of controls units to be matched to each treated unit. See
section "Variable Ratio Matching" in Details below.}

\item{verbose}{\code{logical}; whether information about the matching
process should be printed to the console. When \code{TRUE}, a progress bar
implemented using \emph{RcppProgress} will be displayed along with an estimate of the time remaining.}

\item{\dots}{additional arguments that control the matching specification:
\describe{
\item{\code{reuse.max}}{ \code{numeric}; the maximum number of
times each control can be used as a match. Setting \code{reuse.max = 1}
corresponds to matching without replacement (i.e., \code{replace = FALSE}),
and setting \code{reuse.max = Inf} corresponds to traditional matching with
replacement (i.e., \code{replace = TRUE}) with no limit on the number of
times each control unit can be matched. Other values restrict the number of
times each control can be matched when matching with replacement.
\code{replace} is ignored when \code{reuse.max} is specified.  }
\item{\code{unit.id}}{ one or more variables containing a unit ID for each
observation, i.e., in case multiple observations correspond to the same
unit. Once a control observation has been matched, no other observation with
the same unit ID can be used as matches. This ensures each control unit is
used only once even if it has multiple observations associated with it.
Omitting this argument is the same as giving each observation a unique ID.}
}}
}
\description{
In \code{\link[=matchit]{matchit()}}, setting \code{method = "nearest"} performs greedy nearest
neighbor matching. A distance is computed between each treated unit and each
control unit, and, one by one, each treated unit is assigned a control unit
as a match. The matching is "greedy" in the sense that there is no action
taken to optimize an overall criterion; each match is selected without
considering the other matches that may occur subsequently.

This page details the allowable arguments with \code{method = "nearest"}.
See \code{\link[=matchit]{matchit()}} for an explanation of what each argument means in a general
context and how it can be specified.

Below is how \code{matchit()} is used for nearest neighbor matching:
\preformatted{
matchit(formula,
        data = NULL,
        method = "nearest",
        distance = "glm",
        link = "logit",
        distance.options = list(),
        estimand = "ATT",
        exact = NULL,
        mahvars = NULL,
        antiexact = NULL,
        discard = "none",
        reestimate = FALSE,
        s.weights = NULL,
        replace = TRUE,
        m.order = NULL,
        caliper = NULL,
        ratio = 1,
        min.controls = NULL,
        max.controls = NULL,
        verbose = FALSE,
        ...) }
}
\details{
\subsection{Mahalanobis Distance Matching}{

Mahalanobis distance matching can be done one of two ways:
\enumerate{
\item{If no propensity score needs to be estimated, \code{distance} should be
set to \code{"mahalanobis"}, and Mahalanobis distance matching will occur
using all the variables in \code{formula}. Arguments to \code{discard} and
\code{mahvars} will be ignored, and a caliper can only be placed on named
variables. For example, to perform simple Mahalanobis distance matching, the
following could be run:

\preformatted{
matchit(treat ~ X1 + X2, method = "nearest",
        distance = "mahalanobis") }

With this code, the Mahalanobis distance is computed using \code{X1} and
\code{X2}, and matching occurs on this distance. The \code{distance}
component of the \code{matchit()} output will be empty.
}
\item{If a propensity score needs to be estimated for any reason, e.g., for
common support with \code{discard} or for creating a caliper,
\code{distance} should be whatever method is used to estimate the propensity
score or a vector of distance measures. Use \code{mahvars} to specify the
variables used to create the Mahalanobis distance. For example, to perform
Mahalanobis within a propensity score caliper, the following could be run:

\preformatted{
matchit(treat ~ X1 + X2 + X3, method = "nearest",
        distance = "glm", caliper = .25,
        mahvars = ~ X1 + X2) }

With this code, \code{X1}, \code{X2}, and \code{X3} are used to estimate the
propensity score (using the \code{"glm"} method, which by default is
logistic regression), which is used to create a matching caliper. The actual
matching occurs on the Mahalanobis distance computed only using \code{X1}
and \code{X2}, which are supplied to \code{mahvars}. Units whose propensity
score difference is larger than the caliper will not be paired, and some
treated units may therefore not receive a match. The estimated propensity
scores will be included in the \code{distance} component of the
\code{matchit()} output. See Examples.
}
}
}

\subsection{Estimand}{

The \code{estimand} argument controls whether control units are selected to be
matched with treated units (\code{estimand = "ATT"}) or treated units are
selected to be matched with control units (\code{estimand = "ATC"}). The
"focal" group (e.g., the treated units for the ATT) is typically made to be
the smaller treatment group, and a warning will be thrown if it is not set
that way unless \code{replace = TRUE}. Setting \code{estimand = "ATC"} is
equivalent to swapping all treated and control labels for the treatment
variable. When \code{estimand = "ATC"}, the default \code{m.order} is
\code{"smallest"}, and the \code{match.matrix} component of the output will
have the names of the control units as the rownames and be filled with the
names of the matched treated units (opposite to when \code{estimand = "ATT"}). Note that the argument supplied to \code{estimand} doesn't
necessarily correspond to the estimand actually targeted; it is merely a
switch to trigger which treatment group is considered "focal".
}

\subsection{Variable Ratio Matching}{

\code{matchit()} can perform variable ratio "extremal" matching as described by Ming and Rosenbaum (2000). This
method tends to result in better balance than fixed ratio matching at the
expense of some precision. When \code{ratio > 1}, rather than requiring all
treated units to receive \code{ratio} matches, each treated unit is assigned
a value that corresponds to the number of control units they will be matched
to. These values are controlled by the arguments \code{min.controls} and
\code{max.controls}, which correspond to \eqn{\alpha} and \eqn{\beta},
respectively, in Ming and Rosenbaum (2000), and trigger variable ratio
matching to occur. Some treated units will receive \code{min.controls}
matches and others will receive \code{max.controls} matches (and one unit
may have an intermediate number of matches); how many units are assigned
each number of matches is determined by the algorithm described in Ming and
Rosenbaum (2000, p119). \code{ratio} controls how many total control units
will be matched: \code{n1 * ratio} control units will be matched, where
\code{n1} is the number of treated units, yielding the same total number of
matched controls as fixed ratio matching does.

Variable ratio matching cannot be used with Mahalanobis distance matching or
when \code{distance} is supplied as a matrix. The calculations of the
numbers of control units each treated unit will be matched to occurs without
consideration of \code{caliper} or \code{discard}. \code{ratio} does not
have to be an integer but must be greater than 1 and less than \code{n0/n1},
where \code{n0} and \code{n1} are the number of control and treated units,
respectively. Setting \code{ratio = n0/n1} performs a crude form of full
matching where all control units are matched. If \code{min.controls} is not
specified, it is set to 1 by default. \code{min.controls} must be less than
\code{ratio}, and \code{max.controls} must be greater than \code{ratio}. See
Examples below for an example of their use.
}

\subsection{Using \code{m.order = "closest"} or \code{"farthest"}}{

\code{m.order} can be set to \code{"closest"} or \code{"farthest"}, which work regardless of how the distance measure is specified. This matches in order of the distance between units. First, all the closest match is found for all treated units and the pairwise distances computed; when \code{m.order = "closest"} the pair with the smallest of the distances is matched first, and when \code{m.order = "farthest"}, the pair with the largest of the distances is matched first. Then, the pair with the second smallest (or largest) is matched second. If the matched control is ineligible (i.e., because it has already been used in a prior match), a new match is found for the treated unit, the new pair's distance is re-computed, and the pairs are re-ordered by distance.

Using \code{m.order = "closest"} ensures that the best possible matches are given priority, and in that sense should perform similarly to \code{m.order = "smallest"}. It can be used to ensure the best matches, especially when matching with a caliper. Using \code{m.order = "farthest"} ensures that the hardest units to match are given their best chance to find a close match, and in that sense should perform similarly to \code{m.order = "largest"}. It can be used to reduce the possibility of extreme imbalance when there are hard-to-match units competing for controls. Note that \code{m.order = "farthest"} \strong{does not} implement "far matching" (i.e., finding the farthest control unit from each treated unit); it defines the order in which the closest matches are selected.
}

\subsection{Reproducibility}{

Nearest neighbor matching involves a random component only when \code{m.order = "random"} (or when the propensity is estimated using a method with randomness; see \code{\link{distance}} for details), so a seed must be set in that case using \code{\link[=set.seed]{set.seed()}} to ensure reproducibility. Otherwise, it is purely deterministic, and any ties are broken based on the order in which the data appear.
}
}
\note{
Sometimes an error will be produced by \emph{Rcpp} along the lines of
\code{"function 'Rcpp_precious_remove' not provided by package 'Rcpp'"}. It
is not immediately clear why this happens, though
\href{https://lists.r-forge.r-project.org/pipermail/rcpp-devel/2021-July/010648.html}{this}
thread appears to provide some insight. In a fresh session, run
\verb{remove.packages(c("MatchIt", "Rcpp")); install.packages("MatchIt")}.
This should sync \emph{MatchIt} and \emph{Rcpp} and ensure they work
correctly.
}
\section{Outputs}{

All outputs described in \code{\link[=matchit]{matchit()}} are returned with
\code{method = "nearest"}. When \code{replace = TRUE}, the \code{subclass}
component is omitted. \code{include.obj} is ignored.
}

\examples{
data("lalonde")

# 1:1 greedy NN matching on the PS
m.out1 <- matchit(treat ~ age + educ + race + nodegree +
                    married + re74 + re75, data = lalonde,
                  method = "nearest")
m.out1
summary(m.out1)

# 3:1 NN Mahalanobis distance matching with
# replacement within a PS caliper
m.out2 <- matchit(treat ~ age + educ + race + nodegree +
                    married + re74 + re75, data = lalonde,
                  method = "nearest", replace = TRUE,
                  mahvars = ~ age + educ + re74 + re75,
                  ratio = 3, caliper = .02)
m.out2
summary(m.out2, un = FALSE)

# 1:1 NN Mahalanobis distance matching within calipers
# on re74 and re75 and exact matching on married and race
m.out3 <- matchit(treat ~ age + educ + re74 + re75, data = lalonde,
                  method = "nearest", distance = "mahalanobis",
                  exact = ~ married + race,
                  caliper = c(re74 = .2, re75 = .15))
m.out3
summary(m.out3, un = FALSE)

# 2:1 variable ratio NN matching on the PS
m.out4 <- matchit(treat ~ age + educ + race + nodegree +
                    married + re74 + re75, data = lalonde,
                  method = "nearest", ratio = 2,
                  min.controls = 1, max.controls = 12)
m.out4
summary(m.out4, un = FALSE)

# Some units received 1 match and some received 12
table(table(m.out4$subclass[m.out4$treat == 0]))

}
\references{
In a manuscript, you don't need to cite another package when
using \code{method = "nearest"} because the matching is performed completely
within \emph{MatchIt}. For example, a sentence might read:

\emph{Nearest neighbor matching was performed using the MatchIt package (Ho, Imai, King, & Stuart, 2011) in R.}
}
\seealso{
\code{\link[=matchit]{matchit()}} for a detailed explanation of the inputs and outputs of
a call to \code{matchit()}.

\code{\link[=method_optimal]{method_optimal()}} for optimal pair matching, which is similar to
nearest neighbor matching without replacement except that an overall distance criterion is
minimized (i.e., as an alternative to specifying \code{m.order}).
}
