% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parseDatetime.R
\name{parseDatetime}
\alias{parseDatetime}
\title{Parse datetime strings}
\usage{
parseDatetime(datetime = NULL, timezone = NULL, expectAll = FALSE,
  isJulian = FALSE, quiet = TRUE)
}
\arguments{
\item{datetime}{Vector of character or integer datetimes in Ymd[HMS] format
(or POSIXct).}

\item{timezone}{Olson timezone used to interpret dates (required).}

\item{expectAll}{Logical value determining if the function should fail if
any elements fail to parse (default \code{FALSE}).}

\item{isJulian}{Logical value determining whether \code{datetime} should be
interpreted as a Julian date with day of year as a decimal number.}

\item{quiet}{Logical value passed on to \code{lubridate::parse_date_time} to
optionally suppress warning messages.}
}
\value{
A vector of POSIXct datetimes.
}
\description{
Transforms numeric and string representations of Ymd[HMS] datetimes to
\code{POSIXct} format.

Y, Ym, Ymd, YmdH, YmdHM, and YmdHMS formats are understood, where:

\describe{
  \item{Y}{four digit year}
  \item{m}{month number (1-12, 01-12) or english name month (October, oct.)}
  \item{d}{day number of the month (0-31 or 01-31)}
  \item{H}{hour number (0-24 or 00-24)}
  \item{M}{minute number (0-59 or 00-59)}
  \item{S}{second number (0-61 or 00-61)}
}

This allows for mixed inputs. For example, 20181012130900,
"2018-10-12-13-09-00", and "2018 Oct. 12 13:09:00" will all be converted to
the same \code{POSIXct} datetime. The incoming datetime vector does not need
to have a homogeneous format either -- "20181012" and "2018-10-12 13:09" can
exist in the same vector without issue. All incoming datetimes will be
interpreted in the specified timezone.

If \code{datetime} is a \code{POSIXct} it will be returned unmodified, and
formats not recognized will be returned as \code{NA}.
}
\section{Mazama Science Conventions}{

Within Mazama Science package, datetimes not in \code{POSIXct} format are
often represented as decimal values with no separation (ex: 20181012,
20181012130900), either as numerics or strings.
}

\section{Implementation}{

\code{parseDatetime} is essentially a wrapper around
\code{\link[lubridate]{parse_date_time}}, handling which formats we want to
account for.
}

\examples{
# All y[md-hms] formats are accepted
parseDatetime(2018, timezone = "America/Los_Angeles")
parseDatetime(201808, timezone = "America/Los_Angeles")
parseDatetime(20180807, timezone = "America/Los_Angeles")
parseDatetime(2018080718, timezone = "America/Los_Angeles")
parseDatetime(201808071812, timezone = "America/Los_Angeles")
parseDatetime(20180807181215, timezone = "America/Los_Angeles")
parseDatetime("2018-08-07 18:12:15", timezone = "America/Los_Angeles")

# Julian days are accepeted
parseDatetime(2018219181215, timezone = "America/Los_Angeles",
              isJulian = TRUE)

# Vector dates are accepted and daylight savings is respected
parseDatetime(
  c("2018-10-24 12:00", "2018-10-31 12:00",
    "2018-11-07 12:00", "2018-11-08 12:00"),
  timezone = "America/New_York"
)

badInput <- c("20181013", NA, "20181015", "181016", "10172018")

# Return a vector with \\code{NA} for dates that could not be parsed
parseDatetime(badInput, timezone = "UTC", expectAll = FALSE)

\dontrun{
# Fail if any dates cannot be parsed
parseDatetime(badInput, timezone = "UTC", expectAll = TRUE)
}

}
\seealso{
\code{\link[lubridate]{parse_date_time}} for implementation details.
}
