\name{model.interaction.plot}
\alias{model.interaction.plot}

\title{
plot of two-way model interactions
}
\description{
Image or Perspective plot of two-way model interactions. Ranges of two specified predictor variables are plotted on X and Y axis, and fitted model values are plotted on the Z axis. The remaining predictor variables are fixed at their mean (for continuous predictors) or their most common value (for categorical predictors).
}
\usage{
model.interaction.plot(model.obj = NULL, x = NULL, y = NULL, 
response.category=NULL, quantiles=NULL, all=FALSE, obs=1, qdata.trainfn = NULL, 
folder = NULL, MODELfn = NULL,  PLOTfn = NULL, pred.means = NULL, 
xlab = NULL, ylab = NULL, x.range = NULL, y.range = NULL, 
z.range = NULL, ticktype = "detailed", theta = 55, phi = 40, 
smooth = "none", plot.type = NULL, device.type = NULL, 
res=NULL, jpeg.res = 72, device.width = 7,  device.height = 7, 
units="in", pointsize=12, cex=par()$cex, 
col = NULL, xlim = NULL, ylim = NULL, zlim = NULL, ...)
}

\arguments{
    \item{model.obj}{ \code{R} model object.  The model object to use for prediction.  The model object must be of type \code{"RF"} (random forest), \code{"QRF"} (quantile random forest), \code{"CF"} (conditional forest), or \code{"SGB"} (stochastic gradient boosting). }

  \item{x}{String or Integer. Name of predictor variable to be plotted on the x axis. Alternativly, can be a number indicating a variable name from \code{predList}.}

  \item{y}{String or Integer. Name of predictor variable to be plotted on the y axis. Alternatively, can be a number indicating a variable name from \code{predList}.}

   \item{response.category}{String. Used for categorical response models. Specify which category of response variable to use. This category's probabilities will be plotted on the z axis.}

   \item{quantiles}{Numeric. Used for QRF models. Specify which quantile of response variable to use. This quantile will be plotted on the z axis. Note: unlike other functions \code{model.interaction.plot} will only use a single quantile. If \code{quantiles} is a vector only the first value will be used.}


   \item{all}{Logical. Used for QRF models. A logical value. \code{all=TRUE} uses all observations for prediction. \code{all=FALSE} uses only a certain number of observations per node for prediction (set with argument \code{obs}). The default is \code{all=FALSE}.}


   \item{obs}{Numeric. Used for QRF models. An integer number. Determines the maximal number of observations per node to use for prediction. The input is ignored for all=TRUE. The default is obs=1.}

   \item{qdata.trainfn}{String.  The name (full path or base name with path specified by \code{folder}) of the training data file used for building the model (file should include columns for both response and predictor variables).  The file must be a comma-delimited file \code{*.csv} with column headings. \code{qdata.trainfn} can also be an \code{R} dataframe. If predictions will be made (\code{predict = TRUE} or \code{map=TRUE}) the predictor column headers must match the names of the raster layer files, or a \code{rastLUT} must be provided to match predictor columns to the appropriate raster and band.  If \code{qdata.trainfn = NULL} (the default), a GUI interface prompts user to browse to the training data file.  }

  \item{folder}{ String.  The folder used for all output.  Do not add ending slash to path string.  If \code{folder = NULL} (default), a GUI interface prompts user to browse to a folder.  To use the working directory, specify \code{folder = getwd()}.}

  \item{MODELfn}{ String.  The file name used to save the generated model object, only used if \code{PLOTfn = NULL}.  If \code{MODELfn} is supplied and If \code{PLOTfn = NULL}, a graphical file name is generated by pasting \code{MODELfn_plot.type_x.name_y.name}. If \code{PLOTfn = NULL} and \code{MODELfn = NULL}, a default name is generated by pasting \code{model.type_response.type_response.name_plot.type_x.name_y.name}. The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}.}

  \item{PLOTfn}{ String.  The file name to use to save the generated graphical plots. The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}.}

  \item{pred.means}{Vector. Allows specification of values for other predictor variables. If Null, other predictors are set to their mean value (for continuous predictors) or their most common value (for factored predictors).}

  \item{xlab}{String. Allows manual specification of the x label.}

  \item{ylab}{String. Allows manual specification of the y label.}

  \item{x.range}{Vector. Manual range specification for the x axis. Alternate argument name for \code{xlim}. Use one or the other. Do not provide both \code{x.range} and \code{xlim}.}

  \item{y.range}{Vector. Manual range specification for the y axis. Alternate argument name for \code{ylim}. Use one or the other. Do not provide both \code{y.range} and \code{ylim}.}

  \item{z.range}{Vector. Manual range specification for the z axis. Alternate argument name for \code{zlim}. Use one or the other. Do not provide both \code{z.range} and \code{zlim}.}

  \item{ticktype}{Character: "simple" draws just an arrow parallel to the axis to indicate direction of increase; "detailed" (default) draws normal ticks as per 2D plots. If \code{X} or \code{y} is factored, ticks will be drawn on both axes.}

  \item{theta}{Numeric. Angles defining the viewing direction. \code{theta} gives the azimuthal direction.}

  \item{phi}{Numeric. Angles defining the viewing direction. \code{phi} gives the colatitude.}

  \item{smooth}{String. controls smoothing of the predicted surface. Options are \code{"none"} (default), \code{"model"} which uses a glm model to smooth the surface, and \code{"average"} which applies a 3x3 smoothing average. Note: smoothing is not appropriate if \code{X} or \code{y} is factored.}

  \item{plot.type}{Character. \code{"persp"} gives a 3-D perspective plot. \code{"image"} gives an image plot.}
 
 \item{device.type}{ String or vector of strings.  Model validation.  One or more device types for graphical output from model validation diagnostics. 

Current choices:

\tabular{lllll}{
	  \tab \tab \tab \code{"default"} \tab default graphics device\cr
	  \tab \tab \tab \code{"jpeg"} \tab *.jpg files\cr
	  \tab \tab \tab \code{"none"} \tab no graphics device generated\cr	
	  \tab \tab \tab \code{"pdf"} \tab *.pdf files\cr
	  \tab \tab \tab \code{"postscript"} \tab *.ps files\cr
	  \tab \tab \tab \code{"win.metafile"} \tab *.emf files }

 }

  \item{res}{ Integer.  Model validation.  Pixels per inch for jpeg, png, and tiff plots.  The default is 72dpi, good for on screen viewing. For printing, suggested setting is 300dpi. }

  \item{jpeg.res}{ Integer.  Model validation.  Deprecated. Ignored unless \code{res} not provided. }

  \item{device.width}{ Integer.  Model validation.  The device width for diagnostic plots in inches. }

  \item{device.height}{ Integer.  Model validation.  The device height for diagnostic plots in inches. }

  \item{units}{ Model validation.  The units in which \code{device.height} and \code{device.width} are given. Can be \code{"px"} (pixels), \code{"in"} (inches, the default), \code{"cm"} or \code{"mm"}. }

  \item{pointsize}{ Integer.  Model validation.  The default pointsize of plotted text, interpreted as big points (1/72 inch) at \code{res} ppi}

  \item{cex}{ Integer.  Model validation.  The cex for diagnostic plots. }

  \item{col}{ Vector.  Color table to use for image plots ( see help file on image for details). }

  \item{xlim}{Vector. X limits. Alternate argument name for \code{x.range}. Use one or the other. Do not provide both \code{x.range} and \code{xlim}.}

  \item{ylim}{Vector. Y limits. Alternate argument name for \code{y.range}. Use one or the other. Do not provide both \code{y.range} and \code{ylim}.}

  \item{zlim}{Vector. Z limits. Alternate argument name for \code{z.range}. Use one or the other. Do not provide both \code{z.range} and \code{zlim}.}

  \item{\dots}{additional graphical parameters (see \code{\link{par}}).}
}
\details{
This function provides a diagnostic plot useful in visualizing two-way interactions between predictor variables. Two of the predictor variables from the model are used to produce a grid of possible combinations of predictor values over the range of both variables. The remaining predictor variables from the model are fixed at either their means (for continuous predictors) or their most common value (for categorical predictors). Model predictions are generated over this grid and plotted as the z axis.

This function works with both continuous and categorical predictors, though the perspective plot should be interpreted with care for categorical predictors. In particular, the \code{smooth} option is not appropriate if either of the two selected predictor variables is categorical.

For categorical response models, a particular value must be specified for the response using the \code{response.category} argument. 
}

\references{
This function is adapted from \code{gbm.perspec} version 2.9 April 2007, J Leathwick/J Elith. See appendix S3 from:


Elith, J., Leathwick, J. R. and Hastie, T. (2008). A working guide to boosted regression trees. Journal of Animal Ecology. 77:802-813.
}
\author{ Elizabeth Freeman }


\examples{
###########################################################################
############################# Run this set up code: #######################
###########################################################################

# set seed:
seed=38

# Define training and test files:

qdata.trainfn = system.file("extdata", "helpexamples","DATATRAIN.csv", package = "ModelMap")
qdata.testfn = system.file("extdata", "helpexamples","DATATEST.csv", package = "ModelMap")

# Define folder for all output:
folder=getwd()	

########## Continuous Response, Categorical Predictors ############


#file name to store model:
MODELfn="RF_BIO_TCandNLCD"			

#predictors:
predList=c("TCB","TCG","TCW","NLCD")

#define which predictors are categorical:
predFactor=c("NLCD")

# Response name and type:
response.name="BIO"
response.type="continuous"

#identifier for individual training and test data points

unique.rowname="ID"


###########################################################################
########################### build model: ##################################
###########################################################################


### create model ###

model.obj = model.build( model.type="RF",
                       qdata.trainfn=qdata.trainfn,
                       folder=folder,		
                       unique.rowname=unique.rowname,		
                       MODELfn=MODELfn,
                       predList=predList,
                       predFactor=predFactor,
                       response.name=response.name,
                       response.type=response.type,
                       seed=seed,
                       na.action=na.roughfix
)

###########################################################################
###################### make interaction plots: ############################
###########################################################################

#########################
### Perspective Plots ###
#########################


### specify first and third  predictors in 'predList (both continuous) ###

model.interaction.plot(	model.obj,
			x=1,y=3, 
			main=response.name, 
			plot.type="persp", 
			device.type="default") 

\dontrun{
### specify predictors in 'predList' by name (one continuous one factored) ###

model.interaction.plot(model.obj,
			x="TCB", y="NLCD", 
			main=response.name, 
			plot.type="persp", 
			device.type="default") 


###################
### Image Plots ###
###################

### same as previous example, but image plot ###


l <- seq(100,0,length.out=101)
c <- seq(0,100,length.out=101)
col.ramp <- hcl(h = 120, c = c, l = l)
 

model.interaction.plot(	model.obj,
				x="TCB", y="NLCD",
				main=response.name,
				plot.type="image", 
				device.type="default",
				col = col.ramp) 


#########################
### 3-way Interaction ###
#########################

### use 'pred.means' argument to fix values of additional predictors ###

### factored 3rd predictor ###

interaction between TCG and TCW for 3 most common values of NLCD

nlcd<-levels(model.obj$predictor.data$NLCD)
nlcd.counts<-table(model.obj$predictor.data$NLCD)
nlcd.ordered<-nlcd[order(nlcd.counts,decreasing=TRUE)]

for(i in nlcd.ordered[1:3]){
	pred.means=list(NLCD=i)

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("NLCD=",i," (",nlcd.counts[i]," plots)", sep=""),
				pred.means=pred.means, 
				z.range=c(0,110),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}



### continuos 3rd predictor ###


tcb<-seq(	min(model.obj$predictor.data$TCB),
		max(model.obj$predictor.data$TCB),
		length=3)

tcb<-signif(tcb,2)

for(i in tcb){
	pred.means=list(TCB=i)

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("TCB =",i),
				pred.means=pred.means, 
				z.range=c(0,120),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}



### 4-way Interesting combos ###


tcb=c(1300,2900,3400)
nlcd=c(11,90,95)

for(i in 1:3){
	pred.means=list(TCB=tcb[i],NLCD=nlcd[i])

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("TCB =",tcb[i],"        NLCD =",nlcd[i]),
				pred.means=pred.means, 
				z.range=c(0,120),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}
} #end dontRun

}


\keyword{ models }

