% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eigs.R
\name{eigs_sym}
\alias{eigs_sym}
\title{Find a few eigenvalues and vectors on large, sparse Hermitian matrix}
\usage{
eigs_sym(A, NEig = 1, which = "LA", targetShifts = NULL,
  tol = 1e-06, x0 = NULL, ortho = NULL, prec = NULL,
  isreal = NULL, B = NULL, ...)
}
\arguments{
\item{A}{symmetric/Hermitian matrix or a function with signature f(x) that
returns \code{A \%*\% x}.}

\item{NEig}{number of eigenvalues and vectors to seek.}

\item{which}{which eigenvalues to find:
\describe{
   \item{\code{"LA"}}{the largest (rightmost) values;}
   \item{\code{"SA"}}{the smallest (leftmost) values;}
   \item{\code{"LM"}}{the farthest from \code{targetShifts};}
   \item{\code{"SM"}}{the closest to \code{targetShifts};}
   \item{\code{"CLT"}}{the closest but left to \code{targetShifts};}
   \item{\code{"CGT"}}{the closest but greater than \code{targetShifts};}
   \item{vector of numbers}{the closest values to these points.}
}}

\item{targetShifts}{return the closest eigenvalues to these points as
indicated by \code{target}.}

\item{tol}{the convergence tolerance:
\eqn{\|A x - x\lambda\| \le tol\|A\|}{||A*x - x*lambda|| <= tol*||A||}.}

\item{x0}{matrix whose columns are educated guesses of the eigenvectors to
to find.}

\item{ortho}{find eigenvectors orthogonal to the space spanned by the
columns of this matrix; useful to avoid finding some eigenvalues or
to find new solutions.}

\item{prec}{preconditioner used to accelerated the convergence; usually it
is an approximation of the inverse of \eqn{A - \sigma I} if finding
the closest eigenvalues to \eqn{\sigma}. If it is a matrix
it is used as prec \%*\% x; otherwise it is used as prec(x).}

\item{isreal}{whether A \%*\% x always returns real number and not complex.}

\item{B}{symmetric/Hermitian positive definite matrix or a function with
signature f(x) that returns \code{B \%*\% x}. If given, the function
returns the eigenpairs of (A,B).}

\item{...}{other PRIMME options (see details).}
}
\value{
list with the next elements
   \describe{
      \item{\code{values}}{the eigenvalues \eqn{\lambda_i}}
      \item{\code{vectors}}{the eigenvectors \eqn{x_i}}
      \item{\code{rnorms}}{the residual vector norms
         \eqn{\|A x_i - \lambda_i B x_i\|}{||A*x_i - lambda_i*B*x_i||}.}
      \item{\code{stats$numMatvecs}}{number of matrix-vector products performed}
      \item{\code{stats$numPreconds}}{number of preconditioner applications performed}
      \item{\code{stats$elapsedTime}}{time expended by the eigensolver}
      \item{\code{stats$timeMatvec}}{time expended in the matrix-vector products}
      \item{\code{stats$timePrecond}}{time expended in applying the preconditioner}
      \item{\code{stats$timeOrtho}}{time expended in orthogonalizing}
      \item{\code{stats$estimateMinEval}}{estimation of the smallest eigenvalue of A}
      \item{\code{stats$estimateMaxEval}}{estimation of the largest eigenvalue of A}
      \item{\code{stats$estimateANorm}}{estimation of the norm of A}
   }
}
\description{
Compute a few eigenpairs from a specified region (the largest, the smallest,
the closest to a point) on a symmetric/Hermitian matrix using PRIMME [1].
Generalized symmetric/Hermitian problem is also supported.
Only the matrix-vector product of the matrix is required. The used method is
usually faster than a direct method (such as \code{\link{eigen}}) if
seeking a few eigenpairs and the matrix-vector product is cheap. For
accelerating the convergence consider to use preconditioning and/or educated
initial guesses.
}
\details{
Optional arguments to pass to PRIMME eigensolver (see further details at [2]):

\describe{
\item{\code{method}}{ used by the solver, one of:
   \describe{
   \item{\code{"DYNAMIC"}}{                  switches dynamically between DEFAULT_MIN_TIME and DEFAULT_MIN_MATVECS}
   \item{\code{"DEFAULT_MIN_TIME"}}{         best method for light matrix-vector product}
   \item{\code{"DEFAULT_MIN_MATVECS"}}{      best method for heavy matrix-vector product or preconditioner}
   \item{\code{"Arnoldi"}}{                  an Arnoldi not implemented efficiently}
   \item{\code{"GD"}}{                       classical block Generalized Davidson }
   \item{\code{"GD_plusK"}}{                 GD+k block GD with recurrence restarting}
   \item{\code{"GD_Olsen_plusK"}}{           GD+k with approximate Olsen preconditioning}
   \item{\code{"JD_Olsen_plusK"}}{           GD+k, exact Olsen (two preconditioner applications per step)}
   \item{\code{"RQI"}}{                      Rayleigh Quotient Iteration, also Inverse Iteration
                                             if \code{targetShifts} is provided}
   \item{\code{"JDQR"}}{                     original block, Jacobi Davidson}
   \item{\code{"JDQMR"}}{                    our block JDQMR method (similar to JDCG)}
   \item{\code{"JDQMR_ETol"}}{               slight, but efficient JDQMR modification}
   \item{\code{"STEEPEST_DESCENT"}}{         equivalent to GD(\code{maxBlockSize},2*\code{maxBlockSize})}
   \item{\code{"LOBPCG_OrthoBasis"}}{        equivalent to GD(\code{neig},3*\code{neig})+\code{neig}}
   \item{\code{"LOBPCG_OrthoBasis_Window"}}{ equivalent to GD(\code{maxBlockSize},3*\code{maxBlockSize})+\code{maxBlockSize} when neig>\code{maxBlockSize}}
   }}
   \item{\code{aNorm}}{estimation of norm-2 of A, used in convergence test (if not
       provided, it is estimated as the largest eigenvalue in magnitude
       seen).}
   \item{\code{maxBlockSize}}{maximum block size (like in subspace iteration or
       LOBPCG).}
   \item{\code{printLevel}}{message level reporting, from 0 (no output) to 5 (show all).} 
   \item{\code{locking}}{1, hard locking; 0, soft locking.}
   \item{\code{maxBasisSize}}{maximum size of the search subspace.}
   \item{\code{minRestartSize}}{ minimum Ritz vectors to keep in restarting.}
   \item{\code{maxMatvecs}}{ maximum number of matrix vector multiplications.}
   \item{\code{maxit}}{ maximum number of outer iterations.}
   \item{\code{scheme}}{ the restart scheme (thick restart by default).}
   \item{\code{maxPrevRetain}}{ number of approximate eigenvectors retained from
         previous iteration, that are kept after restart.}
   \item{\code{robustShifts}}{ set to true to avoid stagnation.}
   \item{\code{maxInnerIterations}}{ maximum number of inner QMR iterations.}
   \item{\code{LeftQ}}{ use the locked vectors in the left projector.}
   \item{\code{LeftX}}{ use the approx. eigenvector in the left projector.}
   \item{\code{RightQ}}{ use the locked vectors in the right projector.}
   \item{\code{RightX}}{ use the approx. eigenvector in the right projector.}
   \item{\code{SkewQ}}{ use the preconditioned locked vectors in the right projector.}
   \item{\code{SkewX}}{ use the preconditioned approximate eigenvector in the right
               projector.}
   \item{\code{relTolBase}}{ a legacy from classical JDQR (recommend not use).}
   \item{\code{iseed}}{ an array of four numbers used as a random seed.}
}
}
\examples{
A <- diag(1:10)  # the eigenvalues of this matrix are 1:10 and the
                 # eigenvectors are the columns of diag(10)
r <- eigs_sym(A, 3);
r$values  # the three largest eigenvalues on diag(1:10)
r$vectors # the corresponding approximate eigenvectors
r$rnorms # the corresponding residual norms
r$stats$numMatvecs # total matrix-vector products spend

r <- eigs_sym(A, 3, 'SA') # compute the three smallest values

r <- eigs_sym(A, 3, 2.5) # compute the three closest values to 2.5

r <- eigs_sym(A, 3, 2.5, tol=1e-3); # compute the values with
r$rnorms                                    # residual norm <= 1e-3*||A||

B <- diag(rev(1:10));
r <- eigs_sym(A, 3, B=B); # compute the 3 largest eigenpairs of
                          # the generalized problem (A,B)

# Build a Jacobi preconditioner (too convenient for a diagonal matrix!)
# and see how reduce the number matrix-vector products
A <- diag(1:1000)   # we use a larger matrix to amplify the difference
P <- diag(diag(A) - 2.5)
eigs_sym(A, 3, 2.5, tol=1e-3)$stats$numMatvecs
eigs_sym(A, 3, 2.5, tol=1e-3, prec=P)$stats$numMatvecs

# Passing A and the preconditioner as functions
Af <- function(x) (1:100) * x; # = diag(1:100) \%*\% x
Pf <- function(x) x / (1:100 - 2.5); # = solve(diag(1:100 - 2.5), x)
r <- eigs_sym(Af, 3, 2.5, tol=1e-3, prec=Pf, n=100)

# Passing initial guesses
A <- diag(1:1000)   # we use a larger matrix to amplify the difference
x0 <- diag(1,1000,4) + matrix(rnorm(4000), 1000, 4)/100;
eigs_sym(A, 4, "SA", tol=1e-3)$stats$numMatvecs
eigs_sym(A, 4, "SA", tol=1e-3, x0=x0)$stats$numMatvecs

# Passing orthogonal constrain, in this case, already compute eigenvectors
r <- eigs_sym(A, 4, "SA", tol=1e-3); r$values
eigs_sym(A, 4, "SA", tol=1e-3, ortho=r$vectors)$values

}
\references{
[1] A. Stathopoulos and J. R. McCombs \emph{PRIMME: PReconditioned Iterative
    MultiMethod Eigensolver: Methods and software description}, ACM
    Transaction on Mathematical Software Vol. 37, No. 2, (2010)
    21:1-21:30.

[2] \url{http://www.cs.wm.edu/~andreas/software/doc/primmec.html#parameters-guide}
}
\seealso{
\code{\link{eigen}} for computing all values;
\code{\link{svds}} for computing a few singular values
}
