#' Plot actual and forecasted values of an univariate time series
#'
#' Takes an univariate time series and a vector with forecasted values.
#' @param x The trained PSF model generated by psf() function.
#' @param predictions A vector with previously computed forecasted values.
#' @param cycle The number of values that conform a cycle in the time series (e.g. 24 hours per day, 12 month per year, and so on). Only used when input data is not in time series (ts) format.
#' @param \dots Additional graphical parameters given to plot function.
#' @export
#' @examples
#' ## Train a PSF model from the univariate time series 'nottem' (package:datasets).
#' p <- psf(nottem)
#'
#' ## Forecast the next 12 values of such time series.
#' pred <- predict(p, n.ahead = 12)
#'
#' ## Plot forecasted values.
#' plot(p, pred)
plot.psf <- function (x, predictions = c(), cycle = 24, ...) {

  if (is.ts(x))
    all = ts(c(x$original_series, predictions), start=start(x$original_series), frequency = frequency(x$original_series))
  else
    all = ts(c(x$original_series, predictions), start=1, frequency = cycle)

  args <- list(...)

  if (length(args) == 0)
  {
    plot(window(all, end = time(all)[length(all) - length(predictions)]), xlim = c(time(all)[1], time(all)[length(all)]), xlab = "Time", ylab = "Value")
    points(window(all, start = time(all)[length(all) - length(predictions) + 1]), type = "o", col = "blue", lty = 3, pch = 16, cex = 0.4)
  }
  else
  {
    plot(window(all, end = time(all)[length(all) - length(predictions)]), xlim = c(time(all)[1], time(all)[length(all)]), ...)
    points(window(all, start = time(all)[length(all) - length(predictions) + 1]), type = "o", col = "blue", ...)
  }

}
