###########################################################################/**
# @RdocClass TabularTextFile
#
# @title "The TabularTextFile class"
#
# \description{
#  @classhierarchy
#
#  A TabularTextFile is an object refering to a tabular text file 
#  on a file system containing data in a tabular format.
#  Methods for reading all or a subset of the tabular data exist.
# }
# 
# \usage{TabularTextFile(..., sep=c("\t", ","), quote="\"", fill=FALSE, skip=0, columnNames=NA, commentChar="#", .verify=TRUE, verbose=FALSE)}
#
# \arguments{
#   \item{...}{Arguments passed to @see "GenericTabularFile".}
#   \item{sep}{A @character specifying the symbol used to separate the 
#     cell entries.  If more than one symbol is specified, it will try to
#     select the correct one by peeking into the file.}
#   \item{quote}{A @character specifying the quote symbol used, if any.}
#   \item{fill}{As in @see "utils::read.table".}
#   \item{skip}{As in @see "utils::read.table".}
#   \item{columnNames}{A @logical or a @character @vector. If @TRUE,
#      then column names are inferred from the file.  If a @character
#      @vector, then the column names are given by this argument.}
#   \item{commentChar}{A single @character specifying which symbol
#      should be used for comments, cf. @see "utils::read.table".}
#   \item{.verify, verbose}{(Internal only) If @TRUE, the file is 
#      verified while the object is instantiated by the constructor.
#      The verbose argument is passed to the verifier function.}
# }
#
# \section{Fields and Methods}{
#  @allmethods "public"
# }
#
# @author
#
# \seealso{
#   An object of this class is typically part of an @see "TabularTextFileSet".
# }
#*/###########################################################################
setConstructorS3("TabularTextFile", function(..., sep=c("\t", ","), quote="\"", fill=FALSE, skip=0, columnNames=NA, commentChar="#", .verify=TRUE, verbose=FALSE) {
  # Argument 'columnNames':
  if (is.logical(columnNames)) {
    readColumnNames <- columnNames;
    columnNames <- NULL;
  } else if (is.character(columnNames)) {
    readColumnNames <- FALSE;
  } else {
    throw("Argument 'columnNames' must be either a logical or a character vector: ", class(columnNames)[1]);
  }

  # Argument 'commentChar':
  if (!is.null(commentChar)) {
    commentChar <- Arguments$getCharacter(commentChar, nchar=c(1,1));
  }


  this <- extend(GenericTabularFile(..., .verify=FALSE), "TabularTextFile",
    .fileHeader = NULL,
    .columnNameTranslator = NULL,
    sep = sep,
    quote = quote,
    fill = fill,
    skip = skip,
    .commentChar = commentChar,
    .columnNames = columnNames,
    readColumnNames = readColumnNames
  );

  if (.verify) {
    verify(this, ..., verbose=verbose);
    # Clear temporary settings
    this$.fileHeader <- NULL;
  }

  this;
})


setMethodS3("as.character", "TabularTextFile", function(x, ...) {
  # To please R CMD check
  this <- x;

  s <- NextMethod("as.character");
  class <- class(s);
  colnames <- getColumnNames(this);
  if (length(colnames) > 0L) {
    columns <- paste("'", colnames, "'", sep="");
    s <- c(s, sprintf("Columns [%d]: %s", length(columns), paste(columns, collapse=", ")));
  } else {
    s <- c(s, sprintf("Columns [NA]: <not reading column names>"));
  }
  s <- c(s, sprintf("Number of text lines: %d", nbrOfLines(this, fast=TRUE)));

  class(s) <- class;
  s;
})


setMethodS3("verify", "TabularTextFile", function(this, ..., verbose=FALSE) {
  # Nothing to do?
  pathname <- getPathname(this);
  if (is.null(pathname) || is.na(pathname))
    return(invisible(this));


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Argument 'verbose':
  verbose <- Arguments$getVerbose(verbose);
  if (verbose) {
    pushState(verbose);
    on.exit(popState(verbose));
  }


  verbose && enter(verbose, "Validating file contents");

  tryCatch({
    data <- readDataFrame(this, skip=this$skip, nrow=10, verbose=verbose);
  }, error = function(ex) {
    throw("File format error of the tabular file ('", getPathname(this), "'): ", ex$message);
  })

  verbose && exit(verbose);

  invisible(this);
}, private=TRUE)


setMethodS3("getCommentChar", "TabularTextFile", function(this, ...) {
  this$.commentChar;
})


setMethodS3("setCommentChar", "TabularTextFile", function(this, ch, ...) {
  if (!is.null(ch)) {
    ch <- Arguments$getCharacter(ch, nchar=c(1,1));
  }
  this$.commentChar <- ch;
  invisible(this);
})


setMethodS3("readColumnNames", "TabularTextFile", function(this, ...) {
  res <- as.logical(this$readColumnNames);

  # No need to infer from header?
  if (!is.na(res)) {
    return(res);
  }

  hdr <- readRawHeader(this, ...);
  namesHdr <- hdr$commentArgs$columnNames;
  if (is.null(namesHdr)) {
    msg <- sprintf("Cannot infer whether the data table has column names or not, because header comment argument 'columnNames' is missing. Will assume there are column names (readColumnNames=TRUE): %s", getPathname(this));
#    warning(msg);
    return(TRUE);
  }

  # If argument 'columnNames' equals the first data row, then yes.
  namesData <- hdr$topRows[[1]];
  res <- all(namesData == namesHdr);

  # Store results(?!?)
#  this$readColumnNames <- res;

  res;
}, protected=TRUE)


###########################################################################/**
# @RdocMethod hasColumnHeader
#
# @title "Checks if there are column names in the header"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Not used.}
# }
#
# \value{
#   Returns a @logical.
# }
# @author
#
# \seealso{
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("hasColumnHeader", "TabularTextFile", function(this, ...) {
  readColumnNames(this);
}, protected=TRUE)



###########################################################################/**
# @RdocMethod getDefaultColumnNames
#
# @title "Gets the default column names"
#
# \description{
#  @get "title" by inferring the from the file header.
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Optional arguments passed @seemethod "getHeader".}
# }
#
# \value{
#   Returns @character @vector,
#   or @NULL if there are no column names in the file header.
# }
# @author
#
# \seealso{
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("getDefaultColumnNames", "TabularTextFile", function(this, ...) {
  hdr <- getHeader(this, ...);

  # (a) Infer column names from data table
  if (hasColumnHeader(this)) {
    names <- hdr$columns;
    return(names);
  }

  # (b) Infer column names from header argument 'columnNames'?
  useHeaderArgs <- this$.useHeaderArgs;
  if (is.null(useHeaderArgs)) useHeaderArgs <- TRUE;
  if (useHeaderArgs) {
    args <- hdr$commentArgs;
    if (length(args) > 0L) {
      names <- args$columnNames;
      return(names);
    }
  }

  # (c) There are no column names
  NULL;
}, protected=TRUE)


setMethodS3("getDefaultColumnClasses", "TabularTextFile", function(this, ...) {
  ncol <- nbrOfColumns(this);

  hdr <- getHeader(this, ...);

  # (a) Infer column names from header argument 'columnNames'?
  useHeaderArgs <- this$.useHeaderArgs;
  if (is.null(useHeaderArgs)) useHeaderArgs <- TRUE;
  if (useHeaderArgs) {
    args <- hdr$commentArgs;
    if (length(args) > 0L) {
      colClasses <- args$columnClasses;
      if (!is.null(colClasses)) {
        # Sanity check
        stopifnot(length(colClasses) == ncol);
        return(colClasses);
      }
    }
  }

  # (b) Column classes are not specified in the file
  NULL;
}, protected=TRUE)


setMethodS3("getDefaultColumnClassPatterns", "TabularTextFile", function(this, ...) {
  names <- getColumnNames(this, ...);
  colClasses <- getDefaultColumnClasses(this, ...);
  if (is.null(colClasses)) return(NULL);

  names <- sprintf("^%s$", names);
  names(colClasses) <- names;
  colClasses;
}, protected=TRUE)



###########################################################################/**
# @RdocMethod getHeader
#
# @title "Gets the file header"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Passed to internal @seemethod "readRawHeader".}
#   \item{force}{If @TRUE, an already retrieved header will be ignored.}
# }
#
# \value{
#   Returns a named @list.
# }
# @author
#
# \seealso{
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("getHeader", "TabularTextFile", function(this, ..., force=FALSE) {
  hdr <- this$.fileHeader;
  if (force || is.null(hdr) || hasBeenModified(this)) {
    hdr <- readRawHeader(this, ...);
    if (hasColumnHeader(this)) {
      hdr$columns <- hdr$topRows[[1]];
    }
    this$.fileHeader <- hdr;
  }
  hdr;
})




setMethodS3("readRawHeader", "TabularTextFile", function(this, con=NULL, ..., verbose=FALSE) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Argument 'verbose':
  verbose <- Arguments$getVerbose(verbose);
  if (verbose) {
    pushState(verbose);
    on.exit(popState(verbose));
  }

  verbose && enter(verbose, "Reading tabular file header from ", class(this)[1]);

  # Open a file connection?
  if (is.null(con)) {
    pathname <- getPathname(this);
    verbose && cat(verbose, "Pathname: ", pathname);

    # Open file connection
    con <- file(pathname, open="r");
    on.exit({
      if (!is.null(con)) {
        close(con);
        con <- NULL;
      }
    })
  }


  # Read header comments
  comments <- c();
  skip <- this$skip;
  ch <- getCommentChar(this);
  if (!is.null(ch)) {
    pattern <- sprintf("^%s", ch);
    ready <- FALSE;
    while (!ready) {
      line <- readLines(con, n=1);
      isEmpty <- (regexpr("^$", line) != -1);
      if (!isEmpty) {
        isComments <- (regexpr(pattern, line) != -1);
        if (!isComments) {
          if (skip == 0)
            break;
          skip <- skip - 1;
        }
        comments <- c(comments, line);
      }
    } # while(!ready)
  }

  verbose && cat(verbose, "Header comments:", level=-20);
  verbose && str(verbose, comments, level=-20);


  # Parse header comments
  pattern <- sprintf("^%s[ ]*([^:]+):[ ]*(.*)", ch);
  commentArgs <- grep(pattern, comments, value=TRUE);
  if (length(commentArgs) > 0L) {
    keys <- gsub(pattern, "\\1", commentArgs);
    keys <- trim(keys);
    commentArgs <- gsub(pattern, "\\2", commentArgs);
    commentArgs <- trim(commentArgs);
    commentArgs <- strsplit(commentArgs, split="\t", fixed=TRUE);
    names(commentArgs) <- keys;
  } else {
    commentArgs <- NULL;
  }
  verbose && cat(verbose, "Header comment arguments:", level=-20);
  verbose && str(verbose, commentArgs, level=-20);


  # Infer column separator from the first line after the header comments?
  sep <- this$sep;
  if (length(sep) > 1) {
    verbose && enter(verbose, "Identifying the separator that returns most columns");
    verbose && cat(verbose, "Line:");
    verbose && print(verbose, line);
    verbose && cat(verbose, "Separators:");
    verbose && str(verbose, sep);
    columns <- base::lapply(sep, FUN=function(split) {
      strsplit(line, split=split)[[1]];
    });
    nbrOfColumns <- sapply(columns, FUN=length);
    max <- which.max(nbrOfColumns);
    sep <- sep[max];
    verbose && printf(verbose, "Choosen separator: '%s' (0x%s)\n", sep, charToRaw(sep));
    verbose && exit(verbose);
  }

  lines <- c(line, readLines(con, n=9));
  verbose && print(verbose, line);
  topRows <- strsplit(lines, split=sep);
  topRows <- lapply(topRows, trim);
  verbose && print(verbose, topRows);

  # Remove quotes?
  quote <- this$quote;
  if (!is.null(quote)) {
    for (pattern in c(sprintf("^%s", quote), sprintf("%s$", quote))) {
      topRows <- lapply(topRows, FUN=function(row) {
        gsub(pattern, "", row);
      })
    }
  }

  verbose && cat(verbose, "Columns: ", paste(paste("'", topRows, "'", sep=""), collapse=", "), level=-10);

  hdr <- list(
    comments=comments,
    commentArgs=commentArgs,
    sep=sep,
    quote=quote,
    skip=this$skip,
    topRows=topRows
  );

  verbose && str(verbose, hdr);

  verbose && exit(verbose);

  hdr;
}, protected=TRUE); # readRawHeader()


setMethodS3("getReadArguments", "TabularTextFile", function(this, fileHeader=NULL, colClassPatterns=c("*"=NA, getDefaultColumnClassPatterns(this)), defColClass="NULL", stringsAsFactors=FALSE, ..., verbose=FALSE) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Argument 'fileHeader':
  if (is.null(fileHeader)) {
    fileHeader <- getHeader(this);
  }

  # Argument 'verbose':
  verbose <- Arguments$getVerbose(verbose);
  if (verbose) {
    pushState(verbose);
    on.exit(popState(verbose));
  }



  verbose && enter(verbose, "Building arguments for read.table()");


  # Optional user arguments
  userArgs <- list(stringsAsFactors=stringsAsFactors, ...);
  verbose && cat(verbose, "User arguments:");
  verbose && str(verbose, userArgs);


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Infer column classes
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Default column classes
  columns <- getColumnNames(this);
  if (!is.null(columns)) {
    nbrOfColumns <- length(columns);
    defColClasses <- rep(defColClass, times=nbrOfColumns);
    defColClassPatterns <- defColClasses;

    # Default columns?
    names <- names(colClassPatterns);
    pos <- which(names == "*");
    if (length(pos) > 0) {
      # Exclude extra '*':s
      if (length(pos) > 1) {
        colClassPatterns <- colClassPatterns[-(pos[-1])];
        pos <- pos[1];
      }
  
      # Insert defaults
      colClass <- colClassPatterns[pos];
      names <- names(colClassPatterns);
      if (length(colClassPatterns) > 1) {
        names <- insert(names[-pos], at=pos, values=rep("*", times=nbrOfColumns));
        idxs <- whichVector(names == "*");
        names[idxs] <- sprintf("^%s$", columns);
  
        colClassPatterns <- insert(colClassPatterns[-pos], at=pos, 
                                   values=rep("*", times=nbrOfColumns));
        names(colClassPatterns) <- names;
        colClassPatterns[idxs] <- colClass;
      } else {
        colClassPatterns <- rep(colClass, times=nbrOfColumns);
        names(colClassPatterns) <- sprintf("^%s$", columns);
      }
    }

    verbose && cat(verbose, "Pattern used to identify column classes:", level=-20);
    verbose && print(verbose, colClassPatterns, level=-20);
  
    verbose && cat(verbose, "Generate column classes:");
    # Read everything by default
    colClasses <- defColClasses;
    names(colClasses) <- columns;
  
    # Update column classes according to patterns
    for (kk in seq_along(colClassPatterns)) {
      pattern <- names(colClassPatterns)[kk];
      idxs <- which(regexpr(pattern, columns) != -1);
      if (length(idxs) > 0) {
        colClass <- colClassPatterns[kk];
        colClasses[idxs] <- colClass;
      }
    }
  } else {
    colClasses <- userArgs$colClasses;
  }
  
  verbose && cat(verbose, "Column classes:", level=-20);
  verbose && print(verbose, colClasses, level=-20);


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Setup read.table() arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Inferred arguments
  args <- list(
    header       = hasColumnHeader(this),
    colClasses   = colClasses,
    skip         = fileHeader$skip,
    sep          = fileHeader$sep,
    quote        = fileHeader$quote,
    fill         = this$fill,
    comment.char = getCommentChar(this),
    check.names  = FALSE,
    na.strings   = c("---", "NA")
  );

  # Overwrite with user specified arguments, if any
  if (length(userArgs) > 0) {
    verbose && enter(verbose, "Overwriting inferred arguments with user arguments");
    for (key in names(userArgs)) {
      args[[key]] <- userArgs[[key]];
    }
    verbose && exit(verbose);
  }

  verbose && exit(verbose);


  args;
}, protected=TRUE);




###########################################################################/**
# @RdocMethod readDataFrame
#
# @title "Reads the tabular data as a data frame"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{con}{(Internal) If a @connection, then it is used, otherwies
#   a new file connection is temporarly opened and used.}
#   \item{rows}{(Optional) An @integer @vector specifying which rows to
#    be read.}
#   \item{nrow}{(Optional) An @integer specifying how many rows to read.
#    If specified, it corresponds to specifying \code{rows=seq_len(nrow)}.}
#   \item{trimQuotes}{(Optional) If @TRUE, quotes are trimmed from numeric
#    columns before parsing them as numerics.  This makes it possible to
#    read quoted numeric values.}
#   \item{...}{Passed to internal @seemethod "getReadArguments".}
#   \item{debug}{If @TRUE, additional details on the file and how it was
#    read is returned as part of the attributes.}
#   \item{verbose}{A @logical or a @see "R.utils::Verbose" object.}
# }
#
# \value{
#   Returns a @data.frame.
# }
#
# \section{Reading quoted numerics}{
#   If a specific data column is specified as being numeric in
#   argument \code{colClasses} and that column contains quoted values
#   it is necessary to use argument \code{trimQuotes=TRUE}, otherwise
#   @see "base::scan" throws an exception similar to:
#   \code{scan() expected 'a real', got '"1.0"'}.
# }
#
# @author
#
# \seealso{
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("readDataFrame", "TabularTextFile", function(this, con=NULL, rows=NULL, nrow=NULL, trimQuotes=FALSE, ..., debug=FALSE, verbose=FALSE) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Argument 'rows' and 'nrow':
  if (!is.null(rows) && !is.null(nrow)) {
    throw("Only one of arguments 'rows' and 'nrow' can be specified.");
  }

  if (!is.null(rows)) {
    rows <- Arguments$getIndices(rows);
    nrow <- max(rows);
  }

  if (!is.null(nrow)) {
    nrow <- Arguments$getInteger(nrow, range=c(1,Inf));
  }

  # Argument 'trimQuotes':
  trimQuotes <- Arguments$getLogical(trimQuotes);

  # Argument 'debug':
  debug <- Arguments$getLogical(debug);
  
  # Argument 'verbose':
  verbose <- Arguments$getVerbose(verbose);
  if (verbose) {
    pushState(verbose);
    on.exit(popState(verbose));
  }

  verbose && enter(verbose, "Reading ", class(this)[1]);


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Reading header to infer read.table() arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  hdr <- getHeader(this, verbose=less(verbose, 5));


  # Get read arguments
  args <- getReadArguments(this, fileHeader=hdr, nrow=nrow, ..., 
                                               verbose=less(verbose, 5));

  verbose && cat(verbose, "Arguments inferred from file header:");
  verbose && print(verbose, args);

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Identify names of columns read
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  columns <- getColumnNames(this);
  verbose && printf(verbose, "Column names (%d):\n", length(columns));
  verbose && cat(verbose, paste(columns, collapse=", "));

  if (!is.null(columns)) {
    verbose && enter(verbose, "Matching column names:");
    verbose && printf(verbose, "Column classes (%d):\n", 
                                                length(args$colClasses));
    verbose && cat(verbose, paste(args$colClasses, collapse=", "));
    columns[args$colClasses == "NULL"] <- NA;
    columns <- columns[!is.na(columns)];
    verbose && exit(verbose);
  }


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Open a file connection?
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (is.null(con)) {
    pathname <- getPathname(this);
    verbose && cat(verbose, "Pathname: ", pathname);

    # Open file connection
    con <- file(pathname, open="r");
    on.exit({
      if (!is.null(con)) {
        close(con);
        con <- NULL;
      }
    })
  }


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Reading data
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  fcnName <- "read.table";
#  fcnName <- "readTable";  ## BUGGY /HB 2008-06-17
  verbose && enter(verbose, sprintf("Calling %s()", fcnName));


  # SPECIAL CASE/WORKAROUND: read.table()/scan() will give an error
  # if a numeric value is quoted and 'colClasses' specifies it as
  # a numeric value.  In order to read such values, we need to remove
  # the quotes first. /HB 2011-07-13

  # Check if we need to trim quotes
  trimQuotes <- (trimQuotes && nchar(args$quote) > 0);
  trimQuotes <- (trimQuotes && length(args$colClasses) > 0);
  if (trimQuotes) {
    classesToPatch <- c("integer", "numeric", "double", "complex");
    toPatch <- is.element(args$colClasses, classesToPatch);
    trimQuotes <- c(trimQuotes && any(toPatch));
  }

  # Read by first trimming quotes?
  if (trimQuotes) {
    verbose && enter(verbose, "Trimming quotes from numeric columns before parsing them as numbers");
    # This is a workaround for reading quoted numerics. /HB 2011-07-13
    verbose && cat(verbose, "Columns that need to have quotes trimmed:");
    verbose && str(verbose, which(toPatch));
    args0 <- args;

    # (1) Read as to-be-patched columns as characters.
    colClasses <- args$colClasses;
    colClasses[toPatch] <- "character";
    args$colClasses <- colClasses;
  }

  verbose && cat(verbose, "Arguments used to read tabular file:");
  args <- c(list(con), args);
  verbose && print(verbose, args);
  data <- do.call(fcnName, args=args);
  nbrOfRowsRead <- nrow(data);
  verbose && cat(verbose, "Raw data read by read.table():");
  verbose && str(verbose, data);
  verbose && cat(verbose, "Number of rows read: ", nbrOfRowsRead);
  verbose && cat(verbose, "Number of columns read: ", ncol(data));

  # Not needed anymore
  rm(args);

  # Was data read by first trimming quotes?
  if (trimQuotes) {
    # (2) Re-read numeric columns one by one
    colClasses <- args0$colClasses;

    # Note that some columns may have been ignored
    keep <- (colClasses != "NULL");
    colClasses <- colClasses[keep];
    toPatch <- toPatch[keep];

    # Sanity check
    stopifnot(length(colClasses) == ncol(data));

    na.strings <- args0$na.strings;

    verbose && enter(verbose, "Parsing numeric columns");
    toPatch <- which(toPatch);
    for (kk in seq_along(toPatch)) {
      col <- toPatch[kk];
      colClass <- colClasses[col];
      verbose && enter(verbose, sprintf("Parsing #%d (column #%d as '%s') of %d", kk, col, colClass, length(toPatch)));
      values <- data[[col]];

      # Quick/bad way, but will turn non-valid values into NA
      ##  storage.mode(values) <- colClass;

      verbose && cat(verbose, "Non-parsed values:");
      verbose && str(verbose, values);

      bfr <- paste(values, collapse="\n");
      conT <- textConnection(bfr, open="r");
      on.exit({
        if (!is.null(conT)) close(conT);
      }, add=TRUE);
      
      # Try to read the values as the correct type.
      valuesT <- read.table(file=conT, quote="", colClasses=colClass, na.strings=na.strings, blank.lines.skip=FALSE)[[1]];

      verbose && cat(verbose, "Parsed values:");
      verbose && str(verbose, valuesT);

      # Sanity check
      stopifnot(length(valuesT) == length(values));

      close(conT); conT <- NULL;
      data[[col]] <- valuesT;

      # Not needed anymore
      rm(bfr, values, valuesT);
      verbose && exit(verbose);
    } # for (kk ...)
    verbose && exit(verbose);

    verbose && exit(verbose);
  }


  # Extract subset of rows?
  if (fcnName == "read.table") {
    if (!is.null(rows)) {
      if (max(rows) > nbrOfRowsRead) {
        rows <- intersect(rows, 1:nbrOfRowsRead);
        warning("Argument 'rows' was out of range [1,", nbrOfRowsRead,
                                  "]. Ignored rows beyond this range.");
      }
      data <- data[rows,,drop=FALSE];
    } else {
      rownames(data) <- NULL;
    }
  } else {
    rownames(data) <- NULL;
  }

  # Sanity check
  if (length(columns) > 0) {
    if (ncol(data) != length(columns)) {
      throw("Number of read data columns does not match the number of column headers: ", ncol(data), " !=", length(columns));
    }
    colnames(data) <- columns;
  }

  verbose && str(verbose, data);
  verbose && exit(verbose);

  if (debug) {
    attr(data, "fileHeader") <- hdr;
  }

  verbose && exit(verbose);

  data;
}) # readDataFrame()



setMethodS3("[", "TabularTextFile", function(this, i=NULL, j=NULL, drop=FALSE) {
  # Read data
  data <- readDataFrame(this, rows=i, columns=j);

  # Drop dimensions?
  if (drop) {
    if (ncol(data) == 1) {
      data <- data[,1];
    } else if (nrow(data) == 1) {
      data <- data[1,];
    }
  }
  
  data;
})


setMethodS3("readColumns", "TabularTextFile", function(this, columns, colClasses=rep("character", times=length(columns)), ...) {
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Validate arguments
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # Arguments 'columns':
  if (is.numeric(columns)) {
    columnNames <- getColumnNames(this);
    columns <- Arguments$getIndices(columns, max=length(columnNames));
    columnNames <- columnNames[columns];
  } else {
    columnNames <- Arguments$getCharacters(columns);
  }


  colClassPatterns <- colClasses;
  names(colClassPatterns) <- sprintf("^%s$", columnNames);
  readDataFrame(this, colClassPatterns=colClassPatterns, ...);
}, protected=TRUE)



# AD HOC fix to speed up ll(), which calls dimension() on each object, 
# which in turn calls dim() and dim() is really slow for this class,
# because it has to infer the number of rows by reading the complete
# file. The fix is to return NA for the number of rows if the file size
# is larger than 10MB unless nbrOfLines() has already been called.
# /HB 2008-07-22
setMethodS3("dimension", "TabularTextFile", function(this, ...) {
  c(nbrOfRows(this, fast=TRUE), nbrOfColumns(this, fast=TRUE));
}, private=TRUE);



###########################################################################/**
# @RdocMethod nbrOfRows
#
# @title "Counts the number of data rows"
#
# \description{
#  @get "title".  The count does not include the header rows or comments.
# }
#
# @synopsis
#
# \arguments{
#   \item{fast}{Argument passed to @seemethod "nbrOfLines".}
#   \item{...}{Optional arguments passed to @seemethod "getHeader".}
# }
#
# \value{
#   Returns a @character @vector.
# }
# @author
#
# \seealso{
#    The number of data rows is always less or equal to the number of lines
#    as returned by @seemethod "nbrOfLines".
#   Internally, @see "R.utils::countLines" is used.
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("nbrOfRows", "TabularTextFile", function(this, fast=FALSE, ...) {
  hdr <- getHeader(this, ...);
  n <- nbrOfLines(this, fast=fast);
  n <- n - hdr$skip - as.integer(length(hdr$columns) > 0);
  n <- as.integer(n);
  n;
})



###########################################################################/**
# @RdocMethod nbrOfLines
#
# @title "Counts the number of lines in the data file"
#
# \description{
#  @get "title".  The count include header rows, comments and more.
# }
#
# @synopsis
#
# \arguments{
#   \item{fast}{If @TRUE, @NA is returned for large data files (>1Mb),
#     unless the number of lines has already been counted.}
#   \item{...}{Optional arguments passed to @see "R.utils::countLines".}
# }
#
# \value{
#   Returns a @character @vector.
# }
#
# @author
#
# \seealso{
#    To count the number of data rows is the data table, 
#    use @seemethod "nbrOfRows".
#   Internally, @see "R.utils::countLines" is used.
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("nbrOfLines", "TabularTextFile", function(this, fast=FALSE, ...) {
  pathname <- getPathname(this);

  n <- this$.nbrOfLines;
  mtime <- getLastModifiedOn(this);
  if (is.null(n) || is.na(mtime) || !identical(mtime, attr(n, "mtime"))) {
    if (fast) {
      if (getFileSize(this) < 10e6)
        fast <- FALSE;
    }

    if (fast) {
      n <- as.integer(NA);
    } else {
      n <- countLines(pathname, ...);
      attr(n, "mtime") <- mtime;
      this$.nbrOfLines <- n;
    }
  }

  n;
})




###########################################################################/**
# @RdocMethod readLines
#
# @title "Reads the lines of the data file as strings"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{...}{Optional arguments passed to @see "base::readLines".}
# }
#
# \value{
#   Returns a @character @vector.
# }
# @author
#
# \seealso{
#   @seemethod "readDataFrame".
#   @seeclass
# }
#
# @keyword IO
# @keyword programming
#*/###########################################################################
setMethodS3("readLines", "TabularTextFile", function(con, ...) {
  # To please R CMD check
  this <- con;
  pathname <- getPathname(this);
  readLines(pathname, ...);
})




############################################################################
# HISTORY:
# 2012-11-15
# o Made it possible for TabularTextFile to ignore header comment 
#   arguments when inferring column names and classes.
# 2012-11-08
# o Now getReadArguments() for TabularTextFile also includes column
#   class patterns from getDefaultColumnClassPatterns().
# o Added getDefaultColumnClass() and getDefaultColumnClassPatterns()
#   to TabularTextFile.
# o Now getDefaultColumnNames() for TabularTextFile falls back to 
#   header comment argument 'columnNames', if there are no column
#   names in the actual data table.
# o Now readRawHeader() for TabularTextFile also parses and returns
#   header comment arguments.
# 2012-11-02
# o Added getDefaultColumnNames() for TabularTextFile and dropped
#   getColumnNames(), which is implemented by ColumnNamesInterface.
# 2012-10-31
# o CLEANUP: Now readDataFrame() for TabularTextFile no longer returns
#   attribute 'fileHeader', unless argument 'debug' is TRUE.
# 2012-09-27
# o ROBUSTNESS: Now getHeader() for TabularTextFile checks if the file
#   has been modified before returned cached results.
# o Added argument 'stringsAsFactors=FALSE' to getReadArguments() for 
#   TabularTextFile such that the default is to read strings as character
#   rather than as factors.
# 2011-09-26
# o Added methods set- and getCommentChar() to TabularTextFile and
#   argument 'commentChar' to its constructor.  This allows to use
#   custom comment characters other than just "#".
# 2011-07-13
# o GENERALIZATION: Now readDataFrame() of TabularTextFile can read
#   numeric columns that are quoted and for which 'colClasses' in non-NA.
#   This is done by first reading the as quoted character strings, and
#   dropping the quotes, and then rereading them as numeric values.
# 2011-05-12
# o Added more verbose output to readDataFrame().
# 2011-03-14
# o Improved the verbose output of getReadArguments().
# 2010-08-14
# o Added Rdoc comments.
# 2010-04-22
# o Added "NA" to the default 'na.strings' returned by getReadArguments()
#   for TabularTextFile.
# 2009-10-06
# o Added subsetting via [() to TabularTextFile.
# 2009-05-09
# o Added argument 'translate' to getColumnNames() of TabularTextFile.
# 2008-07-23
# o Now nbrOfLines() cache the results and only recount if the file has
#   been modified since last time (or the file system does not provide
#   information on last modification time).  It also uses the new 
#   countLines() in R.utils.
# 2008-07-22
# o Added ad hoc dimension() to speed up ll(). It may return NA for the
#   number of rows.
# 2008-06-12
# o Added readRawHeader().  Removed readHeader() [now in getHeader()].
# o Added hasColumnHeader() to TabularTextFile.
# 2008-05-16
# o Took the text-file based parts of GenericTabularFile and placed them
#   in new subclass TabularTextFile.
# 2008-05-12
# o Added extractMatrix().
# o BUG FIX: getReadArguments() did not infer column classes if there was
#   no header to read but the column names was manually set.
# o BUG FIX: readDataFrame() did not read the first data row if there was
#   no column header; it was eaten up by a preceeding readHeader().
# 2008-04-29
# o Added readLines(), nbrOfLines(), nbrOfRows() and dim().
# o Now readDataFrame() keeps the row names if arguments rows != NULL.
# 2008-04-25
# o Now argument 'verbose' of the constructor is passed to verfity().
# 2008-04-24
# o Added argument 'rows' to readDataFrame() for TabularTextFile.
# 2008-04-14
# o Renamed readData() to readDataFrame() for TabularTextFile.
# 2008-03-22
# o Added {get|set}ColumnNameTranslator().
# 2008-03-18
# o Now any '...' arguments to getReadArguments() override the inferred 
#   read arguments, e.g. na.strings="NA".
# 2008-02-27
# o Since 'affy' defines standardGeneric("colnames") and because S3 methods
#   are not found by such S4 generic functions, we avoid that method name,
#   and instead use getColumnNames().
# 2007-09-16
# o Removed all 'camelCaseNames' arguments.  Now column names are decided 
#   by getColumnNames() and translateColumnNames(), which can be overridden.
# 2007-09-14
# o Extracted from AffymetrixTabularFile.
# 2007-09-10
# o Created from AffymetrixCsvGenomeInformation.R.
############################################################################
