% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redcap-write.R
\name{redcap_write}
\alias{redcap_write}
\title{Write/Import records to a REDCap project.}
\usage{
redcap_write(ds_to_write, batch_size = 100L, interbatch_delay = 0.5,
  continue_on_error = FALSE, redcap_uri, token, verbose = TRUE,
  config_options = NULL)
}
\arguments{
\item{ds_to_write}{The \code{\link[base:data.frame]{base::data.frame()}} to be imported into the REDCap project.  Required.}

\item{batch_size}{The maximum number of subject records a single batch should contain.  The default is 100.}

\item{interbatch_delay}{The number of seconds the function will wait before requesting a new subset from REDCap. The default is 0.5 seconds.}

\item{continue_on_error}{If an error occurs while writing, should records in subsequent batches be attempted.  The default is \code{FALSE}, which prevents subsequent batches from running.  Required.}

\item{redcap_uri}{The URI (uniform resource identifier) of the REDCap project.  Required.}

\item{token}{The user-specific string that serves as the password for a project.  Required.}

\item{verbose}{A boolean value indicating if \code{message}s should be printed to the R console during the operation.  The verbose output might contain sensitive information (\emph{e.g.} PHI), so turn this off if the output might be visible somewhere public. Optional.}

\item{config_options}{A list of options to pass to \code{POST} method in the \code{httr} package.  See the details in \code{\link[=redcap_read_oneshot]{redcap_read_oneshot()}}. Optional.}
}
\value{
Currently, a list is returned with the following elements,
\itemize{
\item \code{success}: A boolean value indicating if the operation was apparently successful.
\item \code{status_code}: The \href{http://en.wikipedia.org/wiki/List_of_HTTP_status_codes}{http status code} of the operation.
\item \code{outcome_message}: A human readable string indicating the operation's outcome.
\item \code{records_affected_count}: The number of records inserted or updated.
\item \code{affected_ids}: The subject IDs of the inserted or updated records.
\item \code{elapsed_seconds}: The duration of the function.
}
}
\description{
This function uses REDCap's APIs to select and return data.
}
\details{
Currently, the function doesn't modify any variable types to conform to REDCap's supported variables.
See \code{\link[=validate_for_write]{validate_for_write()}} for a helper function that checks for some common important conflicts.

For \code{redcap_write} to function properly, the user must have Export permissions for the
'Full Data Set'.  Users with only 'De-Identified' export privileges can still use
\code{\link[=redcap_write_oneshot]{redcap_write_oneshot()}}.  To grant the appropriate permissions:
\itemize{
\item go to 'User Rights' in the REDCap project site,
\item select the desired user, and then select 'Edit User Privileges',
\item in the 'Data Exports' radio buttons, select 'Full Data Set'.
}
}
\examples{
\dontrun{
#Define some constants
uri           <- "https://bbmc.ouhsc.edu/redcap/api/"
token         <- "D70F9ACD1EDD6F151C6EA78683944E98"

# Read the dataset for the first time.
result_read1  <- redcap_read_oneshot(redcap_uri=uri, token=token)
ds1           <- result_read1$data
ds1$telephone
# The line above returns something like this (depending on its previous state).
# [1] "(432) 456-4848" "(234) 234-2343" "(433) 435-9865" "(987) 654-3210" "(333) 333-4444"

# Manipulate a field in the dataset in a VALID way
ds1$telephone <- sprintf("(405) 321-\%1$i\%1$i\%1$i\%1$i", seq_len(nrow(ds1)))

ds1 <- ds1[1:3, ]
ds1$age       <- NULL; ds1$bmi <- NULL #Drop the calculated fields before writing.
result_write  <- REDCapR::redcap_write(ds=ds1, redcap_uri=uri, token=token)

# Read the dataset for the second time.
result_read2  <- redcap_read_oneshot(redcap_uri=uri, token=token)
ds2           <- result_read2$data
ds2$telephone
# The line above returns something like this.  Notice only the first three lines changed.
# [1] "(405) 321-1111" "(405) 321-2222" "(405) 321-3333" "(987) 654-3210" "(333) 333-4444"

# Manipulate a field in the dataset in an INVALID way.  A US exchange can't be '111'.
ds1$telephone <- sprintf("(405) 111-\%1$i\%1$i\%1$i\%1$i", seq_len(nrow(ds1)))

# This next line will throw an error.
result_write <- REDCapR::redcap_write(ds=ds1, redcap_uri=uri, token=token)
result_write$raw_text
}
}
\references{
The official documentation can be found on the 'API Help Page' and 'API Examples' pages
on the REDCap wiki (ie, https://community.projectredcap.org/articles/456/api-documentation.html and
https://community.projectredcap.org/articles/462/api-examples.html). If you do not have an account
for the wiki, please ask your campus REDCap administrator to send you the static material.
}
\author{
Will Beasley
}
