
// RQuantLib -- R interface to the QuantLib libraries
//
// Copyright 2002, 2003, 2004 Dirk Eddelbuettel <edd@debian.org>
//
// $Id: implieds.cc,v 1.3 2004/04/06 03:34:16 edd Exp $
//
// This file is part of the RQuantLib library for GNU R.
// It is made available under the terms of the GNU General Public
// License, version 2, or at your option, any later version,
// incorporated herein by reference.
//
// This program is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE.  See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA

// NB can be build standalone as   PKG_LIBS=-lQuantLib R CMD SHLIB implieds.cc

#include <ql/quantlib.hpp>	// make QuantLib known

#include <ql/Instruments/vanillaoption.hpp>
#include <ql/TermStructures/flatforward.hpp>
#include <ql/Volatilities/blackconstantvol.hpp>
#include <ql/Calendars/target.hpp>

using namespace QuantLib;

extern "C" {

#include "rquantlib.h"

  SEXP QL_EuropeanOptionImpliedVolatility(SEXP optionParameters) {
    const Size maxEvaluations = 100;
    const double tolerance = 1.0e-6;
    const int nret = 2;		// dimension of return list
    char *type = CHAR(STRING_ELT(getListElement(optionParameters, "type"),0));
    Option::Type optionType;
    if (!strcmp(type, "call")) {
      optionType = Option::Call;
    } else if (!strcmp(type, "put")) {
      optionType = Option::Put;
    } else if (!strcmp(type, "straddle")) {
      optionType = Option::Straddle;
    } else {
      error("Unexpected option type %s, aborting\n", type);
    }

    double strike = REAL(getListElement(optionParameters,"strike"))[0];	
    
    Time maturity = REAL(getListElement(optionParameters, "maturity"))[0];
    int length = int(maturity * 360); // FIXME: this could be better

    Date today = Date::todaysDate();

    // new framework as per QuantLib 0.3.5
    DayCounter dc = Actual360();
    Handle<SimpleQuote> spot(new SimpleQuote(0.0));
    Handle<SimpleQuote> vol(new SimpleQuote(0.0));
    Handle<BlackVolTermStructure> volTS = makeFlatVolatility(vol,dc);
    Handle<SimpleQuote> qRate(new SimpleQuote(0.0));
    Handle<TermStructure> qTS = makeFlatCurve(qRate, dc);
    Handle<SimpleQuote> rRate(new SimpleQuote(0.0));
    Handle<TermStructure> rTS = makeFlatCurve(rRate, dc);

    Date exDate = today.plusDays(length);
    Handle<Exercise> exercise(new EuropeanExercise(exDate));
    Handle<StrikedTypePayoff> 
      payoff(new PlainVanillaPayoff(optionType, strike));
    Handle<VanillaOption> option = makeOption(payoff, exercise, spot,
					      qTS, rTS, volTS);

    spot->setValue(REAL(getListElement(optionParameters, "underlying"))[0]);
    qRate->setValue(REAL(getListElement(optionParameters, 
					"dividendYield"))[0]);
    rRate->setValue(REAL(getListElement(optionParameters,
					"dividendYield"))[0]);
    double volguess = REAL(getListElement(optionParameters, "volatility"))[0];
    vol->setValue(volguess);

    double value = option->NPV();
    double implVol = 0.0; // just to remove a warning...
    if (value != 0.0) {
      vol->setValue(volguess*1.5);	// shift guess somehow
      implVol = option->impliedVolatility(value, tolerance, maxEvaluations);
    }

    SEXP rl = PROTECT(allocVector(VECSXP, nret)); // returned list
    SEXP nm = PROTECT(allocVector(STRSXP, nret)); // names of list elements
    insertListElement(rl, nm, 0, implVol, "impliedVol");
    SET_VECTOR_ELT(rl, 1, optionParameters);
    SET_STRING_ELT(nm, 1, mkChar("parameters"));
    setAttrib(rl, R_NamesSymbol, nm);
    UNPROTECT(2);
    return(rl);
  }

  SEXP QL_AmericanOptionImpliedVolatility(SEXP optionParameters) {
    const Size maxEvaluations = 100;
    const double tolerance = 1.0e-6;
    const int nret = 2;		// dimension of return list
    char *type = CHAR(STRING_ELT(getListElement(optionParameters, "type"),0));
    Option::Type optionType;
    if (!strcmp(type, "call")) {
      optionType = Option::Call;
    } else if (!strcmp(type, "put")) {
      optionType = Option::Put;
    } else if (!strcmp(type, "straddle")) {
      optionType = Option::Straddle;
    } else {
      error("Unexpected option type %s, aborting\n", type);
    }

    double strike = REAL(getListElement(optionParameters,"strike"))[0];	
    
    Time maturity = REAL(getListElement(optionParameters, "maturity"))[0];
    int length = int(maturity * 360); // FIXME: this could be better

    Date today = Date::todaysDate();

    // new framework as per QuantLib 0.3.5
    DayCounter dc = Actual360();
    Handle<SimpleQuote> spot(new SimpleQuote(0.0));
    Handle<SimpleQuote> vol(new SimpleQuote(0.0));
    Handle<BlackVolTermStructure> volTS = makeFlatVolatility(vol,dc);
    Handle<SimpleQuote> qRate(new SimpleQuote(0.0));
    Handle<TermStructure> qTS = makeFlatCurve(qRate, dc);
    Handle<SimpleQuote> rRate(new SimpleQuote(0.0));
    Handle<TermStructure> rTS = makeFlatCurve(rRate, dc);

    Date exDate = today.plusDays(length);
    //Handle<Exercise> exercise(new EuropeanExercise(exDate));
    Handle<Exercise> exercise(new AmericanExercise(today, exDate));
    Handle<StrikedTypePayoff> 
      payoff(new PlainVanillaPayoff(optionType, strike));
    Handle<VanillaOption> option = makeOption(payoff, exercise, spot,
					      qTS, rTS, volTS,
					      JR);

    spot->setValue(REAL(getListElement(optionParameters, "underlying"))[0]);
    qRate->setValue(REAL(getListElement(optionParameters, 
					"dividendYield"))[0]);
    rRate->setValue(REAL(getListElement(optionParameters,
					"dividendYield"))[0]);
    double volguess = REAL(getListElement(optionParameters, "volatility"))[0];
    vol->setValue(volguess);

    double value = option->NPV();
    double implVol = 0.0; // just to remove a warning...
    if (value != 0.0) {
      vol->setValue(volguess*1.5);	// shift guess somehow
      implVol = option->impliedVolatility(value, tolerance, maxEvaluations);
    }

    SEXP rl = PROTECT(allocVector(VECSXP, nret)); // returned list
    SEXP nm = PROTECT(allocVector(STRSXP, nret)); // names of list elements
    insertListElement(rl, nm, 0, implVol, "impliedVol");
    SET_VECTOR_ELT(rl, 1, optionParameters);
    SET_STRING_ELT(nm, 1, mkChar("parameters"));
    setAttrib(rl, R_NamesSymbol, nm);
    UNPROTECT(2);
    return(rl);
  }
 
}
