#' @title Mapping rate and direction of phenotypic change on 3D surfaces.
#' @description Given vectors of RW (or PC) scores, the function selects the RW
#'   (PC) axes linked to highest (and lowest) evolutionary rate values and
#'   reconstructs the morphology weighted on them. In this way, \code{rate.map}
#'   shows where and how the phenotype changed the most between any pair of
#'   taxa.
#' @usage rate.map(x, RR, scores, pcs, mshape, mshape_sur=NULL,refsur=NULL,
#'   refmat=NULL, k=4,out.rem = TRUE, plot=TRUE, pal=NULL,
#'   NAcol="gray90",from=NULL, to=NULL,show.names=TRUE)
#' @param x the species/nodes to be compared; it can be a single species, or a
#'   vector containing two species or a species and any of its parental nodes.
#' @param RR an object generated by using the \code{\link[RRphylo]{RRphylo}}
#'   function.
#' @param scores RW or PC scores.
#' @param pcs RW (or PC) vectors (eigenvectors of the covariance matrix)
#'   returned by RWA/PCA.
#' @param mshape the consensus configuration.
#' @param mshape_sur a \code{mesh3d} object used as a reference for mesh
#'   reconstruction. The vertices of \code{mshape_sur} must be the consensus
#'   configuration. If \code{NULL}, it is automatically generated by applying
#'   \code{\link[Rvcg]{vcgBallPivoting}} on \code{mshape}.
#' @param refsur a list of \code{mesh3d} to be provided for all species in
#'   \code{x}.
#' @param refmat a list of landmark sets to be provided for all species in
#'   \code{x}.
#' @param k the argument \code{k} passed to \code{\link{interpolMesh}}.
#' @param out.rem logical: if \code{TRUE} mesh triangles with outlying area
#'   difference are removed.
#' @param plot logical indicating if the 3d plot must be shown.
#' @param pal a vector of colors to be passed to
#'   \code{\link[grDevices]{colorRampPalette}}.
#' @param NAcol the argument \code{NAcol} passed to \code{\link{col2mesh}}.
#' @param from,to lower and upper limits to be associated to the ends of
#'   \code{pal}.
#' @param show.names logical: if \code{TRUE}, the names of the species are
#'   displayed in the 3d plot.
#' @details After selecting the RW (PC) axes, \code{rate.map} automatically
#'   builds a 3D mesh on the mean shape calculated from the Relative Warp
#'   Analysis (RWA) or Principal Component Analysis (PCA) (\cite{Schlager 2017})
#'   by applying the function \code{\link[Rvcg]{vcgBallPivoting}} (\pkg{Rvcg}).
#'   The reconstruction of species 3d surfaces is based on \code{mshape_sur},
#'   either provided by the user or generated within the function. Finally, for
#'   each species in \code{x}, the function computes the area differences
#'   between corresponding triangles of its reconstructed 3D mesh and the
#'   surface of the ancestor (most recent common ancestor in the case of two
#'   species in \code{x}). In the calculation of differences, the possibility to
#'   find and remove outliers is supplied (\code{out.rem=TRUE}, we suggest
#'   considering this possibility if the mesh may contain degenerate facets).
#'
#'   Finally, \code{rate.map} returns a 3D plot showing such comparisons
#'   displayed on shape of the ancestor/mrca used as the reference. The color
#'   gradient goes from blue to red, where blue areas represent expansion of the
#'   mesh, while the red areas represent contraction of the mesh triangles. If a
#'   list \code{refsur} (and \code{refmat}) is provided, convergence is plotted
#'   onto them (see \code{\link{interpolMesh}} for further details).
#' @export
#' @seealso
#'   \href{https://CRAN.R-project.org/package=RRphylo/vignettes/RRphylo.html}{\code{RRphylo}
#'   vignette} ; \code{\link[Morpho]{relWarps}} ; \code{\link[Morpho]{procSym}}
#' @importFrom rgl open3d mtext3d shade3d next3d mfrow3d
#' @importFrom RRphylo getMommy
#' @return The function returns a list including:
#'   \itemize{\item\strong{$selected} a list of RWs/PCs axes selected for higher
#'   evolutionary rates for each species. \item\strong{$surfaces} a list of
#'   reconstructed colored surfaces of the given species/node and the most recent
#'   common ancestor.\item\strong{differences} a list area differences
#'   between corresponding triangles of species reconstructed 3d mesh and the
#'   surface of the ancestor. \item\strong{lmks} if \code{refmat}
#'   is not \code{NULL}, this is the landmark configuration rotated on the
#'   reconstructed surface .}
#' @author Marina Melchionna, Antonio Profico, Silvia Castiglione, Gabriele
#'   Sansalone, Pasquale Raia
#' @references Schlager, S. (2017). \emph{Morpho and Rvcg-Shape Analysis in R:
#'   R-Packages for geometric morphometrics, shape analysis and surface
#'   manipulations.} In: Statistical shape and deformation analysis. Academic
#'   Press.
#' @references Castiglione, S., Melchionna, M., Profico, A., Sansalone, G.,
#'   Modafferi, M., Mondanaro, A., Wroe, S., Piras, P., & Raia, P. (2021). Human
#'   face-off: a new method for mapping evolutionary rates on three-dimensional
#'   digital models. \emph{Palaeontology}, 65, 1. doi:10.1111/pala.12582
#' @references Melchionna, M., Castiglione, S., Girardi, G., Serio, C.,
#'   Esposito, A., Mondanaro, A., Profico, A., Sansalone, G., & Raia, P. (2024).
#'   RRmorph—a new R packageto map phenotypic evolutionary rates and patterns on
#'   3D meshes. \emph{Communications Biology}, 7, 1009.
#' @examples
#'   \donttest{
#'   da<-"https://github.com/pasraia/RRmorph_example_data/raw/refs/heads/main/RRmorphdata.rda"
#'   download.file(url=da,destfile = paste0(tempdir(),"/RRmorphdata.rda"))
#'   load(paste0(tempdir(),"/RRmorphdata.rda"))
#'
#'   require(Morpho)
#'   require(Rvcg)
#'
#'   pca<-procSym(endo.set)
#'   ldm_homo<-endo.set[,,"Homo_sapiens"]
#'   sur_homo<-endo.sur[["Homo_sapiens"]]
#'   ldm_macaca<-endo.set[,,"Macaca_fuscata"]
#'   sur_macaca<-endo.sur[["Macaca_fuscata"]]
#'
#'   cc<- 2/parallel::detectCores()
#'   RR<-RRphylo::RRphylo(tree.prima,pca$PCscores,clus=cc)
#'
#'   # plotting on reconstructed surfaces
#'   Rmap1<-rate.map(x=c("Homo_sapiens","Macaca_fuscata"),RR=RR, scores=pca$PCscores,
#'                   pcs=pca$PCs, mshape=pca$mshape)
#'
#'   # plotting on real surfaces
#'   Rmap2<-rate.map(x=c("Homo_sapiens","Macaca_fuscata"),RR=RR, scores=pca$PCscores,
#'                   pcs=pca$PCs, mshape=pca$mshape,
#'                   refsur=list("Homo_sapiens"=sur_homo,"Macaca_fuscata"=sur_macaca),
#'                   refmat=list("Homo_sapiens"=ldm_homo,"Macaca_fuscata"=ldm_macaca))
#'   }

rate.map<-function(x, RR, scores, pcs, mshape,mshape_sur=NULL,
                   refsur=NULL, refmat=NULL,k=4,out.rem = TRUE,
                   plot=TRUE, pal=NULL, NAcol="gray90",
                   from=NULL,to=NULL,show.names=TRUE) {

  misspacks<-sapply(c("inflection","ddpcr","ape"),requireNamespace,quietly=TRUE)
  if(any(!misspacks)){
    stop("The following package/s are needed for this function to work, please install it/them:\n ",
         paste(names(misspacks)[which(!misspacks)],collapse=", "),
         call. = FALSE)
  }

  rates<-RR$multiple.rates
  tree<-RR$tree
  aces<-RR$aces
  phen<-rbind(aces,scores)

  if(any(!is.null(c(refsur,refmat)))&!all(!is.null(c(refsur,refmat))))
    stop("Please, provide both surface and its landmark configuration")
  if(!is.null(refsur)){
    if (!inherits(refsur, "list")|!inherits(refmat, "list")) stop("refsur and refmat must be lists")
    if (length(refsur) != length(refmat))
      stop("The objects refsur and refmat should have the same length")
    if (is.null(names(refsur))|is.null(names(refmat))) stop("refsur and/or refmat are missing names")

    if(all(x%in%tree$tip.label)) {
      if(any(c(length(refsur),length(refmat))!=length(x)))
        stop("The objects refsur and refmat should be provided for all the species in x")
    }

    refsur<-refsur[match(x,names(refsur),nomatch = 0)]
    refmat<-refmat[match(x,names(refmat),nomatch = 0)]
  }
  if(!inherits(scores,"matrix")) scores<-as.matrix(scores)

  if(is.null(pal)) pal<-c("darkred","red","orange","white","lightblue","blue","darkblue")

  if(all(x%in%tree$tip.label)) {
    if(length(x)==1){
      mrca<-getMommy(tree,which(tree$tip.label==x))[1]
      rates_sum<-rates[match(x,rownames(rates)),,drop=FALSE]
    } else {
      mrca<-ape::getMRCA(tree,x)
      path1<-c(x[1],getMommy(tree,which(tree$tip.label==x[1])))
      path2<-c(x[2],getMommy(tree,which(tree$tip.label==x[2])))
      rates1<-apply(rates[match(path1[1:(which(path1==mrca)-1)],rownames(rates)),,drop=FALSE],2,sum)
      rates2<-apply(rates[match(path2[1:(which(path2==mrca)-1)],rownames(rates)),,drop=FALSE],2,sum)
      rates_sum<-rbind(rates1,rates2)
    }
  } else {
    mrca<-x[-which(x%in%tree$tip.label)]
    x<-x[which(x%in%tree$tip.label)]
    path1<-c(x,getMommy(tree,which(tree$tip.label==x)))
    if(!mrca%in%path1) stop("the node is not along the species path")
    rates1<-apply(rates[match(path1[1:(which(path1==mrca)-1)],rownames(rates)),,drop=FALSE],2,sum)
    rates_sum<-t(as.matrix(rates1))
  }

  if(is.null(mshape_sur))  mshape_sur <- vcgBallPivoting(mshape, radius = 0)

  ace_vec<-aces[match(mrca,rownames(aces)),]
  temp_ace <- restoreShapes(ace_vec, pcs, mshape)
  sur.ref <- mshape_sur
  sur.ref$vb[1:3,] <- t(temp_ace)

  sele<-sur<-rates.list<-darea<-list()
  for(i in 1:nrow(rates_sum)){
    cutter <- inflection::ede(seq(1:dim(rates_sum)[2]),rates_sum[i,order(rates_sum[i,],decreasing = TRUE)], 0)

    if (cutter[1] == 1 & cutter[2] == dim(rates_sum)[2]) {
      cutter2 <- inflection::ede(seq(1:(dim(rates_sum)[2]-2)),
                                 rates_sum[i,order(rates_sum[i,],decreasing = TRUE)][-c(1,dim(rates_sum)[2])], 0)
      cutter[1:2] <- cutter2[1:2] + 1
    } else if (cutter[1] == 1) {
      cutter2 <- inflection::ede(seq(1:(dim(rates_sum)[2]-1)),
                                 rates_sum[i,order(rates_sum[i,],decreasing = TRUE)][-1], 0)
      cutter[1] <- cutter2[1] + 1
    } else if (cutter[2] == dim(rates_sum)[2]) {
      cutter2 <- inflection::ede(seq(1:(dim(rates_sum)[2]-1)),
                                 rates_sum[i,order(rates_sum[i,],decreasing = TRUE)][-dim(rates_sum)[2]], 0)
      cutter[2] <- cutter2[2]
    }

    up=seq(1:cutter[1])
    dw=seq(cutter[2],dim(rates_sum)[2])

    while ((length(up)+length(dw)) > 0.5 * dim(rates_sum)[2]){
      if (length(up)>=length(dw)) up[-length(up)]->up else dw[-1]->dw
    }

    rates.up<-rates_sum[i,order(rates_sum[i,],decreasing = TRUE)][up]
    rates.dw<-rates_sum[i,order(rates_sum[i,],decreasing = TRUE)][dw]

    sele[[i]]<-match(c(names(rates.up),names(rates.dw)),colnames(rates_sum))
    names(sele[[i]])<-c(names(rates.up),names(rates.dw))

    vec1<-phen[match(as.character(x[i]),rownames(phen)),]
    temp <- restoreShapes(vec1[sele[[i]]], pcs[,sele[[i]]], mshape)
    sur[[i]] <- mshape_sur
    sur[[i]]$vb[1:3, ] <- t(temp)

    darea[[i]]<-areadiff(sur[[i]],sur.ref,out.rem = out.rem,scale01 = FALSE)$darea
    rates.list[[i]]<-c(rates.up,rates.dw)
  }

  names(sele)<-x
  names(sur)<-x
  names(rates.list)<-x

  if(!is.null(refsur)){
    for(i in 1:length(refsur)) {
      refsur[[i]]<-rotmesh.onto(refsur[[i]],refmat[[i]],vert2points(sur[[i]]),scale = TRUE)$mesh
      refmat[[i]]<-rotmesh.onto(refsur[[i]],refmat[[i]],vert2points(sur[[i]]),scale = TRUE)$yrot
    }
  }

  meshes<-list()
  values<-list()

  values<-lapply(1:length(sur),function(w){
    if(!is.null(refsur))
      interpolMesh(sur[[w]],darea[[w]],refsur[[w]],refmat[[w]],element="triangles",k=k) else
      tri2verts(sur[[w]],darea[[w]])
  })

  if(is.null(from)) from=min(unlist(values),na.rm=TRUE)
  if(is.null(to)) to=max(unlist(values),na.rm=TRUE)

  if(plot){
    open3d()
    if(length(sur)>1) mfrow3d(1,2,sharedMouse = TRUE)
  }

  for(w in 1:length(sur)){
    if(!is.null(refsur))
      meshes[[w]]<-col2mesh(refsur[[w]],values[[w]],pal,NAcol = NAcol,from=from,to=to) else
        meshes[[w]]<-col2mesh(sur[[w]],values[[w]],pal,NAcol=NAcol,from=from,to=to)

      if(plot){
        plotLegend(meshes[[w]],values[[w]],main=x[w])
        if(w==2) next3d()
        shade3d(meshes[[w]], specular = "black")
        if (show.names) mtext3d(text = paste(x[w]), font = 2,edge = "x", line = 3)
      }
  }
  names(meshes)<-names(values)<-x

  return(list(selected=rates.list,surfaces=c(mrca=list(sur.ref),meshes),lmks=refmat,differences=values))
}

