#############################################################################
#
#   This file is part of the R package "RSNNS".
#
#   Author: Christoph Bergmeir
#   Supervisor: José M. Benítez
#   Copyright (c) DiCITS Lab, Sci2s group, DECSAI, University of Granada.
#
#   This library is free software; you can redistribute it and/or
#   modify it under the terms of the GNU Library General Public
#   License as published by the Free Software Foundation; either
#   version 2 of the License, or (at your option) any later version.
# 
#   This library is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   Library General Public License for more details.
# 
#   You should have received a copy of the GNU Library General Public License
#   along with this library; see the file COPYING.LIB.  If not, write to
#   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#   Boston, MA 02110-1301, USA.
#
#############################################################################

#' The input matrix is column-wise normalized. 
#' 
#' The parameter \code{type} specifies, how normalization takes place:
#' \describe{
#' \item{0_1}{values are normalized to the [0,1]-interval. The minimum in the data is mapped to zero, the maximum to one.}
#' \item{center}{the data is centered, i.e. the mean is substracted}
#' \item{norm}{the data is normalized to mean zero, variance one}
#' }
#' 
#' @title Data normalization
#' @param x input data
#' @param type 
#' \describe{
#' \item{either}{type string specifying the type of normalization. Implemented are "0_1", "center", and "norm"}
#' \item{or}{attribute list of a former call to this method to apply e.g. normalization of the training data to the test data}
#' }
#' @return column-wise normalized input. The normalization parameters that were used for the normalization are present as attributes
#' of the output. They can be obtained with \code{\link{getNormParameters}}.
#' @seealso \code{\link{denormalizeData}}, \code{\link{getNormParameters}}
#' @export
normalizeData <- function(x, type="norm") {
  
  if(!is.list(type))
    type <- computeNormalizationParameters(x, type)

  res <- normalizeDataWithParams(x, type, norm=TRUE) 
  
  res
}

#' Column-wise normalization of the input matrix is reverted, using the given parameters.
#' 
#' The input matrix is column-wise denormalized using the parameters given by \code{normParams}. 
#' E.g., if \code{normParams} contains mean and sd for every column, the values are multiplied 
#' by sd and the mean is added
#' 
#' @title Revert data normalization
#' @param x input data
#' @param normParams the parameters generated by an earlier call to \code{\link{normalizeData}} that will be used for reverting normalization
#' @return column-wise denormalized input
#' @seealso \code{\link{normalizeData}}, \code{\link{getNormParameters}}
#' @export
#' @examples 
#' data(iris)
#' values <- normalizeData(iris[,1:4])
#' denormalizeData(values, getNormParameters(values))
denormalizeData <- function(x, normParams)
  normalizeDataWithParams(x, normParams, norm=FALSE) 


#' Get the normalization parameters that are appended by \code{\link{normalizeData}}
#' as attributes to the input data. 
#' 
#' This function is equivalent to calling \code{attr(x, "normParams")}.
#' 
#' @title Get normalization parameters of the input data
#' @param x input data
#' @return the parameters generated by an earlier call to \code{\link{normalizeData}}
#' @seealso \code{\link{normalizeData}}, \code{\link{denormalizeData}} 
#' @export
getNormParameters <- function(x)
  attr(x, "normParams")

computeNormalizationParameters <- function(x, type="norm") {
    
    x <- as.matrix(x)
    
    params <- list()
    
    if(type == "0_1") {
      
      colMaxima <- NULL
      colMinima <- NULL
      
      for(i in 1:ncol(x)) {
       
        colMaxima <- c(colMaxima, max(x[,i]))
        colMinima <- c(colMinima, min(x[,i]))
        
      }
      
      params$colMaxima <- colMaxima
      params$colMinima <- colMinima

    } else if(type== "center") {

      colMeans <- NULL
      
      for(i in 1:ncol(x)) {
        
        colMeans <- c(colMeans, mean(x[,i]))
        
      }
      
      params$colMeans <- colMeans
      
    } else {
      
      type="norm"

      colMeans <- NULL
      colSds <- NULL
            
      for(i in 1:ncol(x)) {

        colMeans <- c(colMeans, mean(x[,i]))
        colSds <- c(colSds, sd(x[,i]))
        
      }
      
      params$colMeans <- colMeans
      params$colSds <- colSds

    }
    
    params$type <- type
    params
}  


normalizeDataWithParams <- function(x, normParams, norm=TRUE) {
  
  x <- as.matrix(x)
  
  res <- NULL
  
  type <- normParams$type
  
  if(type == "0_1") {

    for(i in 1:ncol(x)) {
      
      colMax <- normParams$colMaxima[i]
      colMin <- normParams$colMinima[i]
      
      if(norm) {
        if((colMax - colMin) != 0)
          res <- cbind(res, (x[,i] - colMin) / (colMax - colMin))
        else 
          res <- cbind(res, (x[,i]))
      } else
        res <- cbind(res, x[,i] * (colMax - colMin) + colMin)
      
    }    
    
  } else if(type== "center") {
    
    for(i in 1:ncol(x)) {
      
      colMean <- normParams$colMeans[i]
      
      if(norm)
        res <- cbind(res, x[,i] - colMean)
      else  
        res <- cbind(res, x[,i] + colMean)
      
    }
    
  } else if(type== "norm") {
    
    for(i in 1:ncol(x)) {
      
      colMean <- normParams$colMeans[i]
      colSd <- normParams$colSds[i]
      
      if(norm) {
        if(colSd != 0)
          res <- cbind(res, (x[,i] - colMean) / colSd )
        else
          res <- cbind(res, (x[,i] - colMean))
      } else
        res <- cbind(res, x[,i] * colSd + colMean)
    }
    
  }
  
  attr(res, "normParams") <- normParams
  res
  
}
