% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grabMzxmlFunctions.R
\name{grabMzxmlData}
\alias{grabMzxmlData}
\title{Get mass-spectrometry data from an mzXML file}
\usage{
grabMzxmlData(
  filename,
  grab_what,
  verbosity = 0,
  rtrange = NULL,
  mz = NULL,
  ppm = NULL
)
}
\arguments{
\item{filename}{A single filename to read into R's memory. Both absolute and
relative paths are acceptable.}

\item{grab_what}{What data should be read from the file? Options include
"MS1" for data only from the first spectrometer, "MS2" for fragmentation
data, "BPC" for rapid access to the base peak chromatogram, and "TIC" for
rapid access to the total ion chromatogram. These options can be combined
(i.e. `grab_data=c("MS1", "MS2", "BPC")`) or this argument can be set to
"everything" to extract all of the above. Option "EIC" is useful when
working with files whose total size exceeds working memory - it first
extracts all relevant MS1 and MS2 data, then discards data outside of the
mass range(s) calculated from the provided mz and ppm.}

\item{verbosity}{Three levels of processing output to the R console are
available, with increasing verbosity corresponding to higher integers. A
verbosity of zero means that no output will be produced, useful when
wrapping within larger functions. A verbosity of 1 will produce a progress
bar using base R's txtProgressBar function. A verbosity of 2 or higher will
produce timing output for each individual file read in.}

\item{rtrange}{Not supported for mzXML data. Only provided here so as to
throw a friendly warning rather than an unexpected error.}

\item{mz}{A vector of the mass-to-charge ratio for compounds of interest.
Only used when combined with `grab_what = "EIC"` (see above). Multiple
masses can be provided.}

\item{ppm}{A single number corresponding to the mass accuracy (in parts per
million) of the instrument on which the data was collected. Only used when
combined with `grab_what = "EIC"` (see above).}
}
\value{
A list of `data.table`s, each named after the arguments requested in
  grab_what. $MS1 contains MS1 information, $MS2 contains fragmentation info,
  etc. MS1 data has three columns: retention time (rt), mass-to-charge (mz),
  and intensity (int). MS2 data has five: retention time (rt), precursor m/z
  (premz), fragment m/z (fragmz), fragment intensity (int), and collision
  energy (voltage). Data requested that does not exist in the provided files
  (such as MS2 data requested from MS1-only files) will return an empty
  (length zero) data.table.
}
\description{
This function handles the mzXML side of things, reading in files that are
written in the mzXML format. Much of the code is similar to the mzXML format,
but the xpath handles are different and the mz/int array is encoded
simultaneously rather than as two separate entries. This function has been
exposed to the user in case per-file optimization (such as peakpicking or
additional filtering) is desired before the full data object is returned.
}
\examples{
sample_file <- system.file("extdata", "LB12HL_AB.mzXML.gz", package = "RaMS")
file_data <- grabMzxmlData(sample_file, grab_what="MS1")
# Extract MS1 data and a base peak chromatogram
file_data <- grabMzxmlData(sample_file, grab_what=c("MS1", "BPC"))
# Extract EIC for a specific mass
file_data <- grabMzxmlData(sample_file, grab_what="EIC", mz=118.0865, ppm=5)
# Extract EIC for several masses simultaneously
file_data <- grabMzxmlData(sample_file, grab_what="EIC", ppm=5,
                           mz=c(118.0865, 146.118104, 189.123918))

# Extract MS2 data
sample_file <- system.file("extdata", "DDApos_2.mzXML.gz", package = "RaMS")
MS2_data <- grabMzxmlData(sample_file, grab_what="MS2")

}
