% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hMeanChiSq.R
\name{hMeanChiSq}
\alias{hMeanChiSq}
\alias{hMeanChiSqMu}
\alias{hMeanChiSqCI}
\title{harmonic mean chi-squared test}
\usage{
hMeanChiSq(
  z,
  w = rep(1, length(z)),
  alternative = c("greater", "less", "two.sided", "none"),
  bound = FALSE
)

hMeanChiSqMu(
  thetahat,
  se,
  w = rep(1, length(thetahat)),
  mu = 0,
  alternative = c("greater", "less", "two.sided", "none"),
  bound = FALSE
)

hMeanChiSqCI(
  thetahat,
  se,
  w = rep(1, length(thetahat)),
  alternative = c("two.sided", "greater", "less", "none"),
  level = 0.95,
  wGamma = rep(1, length(unique(thetahat)) - 1)
)
}
\arguments{
\item{z}{Numeric vector of z-values.}

\item{w}{Numeric vector of weights.}

\item{alternative}{Either "greater" (default), "less", "two.sided", or "none".
Specifies the alternative to be considered in the computation of the p-value.}

\item{bound}{If \code{FALSE} (default), p-values that cannot be computed are reported as \code{NaN}.
If \code{TRUE}, they are reported as "> bound".}

\item{thetahat}{Numeric vector of parameter estimates.}

\item{se}{Numeric vector of standard errors.}

\item{mu}{The null hypothesis value. Defaults to 0.}

\item{level}{Numeric vector specifying the level of the confidence interval. Defaults to 0.95.}

\item{wGamma}{Numeric vector of length \code{unique(thetahat) - 1} specifying weights used to
summarize the gamma values, i.e.,
the local minima of the p-value function between the thetahats. Defaults is a vector of 1s.}
}
\value{
\code{hMeanChiSq}: returns the p-values from the harmonic mean chi-squared test
based on the study-specific z-values.

\code{hMeanChiSqMu}: returns the p-value from the harmonic mean chi-squared test
based on study-specific estimates and standard errors.

\code{hMeanChiSqCI}: returns a list containing confidence interval(s)
obtained by inverting the harmonic mean chi-squared test based on study-specific
estimates and standard errors. The list contains:
\item{CI}{Confidence interval(s).}\cr\cr
If the \code{alternative} is "none", the list also contains:
\item{gamma}{Local minima of the p-value function between the thetahats.}
\item{gammaMean}{Mean of all gammas weighted by \code{wGamma}.}
\item{gammaHMean}{Harmonic mean of all gammas weighted by \code{wGamma}.}
}
\description{
p-values and confidence intervals from the harmonic mean chi-squared test.
}
\examples{
## Example from Fisher (1999) as discussed in Held (2020)
pvalues <- c(0.0245, 0.1305, 0.00025, 0.2575, 0.128)
lower <- c(0.04, 0.21, 0.12, 0.07, 0.41)
upper <- c(1.14, 1.54, 0.60, 3.75, 1.27)
se <- ci2se(lower = lower, upper = upper, ratio = TRUE)
thetahat <- ci2estimate(lower = lower, upper = upper, ratio = TRUE)

## hMeanChiSq() --------
hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           alternative = "less")
hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           alternative = "two.sided")
hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           alternative = "none")

hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           w = 1 / se^2, alternative = "less")
hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           w = 1 / se^2, alternative = "two.sided")
hMeanChiSq(z = p2z(p = pvalues, alternative = "less"),
           w = 1 / se^2, alternative = "none")


## hMeanChiSqMu() --------
hMeanChiSqMu(thetahat = thetahat, se = se, alternative = "two.sided")
hMeanChiSqMu(thetahat = thetahat, se = se, w = 1 / se^2,
             alternative = "two.sided")
hMeanChiSqMu(thetahat = thetahat, se = se, alternative = "two.sided",
             mu = -0.1)

## hMeanChiSqCI() --------
## two-sided
CI1 <- hMeanChiSqCI(thetahat = thetahat, se = se, w = 1 / se^2,
                    alternative = "two.sided")
CI2 <- hMeanChiSqCI(thetahat = thetahat, se = se, w = 1 / se^2,
                    alternative = "two.sided", level = 0.99875)
## one-sided
CI1b <- hMeanChiSqCI(thetahat = thetahat, se = se, w = 1 / se^2,
                     alternative = "less", level = 0.975)
CI2b <- hMeanChiSqCI(thetahat = thetahat, se = se, w = 1 / se^2,
                     alternative = "less", level = 1 - 0.025^2)

## confidence intervals on hazard ratio scale 
print(exp(CI1$CI), digits = 2)
print(exp(CI2$CI), digits = 2)
print(exp(CI1b$CI), digits = 2)
print(exp(CI2b$CI), digits = 2)


## example with confidence region consisting of disjunct intervals
thetahat2 <- c(-3.7, 2.1, 2.5)
se2 <- c(1.5, 2.2, 3.1)
level <- 0.95; alpha <- 1 - level
muSeq <- seq(-7, 6, length.out = 1000)
pValueSeq <- hMeanChiSqMu(thetahat = thetahat2, se = se2,
                          alternative = "none", mu = muSeq)
(hm <- hMeanChiSqCI(thetahat = thetahat2, se = se2, alternative = "none"))

plot(x = muSeq, y = pValueSeq, type = "l", panel.first = grid(lty = 1),
     xlab = expression(mu), ylab = "p-value")
abline(v = thetahat2, h = alpha, lty = 2)
arrows(x0 = hm$CI[, 1], x1 = hm$CI[, 2], y0 = alpha,
       y1 = alpha, col = "darkgreen", lwd = 3, angle = 90, code = 3)
points(hm$gamma, col = "red", pch = 19, cex = 2)

}
\references{
Held, L. (2020). The harmonic mean chi-squared test to substantiate scientific findings.
\emph{Journal of the Royal Statistical Society: Series C (Applied Statistics)}, \bold{69}, 697-708.
\doi{10.1111/rssc.12410}
}
\author{
Leonhard Held, Florian Gerber
}
