#' @title Add a plot into an html object
#'
#' @description
#' Add a plot into the \code{html} object.
#' 
#' @param doc Object of class \code{html} where paragraph has to be added
#' @param fun plot function
#' @param width plot width in inches (default value is 6).
#' @param height plot height in inches (default value is 6).
#' @param vector.graphic logical scalar, default to FALSE. If TRUE, vector graphics 
#' are produced instead of PNG images. If TRUE, vector graphics are 
#' RaphaelJS instructions(transformed as SVG). 
#' @param pointsize the default pointsize of plotted text in pixels, default to 12.
#' @param fontname the default font family to use, default to getOption("ReporteRs-default-font").
#' @param ... arguments for \code{fun}.
#' @return an object of class \code{"html"}.
#' @examples
#' #START_TAG_TEST
#' doc.dirname = "addPlot_example"
#' @example examples/html.R
#' @example examples/addPage.R
#' @example examples/addTitle1Level1.R
#' @example examples/addBasePlot_vg.R
#' @example examples/addTitle2Level1.R
#' @example examples/addggplot.R
#' @example examples/addTitle3Level1.R
#' @example examples/addLMPlot.R
#' @example examples/writeDoc_directory.R
#' @example examples/STOP_TAG_TEST.R
#' @seealso \code{\link{html}}, \code{\link{addPlot}}, \code{\link{add.plot.interactivity}}
#' @method addPlot html
#' @S3method addPlot html
addPlot.html = function(doc, fun, pointsize=getOption("ReporteRs-fontsize"), vector.graphic = T, width=6, height=6, fontname = getOption("ReporteRs-default-font"), ... ) {

	
	plotargs = list(...)

	dirname = tempfile( )
	dir.create( dirname )
	
	if( !vector.graphic ){
	
		filename = paste( dirname, "/plot%03d.png" ,sep = "" )
		grDevices::png (filename = filename
				, width = width, height = height, units = 'in'
				, pointsize = pointsize, res = 300
		)
		
		fun_res = try( fun(...), silent = T )
		dev.off()
		plotfiles = list.files( dirname , full.names = T )
		
		jimg = .jnew(class.html4r.ImagesList, as.integer( width*72.2 ), as.integer( height*72.2 ) )
		
		for( i in 1:length( plotfiles ) ){
			.tempfile = tempfile()
			base64::encode(plotfiles[i], .tempfile)
			.jcall( jimg, "V", "addImage", as.character(paste(readLines(.tempfile), collapse = "\n")) )
			unlink(.tempfile)
		}
		out = .jcall( doc$current_slide, "I", "add", jimg )
		if( out != 1 ){
			stop( "Problem while trying to add plot." )
		}
	} else {
		filename = file.path( dirname, "plot", fsep = "/" )
		env = raphael( file = filename,width=width*72.2
			, height = height*72.2
			, ps=pointsize, fontname = fontname
			, canvas_id = as.integer(doc$canvas_id) )
		fun(...)
		last_canvas_id = .C("get_current_canvas_id", (dev.cur()-1L), 0L)[[2]]
		dev.off()
		plot_ids = get("plot_ids", envir = env )
		
		if( last_canvas_id < 0 ) stop("unexpected error, could not find device information.")
		else doc$canvas_id = last_canvas_id;

		jimg = .jnew( class.html4r.RAPHAELGraphics )
		
		for(i in 1:length( plot_ids ) ){
			file = as.character(paste(readLines(plot_ids[[i]]$filename), collapse = "\n"))
			div.id = plot_ids[[i]]$div.id
			
			.jcall( jimg, "V", "registerGraphic", as.character(div.id), file )
		}
		out = .jcall( doc$current_slide, "I", "add", jimg )
		if( out != 1 ){
			stop( "Problem while trying to add plot." )
		}
		
	}
	doc
}



#' @title Add a plot into an html object
#'
#' @description
#' Add a plot into the \code{html} object.
#' 
#' @param fun plot function
#' @param width plot width in inches (default value is 6).
#' @param height plot height in inches (default value is 6).
#' @param pointsize the default pointsize of plotted text in pixels, default to 12.
#' @param fontname the default font family to use, default to getOption("ReporteRs-default-font").
#' @param canvas_id canvas id - an integer - unique id in the web page
#' @param ... arguments for \code{fun}.
#' @return an object of class \code{"html"}.
#' @examples
#' #START_TAG_TEST
#' @example examples/raphael.html.R
#' @example examples/STOP_TAG_TEST.R
#' @seealso \code{\link{html}}, \code{\link{addPlot}}, \code{\link{add.plot.interactivity}}
#' , \code{\link{addPlot.html}}
#' @export 
raphael.html = function( fun, pointsize=getOption("ReporteRs-fontsize")
	, width=6, height=6, fontname = getOption("ReporteRs-default-font")
	, canvas_id = 0
	, ... ) {
	
	plotargs = list(...)
	
	dirname = tempfile( )
	dir.create( dirname )
	
	filename = file.path( dirname, "plot", fsep = "/" )
	env = raphael( file = filename,width=width*72.2
			, height = height*72.2
			, ps=pointsize, fontname = fontname
			, canvas_id = as.integer(canvas_id) )
	fun(...)
	dev.off()
	plot_ids = get("plot_ids", envir = env )
	
	jimg = .jnew( class.html4r.RAPHAELGraphics )
	
	for(i in 1:length( plot_ids ) ){
		file = as.character(paste(readLines(plot_ids[[i]]$filename), collapse = "\n"))
		div.id = plot_ids[[i]]$div.id
		
		.jcall( jimg, "V", "registerGraphic", as.character(div.id), file )
	}
	
	out = .jcall( jimg, "S", "getHTML" )
	attr( out, "javascript" ) = .jcall( jimg, "S", "getJS" )
	attr( out, "div_id" ) = sapply( plot_ids, function(x) x$div.id )
	attr( out, "js_id" ) = sapply( plot_ids, function(x) x$js.plotid )
	out
}
