% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Require-package.R, R/Require.R
\docType{package}
\name{Require-package}
\alias{Require}
\alias{Require-package}
\title{Require: Installing and Loading R Packages for Reproducible Workflows}
\usage{
Require(
  packages,
  packageVersionFile,
  libPaths,
  install_githubArgs = list(),
  install.packagesArgs = list(),
  standAlone = getOption("Require.standAlone", FALSE),
  install = getOption("Require.install", TRUE),
  require = getOption("Require.require", TRUE),
  repos = getOption("repos"),
  purge = getOption("Require.purge", FALSE),
  verbose = getOption("Require.verbose", FALSE),
  ...
)
}
\arguments{
\item{packages}{Character vector of packages to install via
\code{install.packages}, then load (i.e., with \code{library}). If it is
one package, it can be unquoted (as in \code{require}). In the case of a
GitHub package, it will be assumed that the name of the repository is the
name of the package. If this is not the case, then pass a named character
vector here, where the names are the package names that could be different
than the GitHub repository name.}

\item{packageVersionFile}{If provided, then this will override all
\code{install.package} calls with \code{versions::install.versions}}

\item{libPaths}{The library path (or libraries) where all packages should be
installed, and looked for to load (i.e., call \code{library}). This can be
used to create isolated, stand alone package installations, if used with
\code{standAlone = TRUE}. Currently, the path supplied here will be
prepended to \code{.libPaths()} (temporarily during this call) to
\code{Require} if \code{standAlone = FALSE} or will set (temporarily)
\code{.libPaths()} to \verb{c(libPaths, tail(libPaths(), 1)} to keep base packages.}

\item{install_githubArgs}{List of optional named arguments, passed to \code{install.packages}
inside \code{installGitHubPackage}.}

\item{install.packagesArgs}{List of optional named arguments, passed to \code{install.packages}.}

\item{standAlone}{Logical. If \code{TRUE}, all packages will be installed to and loaded from
the \code{libPaths} only. NOTE: If \code{TRUE}, THIS WILL CHANGE THE USER'S \code{.libPaths()}, similar
to e.g., the \code{checkpoint} package.
If \code{FALSE}, then \code{libPath} will be prepended to \code{.libPaths()} during the \code{Require} call,
resulting in shared packages, i.e., it will include the user's default package folder(s).
This can be create dramatically faster installs if the user has a substantial number of
the packages already in their personal library.
Default \code{FALSE} to minimize package installing.}

\item{install}{Logical or "force". If \code{FALSE}, this will not try to install anything.
If \code{"force"}, then it will force installation of requested packages,
mimicking a call to e.g., \code{install.packages}.
If \code{TRUE}, the default, then this function will try to install any missing
packages or dependencies.}

\item{require}{Logical or character string. If \code{TRUE}, the default, then the function will
attempt to call \code{require} on all requested \code{packages}, possibly
after they are installed. If a character string, then it will only call \code{require}
on those specific packages (i.e., it will install the ones listed in \code{packages}, but
load the packages listed in \code{require})}

\item{repos}{The remote repository (e.g., a CRAN mirror), passed to either
\code{install.packages}, \code{install_github} or \code{installVersions}.}

\item{purge}{Logical. Should all caches be purged?
Default is \code{getOption("Require.purge", FALSE)}.
There is a lot of internal caching of results throughout the \code{Require} package.
These help with speed and reduce calls to internet sources.
However, sometimes these caches must be purged.
The cached values are renewed when found to be too old, with the age limit.
This maximum age can be set in seconds with the environment variable
\code{R_AVAILABLE_PACKAGES_CACHE_CONTROL_MAX_AGE}, or if unset,
defaults to 3600  (one hour -- see \code{\link[utils:available.packages]{utils::available.packages}}).

Internally, there are calls to \code{available.packages}.}

\item{verbose}{Numeric or logical indicating how verbose should the function be.
If -1 or less, then as little verbosity as possible.
If 0 or FALSE, then minimal outputs; if \code{1} or TRUE, more outputs; \code{2} even more.
NOTE: in \code{Require} function, when \code{verbose >= 2}, the return object will have an attribute:
\code{attr(.., "Require")} which has lots of information about the processes of the installs.}

\item{...}{Passed to \code{install.packages}.
Good candidates are e.g., \code{type} or \code{dependencies}. This can be
used with \code{install_githubArgs} or \code{install.packageArgs} which
give individual options for those 2 internal function calls.}
}
\description{
A single key function, 'Require' that makes rerun-tolerant versions of 'install.packages' and `require` for CRAN packages, packages no longer on CRAN (i.e., archived), specific versions of packages, and GitHub packages. This approach is developed to create reproducible workflows that are flexible and fast enough to use while in development stages, while able to build snapshots once a stable package collection is found. As with other functions in a reproducible workflow, this package emphasizes functions that return the same result whether it is the first or subsequent times running the function, with subsequent times being sufficiently fast that they can be run every time without undue waiting burden on the user or developer.

This is an "all in one" function that will run \code{install.packages} for CRAN and
GitHub \url{https://github.com/} packages and will install
specific versions of each package if versions are specified either via an (in)equality
(e.g., \code{"glue (>=1.6.2)"} or \code{"glue (==1.6.2)"} for an exact version) or with a
\code{packageVersionFile}.
If \code{require = TRUE}, the default, the function will then run \code{require} on all
named packages that satisfy their version requirements. If packages are already installed
(\code{packages} supplied), and their optional version numbers are satisfied,
then the "install" component will be skipped.
}
\details{
\code{standAlone} will either put the \code{Require}d packages and their
dependencies \emph{all} within the \code{libPaths} (if \code{TRUE}) or if
\code{FALSE} will only install packages and their dependencies that are
otherwise not installed in \code{.libPaths()[1]}, i.e., the current active
R package directory. Any packages or dependencies that are not yet installed will
be installed in \code{libPaths}.
}
\note{
For advanced use and diagnosis, the user can set \code{verbose = TRUE} or
\code{1} or \code{2} (or via \code{options("Require.verbose")}). This will
attach an attribute \code{attr(obj, "Require")} to the output of this
function.
}
\section{GitHub Package}{

Follows \code{remotes::install_github} standard.
As with \code{remotes::install_github}, it is not possible to specify a past
version of a GitHub package unless that version is a tag or the user passes
the SHA that had that package version. Similarly, if a developer does a
local install e.g., via \code{pkgload::install}, of an active project, this package
will not be able know of the GitHub state, and thus \code{pkgSnapshot} will not be able to
recover this state as there is no SHA associated with a local
installation. Use \code{Require} (or \code{remotes::install_github}) to create
a record of the GitHub state.
}

\section{Package Snapshots}{

To build a snapshot of the desired packages and their versions,
first run \code{Require} with all packages, then \code{pkgSnapshot}.
If a \code{libPaths} is used, it must be used in both functions.
}

\section{Mutual Dependencies}{

This function works best if all required packages are called within one
\code{Require} call, as all dependencies can be identified together, and all
package versions will be addressed (if there are no conflicts),
allowing a call to \code{\link[=pkgSnapshot]{pkgSnapshot()}} to take a snapshot or "record" of
the current collection of packages and versions.
}

\section{Local Cache of Packages}{

When installing new packages, \code{Require} will put all source and binary files
in an R-version specific subfolder of
\code{getOption("Require.RPackageCache")} whose default is \code{RPackageCache()}, meaning
\emph{cache packages locally in a project-independent location},
and will reuse them if needed. To turn
off this feature, set \code{options("Require.RPackageCache" = FALSE)}.
}

\examples{
\dontrun{
# simple usage, like conditional install.packages then library
library(Require)
Require("stats") # analogous to require(stats), but it checks for
#   pkg dependencies, and installs them, if missing
tempPkgFolder <- file.path(tempdir(), "Packages")

# use standAlone, means it will put it in libPaths, even if it already exists
#   in another local library (e.g., personal library)
Require("crayon", libPaths = tempPkgFolder, standAlone = TRUE)

# make a package version snapshot of installed packages
packageVersionFile <- "_.packageVersionTest.txt"
(pkgSnapshot(libPath = tempPkgFolder, packageVersionFile, standAlone = TRUE))

# Restart R -- to remove the old temp folder (it disappears with restarting R)
library(Require)
tempPkgFolder <- file.path(tempdir(), "Packages")
packageVersionFile <- "_.packageVersionTest.txt"
# Reinstall and reload the exact version from previous
Require(packageVersionFile = packageVersionFile, libPaths = tempPkgFolder, standAlone = TRUE)

# Create mismatching versions -- desired version is older than current installed
# This will try to install the older version, overwriting the newer version
desiredVersion <- data.frame(instPkgs = "crayon", instVers = "1.3.2", stringsAsFactors = FALSE)
write.table(file = packageVersionFile, desiredVersion, row.names = FALSE)
newTempPkgFolder <- file.path(tempdir(), "Packages2")

# Note this will install the 1.3.2 version (older that current on CRAN), but
#   because crayon is still loaded in memory, it will return TRUE, using the current version
#   of crayon. To start using the older 1.3.2, need to unload or restart R
Require("crayon",
  packageVersionFile = packageVersionFile,
  libPaths = newTempPkgFolder, standAlone = TRUE
)

# restart R again to get access to older version
# run again, this time, correct "older" version installs in place of newer one
library(Require)
packageVersionFile <- "_.packageVersionTest.txt"
newTempPkgFolder <- file.path(tempdir(), "Packages3")
Require("crayon",
  packageVersionFile = packageVersionFile,
  libPaths = newTempPkgFolder, standAlone = TRUE
)

# Mutual dependencies, only installs once -- e.g., httr
tempPkgFolder <- file.path(tempdir(), "Packages")
Require(c("cranlogs", "covr"), libPaths = tempPkgFolder, standAlone = TRUE)

##########################################################################################
# Isolated projects -- Just use a project folder and pass to libPaths or set .libPaths() #
##########################################################################################
# GitHub packages -- restart R because crayon is needed
library(Require)
ProjectPackageFolder <- file.path(tempdir(), "ProjectA")
#  THIS ONE IS LARGE -- > 100 dependencies -- use standAlone = FALSE to
#    reuse already installed packages --> this won't allow as much control
#    of package versioning
Require("PredictiveEcology/SpaDES@development",
  libPaths = ProjectPackageFolder, standAlone = FALSE
)

# To keep totally isolated: use standAlone = TRUE
#   --> setting .libPaths() directly means standAlone is not necessary; it will only
#   use .libPaths()
library(Require)
ProjectPackageFolder <- file.path("~", "ProjectA")
setLibPaths(ProjectPackageFolder)
Require("PredictiveEcology/SpaDES@development") # the latest version on GitHub
Require("PredictiveEcology/SpaDES@23002b2a92a92df4ccba7f51cdd82798800b2fa7")
# a specific commit (by using the SHA)


############################################################################
# Mixing and matching GitHub, CRAN, with and without version numbering
############################################################################
# Restart R -- when installing/loading packages, start fresh
pkgs <- c(
  "glue (<=1.0.4)", "digest (<= 0.6.28)", "glmm (<=1.3.0)",
  "achubaty/amc@development", "PredictiveEcology/LandR@development (>=0.0.1)",
  "PredictiveEcology/LandR@development (>=0.0.2)", "ianmseddy/LandR.CS (<=0.0.1)"
)
Require::Require(pkgs)

############################################################################
# Using libPaths -- This will only be used inside this function;
# To change .libPaths() for the whole session use a manually call to
# setLibPaths(newPath) first
############################################################################
Require::Require("SpaDES", libPaths = "~/TempLib2", standAlone = FALSE)

############################################################################
# Persistent separate packages
############################################################################
setLibPaths("~/TempLib2", standAlone = TRUE)
Require::Require("SpaDES") # not necessary to specify standAlone here because .libPaths are set

############################################################################
# Installing on many machines that are connected by a shared drive
############################################################################
options("Require.RPackageCache" = TRUE) # will binaries on the fly.
# Put thes in a shared location.
# May need to install Require in main user library before setting library paths for project
if (!require("Require")) install.packages("Require")
setLibPaths("./packages") # not shared location for library path; no longer using main user lib
Require::Require(
  packageVersionFile = "./packageVersions.txt",
  standAlone = TRUE
)
}

}
\seealso{
Useful links:
\itemize{
  \item \url{https://Require.predictiveecology.org}
  \item \url{https://github.com/PredictiveEcology/Require}
  \item Report bugs at \url{https://github.com/PredictiveEcology/Require/issues}
}

}
\author{
\strong{Maintainer}: Eliot J B McIntire \email{eliot.mcintire@canada.ca} (\href{https://orcid.org/0000-0002-6914-8316}{ORCID})

Other contributors:
\itemize{
  \item Her Majesty the Queen in Right of Canada, as represented by the Minister of Natural Resources Canada [copyright holder]
}

}
