% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forecasting_functions.R
\name{predict_portfolio}
\alias{predict_portfolio}
\title{Predict Optimal Portfolio Weights Using Time-Varying Covariance Estimation}
\usage{
predict_portfolio(
  obj,
  horizon = 1,
  max_factors = 3,
  kernel_func = epanechnikov_kernel,
  min_return = NULL,
  max_SR = NULL,
  rf = NULL
)
}
\arguments{
\item{obj}{An object of class TVMVP with the data.}

\item{horizon}{Integer. Investment horizon over which expected return and risk are computed. Default is 1.}

\item{max_factors}{Integer. The number of latent factors to consider in the Local PCA model. Default is 3.}

\item{kernel_func}{Function. Kernel used for weighting observations in Local PCA. Default is \code{\link{epanechnikov_kernel}}.}

\item{min_return}{Optional numeric. If provided, the function computes a Return-Constrained Portfolio that targets this minimum return.}

\item{max_SR}{Logical. If TRUE, the Maximum Sharpe Ratio Portfolio is also computed. Default is \code{NULL}.}

\item{rf}{Numeric. Log risk-free rate. If \code{NULL}, defaults to 0.}
}
\value{
An object of class \code{PortfolioPredictions} (an R6 object) with:
\describe{
  \item{\code{summary}}{A data frame of evaluation metrics (expected return, risk, Sharpe ratio) for all computed portfolios.}
  \item{\code{MVP}}{A list containing the weights, expected return, risk, and Sharpe ratio for the Global Minimum Variance Portfolio.}
  \item{\code{max_SR}}{(Optional) A list with metrics for the Maximum Sharpe Ratio Portfolio.}
  \item{\code{MVPConstrained}}{(Optional) A list with metrics for the Return-Constrained Portfolio.}
}
}
\description{
This function estimates optimal portfolio weights using a time-varying covariance matrix
derived from Local Principal Component Analysis (Local PCA). The procedure is available either as a stand-alone
function or as a method in the `TVMVP` R6 class. It computes the following portfolios:
\enumerate{
  \item Global Minimum Variance Portfolio (MVP)
  \item Maximum Sharpe Ratio Portfolio (if \code{max_SR = TRUE})
  \item Return-Constrained Minimum Variance Portfolio (if \code{min_return} is provided)
}
}
\details{
Two usage styles:

#' \preformatted{

# R6 method interface
tv <- TVMVP$new()
tv$set_data(returns)
tv$determine_factors(max_m=5)
prediction <- tv$predict_portfolio(horizon = 1, min_return = 0.01, max_SR = TRUE)

#' # Function interface
prediction <- predict_portfolio(obj, horizon = 5, m = 2, min_return = 0.01, max_SR=TRUE)
}
The methods can then be used on \code{prediction} to retrieve the weights.

The function estimates a time-varying covariance matrix using Local PCA:
\deqn{\hat{\Sigma}_{r,t}=\hat{\Lambda}_t \hat{\Sigma}_F \hat{\Lambda}_t' + \tilde{\Sigma}_e}
Where \eqn{\hat{\Lambda}_t} is the factor loadings at time t, \eqn{\hat{\Sigma}_F} is the factor covariance matrix, and \eqn{\tilde{\Sigma}_e} is regularized covariance matrix of the idiosyncratic errors.

It forecasts asset-level expected returns using a simple ARIMA model selection procedure and uses these in portfolio optimization.
Optimization strategies include:
\itemize{
  \item Global minimum variance (analytical)
  \item Maximum Sharpe ratio (if \code{max_SR = TRUE})
  \item Minimum variance with expected return constraint (if \code{min_return} is provided)
}
}
\section{Methods}{

The returned object includes:
\itemize{
  \item \code{$print()}: Nicely prints summary and portfolio access information.
  \item \code{$getWeights(method = c("MVP", "max_SR", "MVPConstrained"))}: Retrieves the weights for the selected portfolio.
}
}

\section{References}{
 
Lillrank, E. (2025). \ifelse{html}{
    \out{<a href='../doc/thesis.pdf'>A Time-Varying Factor Approach to Covariance Estimation</a>}
  }{Master’s thesis (PDF in inst/doc)}
  
Fan, Q., Wu, R., Yang, Y., & Zhong, W. (2024). Time-varying minimum variance portfolio. Journal of Econometrics, 239(2), 105339.
}

\examples{
\donttest{
set.seed(123)
returns <- matrix(rnorm(200 * 20, mean = 0, sd = 0.02), ncol = 20)

# Initialize object
tv <- TVMVP$new()
tv$set_data(returns)

# Optimize weights and predict returns
result <- predict_portfolio(
  tv,
  horizon = 5,
  m = 3,
  min_return = 0.02,
  max_SR = TRUE
)

# Print the portfolio performance summary
print(result)

# Access MVP weights
result$getWeights("MVP")

# Access Max Sharpe weights (if computed)
result$getWeights("max_SR")

# Or use R6 method interface
tv$determine_factors(max_m=5)
prediction <- tv$predict_portfolio(horizon = 1, min_return)
prediction
prediction$getWeights("MVPConstrained")
}

}
