% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Contours.R
\name{TernaryDensityContour}
\alias{TernaryDensityContour}
\title{Add contours of estimated point density to a ternary plot}
\usage{
TernaryDensityContour(
  coordinates,
  bandwidth,
  resolution = 25L,
  tolerance = -0.2/resolution,
  edgeCorrection = TRUE,
  direction = getOption("ternDirection", 1L),
  filled = FALSE,
  nlevels = 10,
  levels = pretty(zlim, nlevels),
  zlim,
  color.palette = function(n) hcl.colors(n, palette = "viridis", alpha = 0.6),
  fill.col = color.palette(length(levels) - 1),
  ...
)
}
\arguments{
\item{coordinates}{A list, matrix, data.frame or vector in which each
element (or row) specifies
the three coordinates of a point in ternary space.}

\item{bandwidth}{Vector of bandwidths for x and y directions.
Defaults to normal reference bandwidth (see \code{MASS::bandwidth.nrd}).
A scalar value will be taken to apply to both directions.}

\item{resolution}{The number of triangles whose base should lie on the longest
axis of the triangle.  Higher numbers will result in smaller subdivisions and smoother
colour gradients, but at a computational cost.}

\item{tolerance}{Numeric specifying how close to the margins the contours
should be plotted, as a fraction of the size of the triangle.
Negative values will cause contour lines to extend beyond the margins of the
plot.}

\item{edgeCorrection}{Logical specifying whether to correct for edge effects
(see details).}

\item{direction}{(optional) Integer specifying the direction that the
current ternary plot should point: 1, up; 2, right; 3, down; 4, left.}

\item{filled}{Logical; if \code{TRUE}, contours will be filled
(using \code{\link[graphics]{.filled.contour}()}.).}

\item{nlevels, levels, zlim, \dots}{parameters to pass to
\code{\link[graphics]{contour}()}.}

\item{color.palette}{parameters to pass to
\code{\link[graphics]{.filled.contour}()}.}

\item{fill.col}{Sent as \code{col} parameter to
\code{\link[graphics]{.filled.contour}()}.
Computed from \code{color.palette} if not specified.}
}
\value{
\code{TernaryDensityContour()} invisibly returns a list containing:
\itemize{
\item \code{x},\code{y}: the Cartesian coordinates of each grid coordinate;
\item \code{z}: The density at each grid coordinate.
}
}
\description{
Use two-dimensional kernel density estimation to plot contours of
point density.
}
\details{
This function is modelled on \code{MASS::kde2d()}, which uses
"an axis-aligned bivariate normal kernel, evaluated on a square grid".

This is to say, values are calculated on a square grid, and contours fitted
between these points.  This produces a couple of artefacts.
Firstly, contours may not extend beyond the outermost point within the
diagram, which may fall some distance from the margin of the plot if a
low \code{resolution} is used.  Setting a negative \code{tolerance} parameter allows
these contours to extend closer to (or beyond) the margin of the plot.

Individual points cannot fall outside the margins of the ternary diagram,
but their associated kernels can. In order to sample regions of the kernels
that have "bled" outside the ternary diagram, each point's value is
calculated by summing the point density at that point and at equivalent
points outside the ternary diagram, "reflected" across the margin of
the plot (see function \code{\link{ReflectedEquivalents}}).  This correction can be
disabled by setting the \code{edgeCorrection} parameter to \code{FALSE}.

A model based on a triangular grid may be more appropriate in certain
situations, but is non-trivial to implement; if this distinction is
important to you, please let the maintainers known by opening a
\href{https://github.com/ms609/Ternary/issues/new?title=Triangular+KDE}{Github issue}.
}
\examples{
# Generate some example data
nPoints <- 400L
coordinates <- cbind(abs(rnorm(nPoints, 2, 3)),
                     abs(rnorm(nPoints, 1, 1.5)),
                     abs(rnorm(nPoints, 1, 0.5)))
# Set up plot
oPar <- par(mar = rep(0, 4))
TernaryPlot(axis.labels = seq(0, 10, by = 1))

# Colour background by density
ColourTernary(TernaryDensity(coordinates, resolution = 10L),
              legend = TRUE, bty = "n", title = "Density")

# Plot points
TernaryPoints(coordinates, col = "red", pch = ".")

# Contour by density
TernaryDensityContour(coordinates, resolution = 30L)

# Reset plotting parameters
par(oPar)
}
\seealso{
Other contour plotting functions: 
\code{\link{ColourTernary}()},
\code{\link{TernaryContour}()},
\code{\link{TernaryPointValues}()}
}
\author{
Adapted from \code{MASS::kde2d()} by Martin R. Smith
}
\concept{contour plotting functions}
