# This file was generated by Rcpp::compileAttributes
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Estimate Transfer Entropy.
#'
#' \code{ComputeTE} Estimate the Transfer Entropy (TE) from one continuous-valued random process to a second process.
#'
#' A function to calculate Transfer Entropy from random process \eqn{Y} to random process \eqn{X}. 
#' The TE, introduced by Schreiber in 2000, extends the concept of mutual information to provide a 
#' direction-sensitive measure of information flow between two time series. Formally, 
#' the transfer entropy from time series \eqn{Y} to \eqn{X} is given by 
#' \eqn{T_{Y \rightarrow X} = \sum p(x_{n+1},x_n^{(k)},y_n^{(l)}) log \frac{p(x_{n+1} \mid x_n^{(k)}, y_n^{(l)})}{p(x_{n+1} \mid x_n^{(k)})}} 
#' where \eqn{x_{n+1}} is the value of \eqn{X} at time \eqn{n+1}, and \eqn{x_n^{(k)}} (\eqn{y_n^{(l)}}) is 
#' the \eqn{k} (\eqn{l}) lagged values of \eqn{X} (\eqn{Y}) at time \eqn{n}.
#' The definition of TE assumes \eqn{X} is an Markov process. The  \code{embedding} dimension should 
#' be chosen to match the delay of the Markov process. The TE measures the additional amount of 
#' information \eqn{Y} contains about \eqn{X} over the information contained in the Markov embedding.
#' Two methods for estimating TE are provided. The first is based on the mutual information distance 
#' \eqn{MI(X_{i+1} | X^{(e) },Y_i) - MI(X_{i+1} | X^{(e)} )}, where \eqn{e} is the embedding dimension. 
#' This approach follows directly from the definition of the TE. Mutual information is estimated using 
#' the k-nearest neighbor approach suggested by Krasvok.
#' The second method is based on the generalized correlation sum. 
#'
#' Things can go wrong in several ways. First, the random processes must meet the assumption of the TE. 
#' That is, \eqn{X} must represent some form of Markov process whose probability distribution may also be 
#' influenced by \eqn{Y}. A more subtle error can occur when multiple points in \eqn{X^(k)} 
#' (or some of its subspaces) have identical coordinates. This can lead to several points which have 
#' identical distance to a query point, which violates the assumptions of the Kraskov estimator, causing it 
#' to throw an error. The solution in this case is to add some small noise to the measurements \eqn{X} prior 
#' to computing TE.
#'
#' @param X Numeric vector, Transfer Entropy is calculated to random process X 
#' @param Y Numeric vector, Transfer Entropy is calculated from random process Y
#' @param embedding Numeric, The embedding dimension. Must be positive integer
#' @param method String, The method to be used to estimate TE from ("MI_dif","Correlation")
#' @param k Numeric, The k'th neighbor used by the Kraskov estimator. Must be positive integer. Kraskov suggests a value in (1,3)
#' @param epsDistace Numeric, The distance used for measuring TE in Correlation method, by default it is the average distance calculated in XKY
#' @param safetyCheck Logical, For computing TE using "mi_diff" method the data need to be noisy otherwise a crach might happen. This parameter can check if there are any idetical points in the spaces made for this use
#' @return Numeric, The estimated transfer entropy
#' 
#' @aliases te transferentropy transfer-entropy
#' 
#' @examples
#' ## Intitialize two vectors of length 10001
#' X <- rep(0,10000+1) 
#' Y <- rep(0,10000+1)
#' ## Create two linked random processes. Y is independent of X,
#' ## while X is determined in part by the previous values of Y.
#' for(i in 1:10000){
#'  Y[i+1] <- 0.6*Y[i] + rnorm(1)
#'  X[i+1] <- 0.4*X[i] + 0.6*Y[i] + rnorm(1)
#' }
#' X <- X[101:10000]
#' Y <- Y[101:10000]
#' ## Compute the TE from Y to X
#' computeTE(X,Y,3,1,"MI_diff")  ## should be circa 0.16
#' ## and from X to Y
#' computeTE(Y,X,3,1,"MI_diff")  ## should be circa zero
#' computeTE(X,Y,3,1,"Correlation",0.4)
#' computeTE(Y,X,3,1,"Correlation",0.4)

computeTE <- function(X, Y, embedding, k, method="MI_diff", epsDistace=-1, safetyCheck=FALSE) {
  methods = c("correlation","mi_diff")
  method = charmatch(tolower(method),methods)
  if (is.na(method) )
    stop("Method not specified correctly. Please choose one of the following. (\"MI_diff\", \"Correlation\")")
  ## Sanity checks:
  if( embedding%%1 != 0 || embedding <= 0 )
    stop("Embedding should be an integer greater than zero")
  if( embedding > sqrt(length(X)) )
    warning("Embedding is big may not be sensible")
  if( k%%1 != 0 || k <= 0 )
    stop("k (neighbor) should be an integer greater than zero")
  if( k > 10 )
    warning("k is big may not be sensible")
  if( length(X) != length(Y))
    stop("X and Y should have the same length")
  
  .Call('Rcpp_ComputeTE', X, Y, embedding, k, methods[method], epsDistace, safetyCheck, PACKAGE = 'TransferEntropy')
}


## WARNING if you want to extend the codebase !!
## The algorithms used here assume nearest neighbors are determined using the max norm. We implement the nearest neigbhor search using the ANN (Approximate Nearest Neighbor) library. ANN uses Euclidian distance by default; using the max norm requires adjusting some variables in the library's header files. Keep this in mind if you choose to extend this library.


