% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stem_segmentation_methods.R
\name{sgmt.ransac.circle}
\alias{sgmt.ransac.circle}
\title{Stem segmentation algorithm: RANSAC circle fit}
\usage{
sgmt.ransac.circle(tol = 0.025, n = 10, conf = 0.99, inliers = 0.8)
}
\arguments{
\item{tol}{\code{numeric} - tolerance offset between absolute radii estimates and hough transform estimates.}

\item{n}{\code{integer} - number of points selected on every RANSAC iteration.}

\item{conf}{\code{numeric} - confidence level.}

\item{inliers}{\code{numeric} - expected proportion of inliers among stem segments' point cloud chunks.}
}
\description{
This function is meant to be used inside \code{\link{stemSegmentation}}. It applies a least squares circle fit algorithm in a RANSAC fashion over stem segments. Mode details are given in the sections below.
}
\section{Output Fields}{


\itemize{
\item \code{TreeID}:  unique tree IDs - available only for multiple stems
\item \code{Segment}: stem segment number (from bottom to top)
\item \code{X}, \code{Y}: circle center coordinates
\item \code{Radius}: estimated circles radii
\item \code{Error}: least squares circle fit error
\item \code{AvgHeight}: average height of stem segments
\item \code{N}: number of points in the stem segments
}
}

\section{Least Squares Circle Fit}{


The circle fit method applied in \emph{TreeLS} estimates the circle parameters from a 
pre-selected (denoised) set of points 
by \href{https://en.wikipedia.org/wiki/QR_decomposition}{QR decompostion}.
The optimization criterion for selecting the best circle parameters among several possible candidates
is the least squares method, that selects a set of circle parameters that minimize the sum of squared 
distances between the model circle and its originating points.
}

\section{RANSAC Algorithm}{


The \strong{RAN}dom \strong{SA}mple \strong{C}onsensus algorithm is a method that relies on resampling 
a data set as many times as necessary to find a subset comprised of only inliers - e.g. observations
belonging to a desired model. The RANSAC algorithm provides a way of estimating the necessary number of
iterations necessary to fit a model using inliers only, at least once, as shown in the equation:
\deqn{k = log(1 - p) / log(1 - w^n)}
where:
\itemize{
\item \emph{k}: number of iterations
\item \emph{p}: confidence level - desired probability of success 
\item \emph{w}: proportion of inliers expected in the \emph{full} dataset
\item \emph{n}: number of observations sampled on every iteration
}

The models reiterated in \emph{TreeLS} usually relate to circle or cylinder 
fitting over a set of 3D coordinates, selecting the best possible model through the RANSAC algorithm

For more information, checkout \href{https://en.wikipedia.org/wiki/Random_sample_consensus}{this wikipedia page}.
}

\examples{
### single tree
file = system.file("extdata", "pine.laz", package="TreeLS")
tls = readTLS(file)
tls = stemPoints(tls)
df = stemSegmentation(tls)

head(df)
tlsPlot(tls, df)

### forest plot
file = system.file("extdata", "pine_plot.laz", package="TreeLS")
tls = readTLS(file)

# normalize the point cloud
tls = tlsNormalize(tls)

# map the trees on a resampled point cloud so all trees have approximately the same point density
thin = tlsSample(tls, voxelize(0.02))
map = treeMap(thin, map.hough(min_density = 0.05))

tls = stemPoints(tls, map)
df = stemSegmentation(tls, sgmt.ransac.circle(n=10))

head(df)
tlsPlot(tls, df, map)
}
\references{
Conto, T. ; Olofsson, K. ; Gorgens, E. B. ; Rodriguez, L. C. E. ; Almeida, G. Performance of stem denoising and stem modelling algorithms on single tree point clouds from terrestrial laser scanning. Computers and Electronics in Agriculture, v. 143, p. 165-176, 2017.
}
