% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/print.R, R/vic.R
\name{print.vic_output}
\alias{print.vic_output}
\alias{vic}
\title{VIC model run for each gridcells}
\usage{
\method{print}{vic_output}(x, ...)

vic(
  forcing,
  soil,
  veg,
  output_info = NULL,
  veglib = NULL,
  snowband = NULL,
  lake = NULL,
  forcing_veg = NULL,
  veg_force_types = c("albedo", "LAI", "fcanopy"),
  parall = FALSE
)
}
\arguments{
\item{x}{Return of \code{\link[=vic]{vic()}} for print.}

\item{...}{Other arguments to print.}

\item{forcing}{meteorological forcing data. Must be a list of numeral matrix
with the name of one of "PREC", "TEMP", "SW", "LW", "WIND", "VP" and "PRESS".
See details.}

\item{soil}{soil parameter data. Must be a data frame or numeral matrix.}

\item{veg}{vegetation parameters. Must be a list containing several matrixs.
See details.}

\item{output_info}{A list containing output contents and timescales
(optional). See details.}

\item{veglib}{Vegetation library parameters (optional). Would using the
library of the NLDAS and GLDAS when not provided.}

\item{snowband}{A data frame or numeral matrix containing snow band
parameters (optional). See details.}

\item{lake}{A dataframe or numeric matrix containing lake parameters
(optional).}

\item{forcing_veg}{Vegetation forcing data (optional). See details.}

\item{veg_force_types}{Type names of vegetation forcing data. Must be
provided when using vegetation forcing data.}

\item{parall}{Determined if run the VIC parallely. If it is TRUE,
\code{registerDoParallel()} in package \pkg{doParallel} is
need to be used before run the VIC.}
}
\value{
A list containing output tables, time table and model settings.

VIC in R supports multiple "output tables" with different output timescales.
For example,
you can put soil moisture and soil temperature in a table with monthly
timescale and put runoff, baseflow in another table with daily timescale.

You can use \code{print()} to print the summary of the returns.
The returns includes:
\tabular{ll}{
\code{output} \tab One or several "output tables" and each of them containing
several output variables, which is determined by the \code{output_info}
parameter. The name of the "table" would also be which \code{output_info}
determined. Each "table" containing several numerial matrices corresponding
to the output variables. Each row of the matrices is output data for a
time step while each column is output data for a gridcell. \cr

\code{timetable} \tab Containing initiation time, model running time and
final (orgnizate outputs) time of the VIC model running.\cr

\code{global_options} \tab Global options setted for this model run. \cr
\code{output_infos} \tab Output settings determined by the \code{output_info}
parameter.
}
}
\description{
Run the VIC model for each gridcells by providing several meteorological
and vegetation (optional) forcing data and land surface parameters (soil,
vegetation, snowband (optional), lake (optional)).
}
\details{
Function \code{vic} is used to run the VIC model in a "classic" style
(Run the model gridcell by gridcell). Meteorological forcing data,
soil parameters and vegetation parameters are the basic necessary inputs
for the VIC model.
}
\section{Forcing data}{

Parameter \code{forcing} must be a list containing several numeral matrixs
that containing forcing data. Name of each matrix (similar to key in
dictionary in Python) must be the specific type names including
"PREC", "TEMP", "SW", "LW", "WIND", "VP" and "PRESS", indicating precipitation
\verb{[mm]}, air temperature \verb{[degree C]}, shortwave radiation \verb{[W]}, longwave radiation
\verb{[W]}, wind speed \verb{[m/s]}, vapor pressure \verb{[kPa]}, and atmospheric pressure \verb{[kPa]}.
All of those types are necessary to run the VIC model except "LW" and "PRESS".
Each row of the matrixs is corresponding to a time step while each column of
the matrixs is corresponding to a gridcell, which other must be the same as
those in soil parameter.

Longwave radiation (LW) and atmospheric pressure (PRESS) could be estimated
via other forcing data when not supplied. Longwave radiation would be
estimated using the Konzelmann formula (Konzelmann et al., 1996) while
atmospheric pressure would be estimated based on the method of VIC 4.0.6,
by assuming the sea level pressure is a constant of 101.3 kPa.
}

\section{Soil parameters}{

Parameter \code{soil} must be a numeric data frame or matrix that containing
soil parameters. The style of this is the same as the soil parameter file
of the classic VIC, that is, each row restores the parameter of a cell
while each column restores one type of parameter. Detail see
\url{http://vic.readthedocs.io/en/master/Documentation/Drivers/Classic/SoilParam/}
in the official VIC documentation webside.
}

\section{Vegetation parameter}{

Parameter \code{veg} must be a list containing several numeral matrixs that
Those matrixs restore the vegetation parameters that are corresponding
to each gridcells, and those order must be the same as the soil parameters.
Each row of the matrix restores the parameters of a vegetation type
while each column restores a type of parameter.
Each row should be like:
\preformatted{
c(veg_type, area_fract, rootzone_1_depth,
  rootzone_1_fract, rootzone_2_depth, rootzone_2_fract, ...)
}
which is similar to the veg param file
of the classic VIC. If the source of LAI, fcanopy or albedo is set
to veg params, it must be follow by a sequence of param value for each
month in a year. The rows of \code{veg} would be similar as:
\preformatted{
c(veg_type, area_fract, rootzone_1_depth,
  rootzone_1_fract, rootzone_2_depth, rootzone_2_fract,
  LAI_Jan, LAI_Feb, LAI_Mar, ..., LAI_Dec,
  fcan_Jan, fcan_Feb, fcan_Mar, ..., fcan_Dec,
  albedo_Jan, albedo_Feb, albedo_Mar, ..., albedo_Dec)
}
}

\section{Output information (Optional)}{

Parameter \code{output_info} is used to determine the output variables,
output timescale (monthly, daily, sub-daily, or each 6 days, etc.),
aggregration of data (mean, sum, max, min, start or end) when output
timescale is larger than input timescale. It should be a list like that:
\preformatted{
output_info <- list(timescale = 'timescale', aggpar = aggpar,
                    outvars = c('OUT_TYPE_1', 'OUT_TYPE_2', 'OUT_TYPE_3', ...),
                    aggtypes = c('aggtype_1', 'aggtype_2', 'aggtype_3'))
}

And a output table (a list containing the output variables in matrix form)
named "output" would be returned. You can obtain the variables use code like
\code{res$output$OUT_...}.
Names of the items in the list (e.g. timescale, outvars) must be those
specified as follows:

\tabular{ll}{
\code{timescale} \tab Output timescale, including 'never', 'step', 'second',
'minute', 'hour', 'day', 'month', 'year', 'date', and 'end'. 'never'
means never output, 'step' means use the input timestep, 'date'
means output at a specific date, and 'end' means output at the last
timestep.\cr
\code{aggpar} \tab If 'timescale' is set to those except 'never', 'date' and
'end', it determined the intervals of the timescale to pass before output.

If 'timescale' is 'day' and 'aggpar' is 6, that means data outputs per 6
days.

If 'timescale' is 'date', it should be a \code{Date} type and it could be
generated use \code{as.Date('YYYY-mm-dd')}.\cr
\code{outvars} \tab A sequence of names of output data types. The available
data types please see the VIC official documentation website at
\url{http://vic.readthedocs.io/en/master/Documentation/OutputVarList/}.\cr
\code{aggtypes} \tab Optional. A sequence of string determine how to
aggregrate the output data when output timescale is larger than input
timescale, Including 'avg' (average), 'begin', 'end', 'max', 'min', 'sum',
'default'.
Each string in it must be corresponding to those in 'outvars'.

If input timescale is daily, while output timescale is monthly, and
aggtype is 'begin', it would output the data of the first day of each
month.
}

If multiple output timescales are used, the outputs could be divided
into several lists and take them into a list as input, e.g.:
\preformatted{
out_info <- list(
  wb = list(timescale = 'hour', aggpar = 6,
            outvars = c('OUT_RUNOFF', 'OUT_BASEFLOW', 'OUT_SOIL_MOIST'),
            aggtypes = c('sum', 'sum', 'end')),
  eb = list(timescale = 'day', aggpar = 1,
            outvars = c('OUT_SWE', 'OUT_SOIL_TEMP'),
            aggtypes = c('avg', 'min'))
)
}
This would return two output tables named "wb" and "eb" respectively.
}

\section{Vegetation library (Optional)}{

Parameter \code{veglib} is a matrix or a numeric dataframe of a vegetation
parameter library. Each row determines a type of vegetation while each
column determines a parameter, including ovetstory (or not), LAI for each
month in a year, etc. If not provided, it would use the default vegetation
library of the NASA Landsurface Data Assimination System (LDAS) (Rodell
et al., 2004), which contains 11 types of vegetation with the vegetation
classification of UMD.
}

\section{Snowband (elevation band) (Optional)}{

Parameter \code{snowband} is a matrix or a numeric dataframe determines
the elevation band information for each gridcells. Each row determines
the band information of a gridcell while a column determines the values
of the elevation band parameters.
This devide a single gridcell into several parts with dfferent elevation
to run individually, to further consider the sub-gridcell heterogeneity
of elevation and the resulted heterogeneity air temperature in a gridcell
with higher variation of elevation. The information of elevation bands
includes area fraction, mean elevation and fraction of precipitation
falled to the gridcell of each elevation band of the gridcell.
The order of the rows must be coresponding to the gridcells determined
in the soil parameters. Each row should be like:
\preformatted{
c(GRID_ID, AFRAC_1, AFRAC_2, ..., AFRAC_n, ELEV_1, ELEV_2, ..., ELEV_n,
  PFRAC_1, PFRAC_2, ..., PFRAC_n)
}
\code{GRID_ID} is the id of the grid; AFRAC_i means area fraction of
each elevation band; ELEV_i is their mean elevation; PFRAC_i is there
precipitation fraction. n is the number of elevation bands for each
gridcell and is determined by \code{'nbands'} in the global options.
This can be set used \code{veg_param('nbands', n)}.
}

\section{Vegetation forcing data (Optional)}{

Parameter \code{forcing_veg} must be a list containing several 3D numeral
arrays that containing vegetation forcing data such as LAI, albedo and
fraction of canopy. Different to parameter \code{forcing}, each 3D array
in the list is the vegetation forcing data for a single gridcell, and the
order of the 3D arrays must be corresponding to the order of the gridcells
determined in the soil parameter.

The dimensions of a 3D array are represents: \verb{[timestep, vegetation tile, forcing type]}

That is, the first dimension determines the data of the timesteps, the
second dimension determines the data for the different vegetation tiles
in this gridcell that determined by the vegetation parameters, while
the third dimension determined the data of different forcing data type
(LAI, albedo or fcanopy), which should be corresponding to the parameter
\code{veg_force_types}.
}

\examples{
# This is a sample data to run VIC.
data(STEHE)

forcing <- STEHE$forcing
soil <- STEHE$soil
veg <- STEHE$veg

# Set the global options for a 7-days run.
vic_param('start_year', 1949)
vic_param('start_month', 1)
vic_param('start_day', 1)
vic_param('end_year', 1949)
vic_param('end_month', 1)
vic_param('end_day', 7)
vic_param('step_per_day', 24)
vic_param('snow_step_per_day', 24)
vic_param('runoff_step_per_day', 24)

# Run VIC.
outputs <- vic(forcing, soil, veg)
print(outputs)

# Use user defind outputs and snowband parameters.
vic_param('nbands', 5)
band <- STEHE$snowbands

out_info <- list(
  wb = list(timescale = 'hour', aggpar = 6,
            outvars = c('OUT_RUNOFF', 'OUT_BASEFLOW', 'OUT_SOIL_MOIST'),
            aggtypes = c('sum', 'sum', 'end')),
  eb = list(timescale = 'day', aggpar = 1,
            outvars = c('OUT_SWE', 'OUT_SOIL_TEMP'),
            aggtypes = c('avg', 'min'))
)

outputs <- vic(forcing, soil, veg, snowband = band, output_info = out_info)
print(outputs)

# Example of parallelization
\dontrun{
library(doParallel)
registerDoParallel(cores=4)
outputs <- vic(forcing, soil, veg, snowband = band, parall = TRUE)
stopImplicitCluster()
print(outputs)
}

}
\references{
Hamman, J. J., Nijssen, B., Bohn, T. J., Gergel, D. R., and Mao, Y. (2018),
The Variable Infiltration Capacity model version 5 (VIC-5): infrastructure
improvements for new applications and reproducibility, Geosci. Model Dev., 11,
3481-3496, \doi{10.5194/gmd-11-3481-2018}.

Konzelmann, T, Van de Wal, R.S.W., Greuell, W., Bintanja, R., Henneken, E.A.C.,
Abe-Ouchi, A., 1996. Parameterization of global and longwave incoming radiation
for the Greenland Ice Sheet. Global Planet. Change, 9:143-164,
\doi{10.1016/0921-8181(94)90013-2}.

Liang, X., Lettenmaier, D. P., Wood, E. F., and Burges, S. J. (1994), A
simple hydrologically based model of land surface water and energy
fluxes for general circulation models, J. Geophys. Res., 99(D7),
14415-14428, \doi{10.1029/94JD00483}.

Liang, X., and Xie, Z., 2001: A new surface runoff parameterization
with subgrid-scale soil heterogeneity for land surface models,
Advances in Water Resources, 24(9-10), 1173-1193.

Rodell, M., Houser, P.R., Jambor, U., Gottschalck, J., Mitchell, K.,
Meng, C.-J., Arsenault, K., Cosgrove, B., Radakovich, J., Bosilovich, M.,
Entin, J.K., Walker, J.P., Lohmann, D., and Toll, D. (2004), The Global
Land Data Assimilation System, Bull. Amer. Meteor. Soc., 85(3), 381-394.
}
