#' @title ExportVirusDataFrame: Export processed hittables and summary stats data frames
#'
#' @description Export data frames generated by Virusparies functions.
#'
#' @param df A summary statistics or VirusHunterGatherer hittable data frame.
#'
#' @param file_name  A character string naming the file, optionally including ".tsv" or ".csv" file extensions.
#' If suffix is not provided, file_type will be used to determine the file type.
#'
#' @param dir_path A character string indicating the directory path where the file will be saved (default: current corking directory).
#'
#' @param file_type A character vector specifying the type of file to export. Can be "csv" or "tsv".
#' If NULL and file_name does not specify a suffix, the function infers the file type based on the prefix in file_name.
#' @param create_path Logical indicating whether to create the directory path
#' specified in dir_path if it does not already exist (default: FALSE).
#'
#' @details
#' Functions in the Virusparies package can generate both plots and new data frames.
#' Data frames contain either summary statistics for contig length, E-value or identity in percentage
#' or a processed hittable for example outlier or observations below threshold, when running
#' \code{\link{VhgBoxplot}}.
#'
#' Both types of data frames can be exported via `ExportVirusDataFrame`. Summary stats and hittables
#' can be exported as CSV files or in TSV format, if the user prefers the file type used in VirusHunterGatherer
#' hittables.
#'
#'
#' @return A message indicating that export was successful.
#'
#' @author Sergej Ruff
#' @examples
#'
#' path <- system.file("extdata", "virushunter.tsv", package = "Virusparies")
#' file <- ImportVirusTable(path)
#'
#'
#' # generate a plot that returns both processed hittables (outlier) and summary stats
#' plot1 <- VhgBoxplot(file, x_column = "best_query", y_column = "ViralRefSeq_E")
#'
#'
#' \donttest{
#'
#' # export hittable as tsv (same format as input hittables)
#' ExportVirusDataFrame(df=plot1$outlier,file_name="outlier",file_type="tsv",
#' dir_path=tempdir())
#'
#' # export summary stats as csv
#' ExportVirusDataFrame(df=plot1$summary_stats,file_name="summarystats",
#' file_type="csv",dir_path=tempdir())
#'
#' }
#'
#' @seealso
#' VirusHunterGatherer is available here: \url{https://github.com/lauberlab/VirusHunterGatherer}.
#'
#' @importFrom utils write.table
#' @export
ExportVirusDataFrame <- function(df,
                                 file_name,
                                 dir_path = NULL,
                                 file_type = NULL,
                                 create_path = FALSE) {
  # If file_type is NULL, determine based on file_name suffix
  if (is.null(file_type)) {
    if (grepl(".csv$", file_name)) {
      file_type <- "csv"
      file_name <- sub(".csv$", "", file_name)
    } else if (grepl(".tsv$", file_name)) {
      file_type <- "tsv"
      file_name <- sub(".tsv$", "", file_name)
    } else {
      stop("File type not specified and could not be determined from file_name suffix.")
    }
  } else {
    # Ensure the file_type argument is one of "csv" or "tsv"
    file_type <- match.arg(file_type, c("csv", "tsv"))

    # Remove suffix if it's already included in file_name
    file_name <- sub(paste0(".", file_type, "$"), "", file_name)
  }

  # If dir_path is NULL, use current working directory
  # if (is.null(dir_path)) {
  #   dir_path <- getwd()
  # }

  # Check if the directory path should be created
  if (create_path && !dir.exists(dir_path)) {
    dir.create(dir_path, recursive = TRUE)
    message("Directory created: ", dir_path)
  }

  # Determine the full file name with extension
  full_name <- paste0(file_name, ".", file_type)

  # Construct the full path
  full_path <- file.path(dir_path, full_name)

  # Export the data frame based on file_type
  if (file_type == "csv") {
    write.csv(df, file = full_path, row.names = FALSE)
  } else if (file_type == "tsv") {
    write.table(df, file = full_path, sep = "\t", row.names = FALSE, col.names = TRUE, quote = FALSE)
  } else {
    stop("Invalid file type. Choose 'csv' or 'tsv'.")
  }

  message("File successfully exported to: ", full_path)
}
