% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/growth_curve_sampler.R
\name{growth_curve_sampler}
\alias{growth_curve_sampler}
\title{MCMC sampler for measurements of individuals with replicates and age information to generate growth curve}
\usage{
growth_curve_sampler(data, priors, subject_info, package_only = FALSE)
}
\arguments{
\item{data}{Photogrammetric data formatted for Xcertainty models, required to
be an object with class \code{obs.parsed}, which can be obtained by running
\code{parse_observations()}}

\item{priors}{\code{list} with components that define the model's prior 
distribution.  See \code{help("flatten_data")} for more details.}

\item{subject_info}{\code{data.frame} with elements \code{Year}, 
\code{Subject}, \code{Group}, \code{ObservedAge}, and \code{AgeType}.  See
\code{help("whale_info")} for descriptions of \code{data.frame} columns.}

\item{package_only}{\code{TRUE} to return the formatted data used to build 
the sampler, otherwise \code{FALSE} to return the sampler}
}
\value{
outputs a function to run a sampler, the function arguments are: 
\describe{
 \item{niter}{set the number of iterations}
 \item{burn}{set the number samples to discard}
 \item{thin}{set the thinning rate}
 }
}
\description{
Build an MCMC sampler that uses calibration data to estimate the total 
length of animals.  The total lengths are assumed to follow a growth curve
model, so replicates across time points that include age information are 
required to fit the model.  The length model is a von-Bertalanffy-Putter 
growth model, following Pirotta & Bierlich et al., (in revision).
}
\examples{
# load example wide-format data
data("calibration")
data("whales")
data("whale_info")

# parse calibration study
calibration_data = parse_observations(
  x = calibration, 
  subject_col = 'CO.ID',
  meas_col = 'Lpix', 
  tlen_col = 'CO.L', 
  image_col = 'image', 
  barometer_col = 'Baro_Alt',
  laser_col = 'Laser_Alt', 
  flen_col = 'Focal_Length', 
  iwidth_col = 'Iw', 
  swidth_col = 'Sw',
  uas_col = 'uas'
)

# parse field study
whale_data = parse_observations(
  x = whales, 
  subject_col = 'whale_ID',
  meas_col = 'TL.pix', 
  image_col = 'Image', 
  barometer_col = 'AltitudeBarometer',
  laser_col = 'AltitudeLaser', 
  flen_col = 'FocalLength', 
  iwidth_col = 'ImageWidth', 
  swidth_col = 'SensorWidth', 
  uas_col = 'UAS',
  timepoint_col = 'year'
)

# build sampler
sampler_data = growth_curve_sampler(
  data = combine_observations(calibration_data, whale_data),
  priors = list(
    image_altitude = c(min = 0.1, max = 130),
    altimeter_bias = rbind(
      data.frame(altimeter = 'Barometer', mean = 0, sd = 1e2),
      data.frame(altimeter = 'Laser', mean = 0, sd = 1e2)
    ),
    altimeter_variance = rbind(
      data.frame(altimeter = 'Barometer', shape = .01, rate = .01),
      data.frame(altimeter = 'Laser', shape = .01, rate = .01)
    ),
    altimeter_scaling = rbind(
      data.frame(altimeter = 'Barometer', mean = 1, sd = 1e1),
      data.frame(altimeter = 'Laser', mean = 1, sd = 1e1)
    ),
    pixel_variance = c(shape = .01, rate = .01),
    # priors from Agbayani et al. 
    zero_length_age = c(mean = -5.09, sd = 0.4),
    growth_rate = c(mean = .18, sd = .01),
    # additional priors
    group_asymptotic_size = rbind(
      Female = c(mean = 12, sd = .5),
      Male = c(mean = 12, sd = .5)
    ),
    group_asymptotic_size_trend = rbind(
      Female = c(mean = 0, sd = 1),
      Male = c(mean = 0, sd = 1)
    ),
    subject_group_distribution = c(Female = .5, Male = .5),
    asymptotic_size_sd = c(min = 0, max = 10),
    min_calf_length = 3.5,
    # To model break points between 1990 and 2015
    group_size_shift_start_year = c(min = 1990, max = 2015)
  ),
  subject_info = whale_info,
  # set to false to return sampler function
  package_only = TRUE
)
}
