% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf.R
\name{orsf}
\alias{orsf}
\alias{orsf_train}
\title{Oblique Random Survival Forest (ORSF)}
\usage{
orsf(
  data,
  formula,
  control = orsf_control_fast(),
  weights = NULL,
  n_tree = 500,
  n_split = 5,
  n_retry = 3,
  mtry = NULL,
  leaf_min_events = 1,
  leaf_min_obs = 5,
  split_min_events = 5,
  split_min_obs = 10,
  split_min_stat = 3.841459,
  oobag_pred_type = "surv",
  oobag_pred_horizon = NULL,
  oobag_eval_every = n_tree,
  oobag_fun = NULL,
  importance = "anova",
  tree_seeds = NULL,
  attach_data = TRUE,
  no_fit = FALSE,
  ...
)

orsf_train(object)
}
\arguments{
\item{data}{a \link{data.frame}, \link[tibble:tibble-package]{tibble}, or \link[data.table:data.table]{data.table} that contains the
relevant variables.}

\item{formula}{(\emph{formula}) The response on the left hand side should
include a time variable, followed by a status variable, and may be
written inside a call to \link[survival:Surv]{Surv} (see examples).
The terms on the right are names of predictor variables.}

\item{control}{(\emph{orsf_control}) An object returned from one of the
\code{orsf_control} functions:
\itemize{
\item \link{orsf_control_fast} (the default) uses a single iteration of Newton
Raphson scoring to identify a linear combination of predictors.
\item \link{orsf_control_cph} uses Newton Raphson scoring until a convergence
criteria is met.
\item \link{orsf_control_net} uses \code{glmnet} to identify linear combinations of
predictors, similar to Jaeger (2019).
\item \link{orsf_control_custom} allows the user to apply their own function
to create linear combinations of predictors.
}}

\item{weights}{(\emph{numeric vector}) Optional. If given, this input should
have length equal to \code{nrow(data)}. Values in \code{weights} are treated like
replication weights, i.e., a value of 2 is the same thing as having 2
observations in \code{data}, each containing a copy of the corresponding
person's data.

\emph{Use} \code{weights} \emph{cautiously}, as \code{orsf} will count the number of
observations and events prior to growing a node for a tree, so higher
values in \code{weights} will lead to deeper trees.}

\item{n_tree}{(\emph{integer}) the number of trees to grow.
Default is \code{n_tree = 500.}}

\item{n_split}{(\emph{integer}) the number of cut-points assessed when splitting
a node in decision trees. Default is \code{n_split = 5}.}

\item{n_retry}{(\emph{integer}) when a node can be split, but the current
linear combination of inputs is unable to provide a valid split, \code{orsf}
will try again with a new linear combination based on a different set
of randomly selected predictors, up to \code{n_retry} times. Default is
\code{n_retry = 3}. Set \code{n_retry = 0} to prevent any retries.}

\item{mtry}{(\emph{integer}) Number of predictors randomly included as candidates
for splitting a node. The default is the smallest integer greater than
the square root of the number of total predictors, i.e.,
\verb{mtry = ceiling(sqrt(number of predictors))}}

\item{leaf_min_events}{(\emph{integer}) minimum number of events in a
leaf node. Default is \code{leaf_min_events = 1}}

\item{leaf_min_obs}{(\emph{integer}) minimum number of observations in a
leaf node. Default is \code{leaf_min_obs = 5}}

\item{split_min_events}{(\emph{integer}) minimum number of events required
in a node to consider splitting it. Default is \code{split_min_events = 5}}

\item{split_min_obs}{(\emph{integer}) minimum number of observations required
in a node to consider splitting it. Default is \code{split_min_obs = 10}.}

\item{split_min_stat}{(double) minimum test statistic required to split
a node. Default is 3.841459 for the log-rank test, which is roughly
a p-value of 0.05}

\item{oobag_pred_type}{(\emph{character}) The type of out-of-bag predictions
to compute while fitting the ensemble. Valid options are
\itemize{
\item 'none' : don't compute out-of-bag predictions
\item 'risk' : predict the probability of having an event at or before \code{oobag_pred_horizon}.
\item 'surv' : 1 - risk.
\item 'chf'  : predict cumulative hazard function
}

Mortality ('mort')is not implemented for out of bag predictions yet, but it
will be in a future update.}

\item{oobag_pred_horizon}{(\emph{numeric}) A numeric value indicating what time
should be used for out-of-bag predictions. Default is the median
of the observed times, i.e., \code{oobag_pred_horizon = median(time)}.}

\item{oobag_eval_every}{(\emph{integer}) The out-of-bag performance of the
ensemble will be checked every \code{oobag_eval_every} trees. So, if
\code{oobag_eval_every = 10}, then out-of-bag performance is checked
after growing the 10th tree, the 20th tree, and so on. Default
is \code{oobag_eval_every = n_tree}.}

\item{oobag_fun}{(\emph{function}) to be used for evaluating out-of-bag prediction accuracy every \code{oobag_eval_every}
trees. When \code{oobag_fun = NULL} (the default), Harrell's C-statistic (1982) is used to evaluate accuracy. if you use your own \code{oobag_fun} note the following:
\itemize{
\item \code{oobag_fun} should have two inputs: \code{y_mat} and \code{s_vec}
\item \code{y_mat} is a two column matrix with first column named 'time', second named 'status'
\item \code{s_vec} is a numeric vector containing predicted survival probabilities.
\item \code{oobag_fun} should return a numeric output of length 1
}

For more details, see the out-of-bag \href{https://bcjaeger.github.io/aorsf/articles/oobag.html#user-supplied-out-of-bag-evaluation-functions}{vignette}.}

\item{importance}{(\emph{character}) Indicate method for variable importance:
\itemize{
\item 'none': no variable importance is computed.
\item 'anova': compute analysis of variance (ANOVA) importance
\item 'negate': compute negation importance
\item 'permute': compute permutation importance
}

For details on these methods, see \link{orsf_vi}.}

\item{tree_seeds}{(\emph{integer vector}) Optional. if specified, random seeds
will be set using the values in \code{tree_seeds[i]}  before growing tree \code{i}.
Two forests grown with the same number of trees and the same seeds will
have the exact same out-of-bag samples, making out-of-bag error
estimates of the forests more comparable. If \code{NULL} (the default),
no seeds are set during the training process.}

\item{attach_data}{(\emph{logical}) if \code{TRUE}, a copy of the training
data will be attached to the output. This is helpful if you
plan on using functions like \link{orsf_pd_oob} or \link{orsf_summarize_uni}
to interpret the forest using its training data. Default is \code{TRUE}.}

\item{no_fit}{(\emph{logical}) if \code{TRUE}, model fitting steps are defined and
saved, but training is not initiated. The object returned can be
directly submitted to \code{orsf_train()} so long as \code{attach_data} is \code{TRUE}.}

\item{...}{Further arguments passed to or from other methods (not currently used).}

\item{object}{an untrained 'aorsf' object, created by setting
\code{no_fit = TRUE} in \code{orsf()}.}
}
\value{
an accelerated oblique RSF object (\code{aorsf})
}
\description{
Fit an oblique random survival forest
}
\details{
This function is based on and highly similar to the \code{ORSF} function
in the \code{obliqueRSF} R package. The primary difference is that this
function runs much faster. The speed increase is attributable to better
management of memory (i.e., no unnecessary copies of inputs) and using
a Newton Raphson scoring algorithm to identify linear combinations of
inputs rather than performing penalized regression using routines in
\code{glmnet}.The modified Newton Raphson scoring algorithm that this
function applies is an adaptation of the C++ routine developed by
Terry M. Therneau that fits Cox proportional hazards models
(see \code{\link[survival:coxph]{survival::coxph()}} and more specifically \code{\link[survival:agreg.fit]{survival::coxph.fit()}}).
}
\section{Details on inputs}{


\strong{formula}:
\itemize{
\item The response in \code{formula} can be a survival
object as returned by the \link[survival:Surv]{Surv} function,
but can also just be the time and status variables. I.e.,
\code{Surv(time, status) ~ .} works just like \code{time + status ~ .}
\item A \code{.} symbol on the right hand side is short-hand for using all
variables in \code{data} (omitting those on the left hand side of
\code{formula}) as predictors.
\item The order of variables in the left hand side matters. i.e.,
writing \code{status + time ~ .} will make \code{orsf} assume your
\code{status} variable is actually the \code{time} variable.
}

\strong{mtry}:

The \code{mtry} parameter may be temporarily reduced to ensure there
are at least 2 events per predictor variable. This occurs when using
\link{orsf_control_cph} because coefficients in the Newton Raphson scoring
algorithm may become unstable when the number of covariates is
greater than or equal to the number of events. This reduction does not
occur when using \link{orsf_control_net}.

\strong{oobag_fun}:

If \code{oobag_fun} is specified, it will be used in to compute negation
importance or permutation importance, but it will not have any role
for ANOVA importance.
}

\section{What is an oblique decision tree?}{


Decision trees are developed by splitting a set of training data into two
new subsets, with the goal of having more similarity within the new subsets
than between them. This splitting process is repeated on the resulting
subsets of data until a stopping criterion is met. When the new subsets of
data are formed based on a single predictor, the decision tree is said to
be axis-based because the splits of the data appear perpendicular to the
axis of the predictor. When linear combinations of variables are used
instead of a single variable, the tree is oblique because the splits of
the data are neither parallel nor at a right angle to the axis

\emph{Figure} : Decision trees for classification with axis-based splitting
(left) and oblique splitting (right). Cases are orange squares; controls
are purple circles. Both trees partition the predictor space defined by
variables X1 and X2, but the oblique splits do a better job of separating
the two classes.

\if{html}{\figure{tree_axis_v_oblique.png}{options: width=95\%}}
}

\section{What is a random forest?}{


Random forests are collections of de-correlated decision trees.
Predictions from each tree are aggregated to make an ensemble
prediction for the forest. For more details, see Breiman at el, 2001.
}

\section{Training, out-of-bag error, and testing}{


In random forests, each tree is grown with a bootstrapped version of
the training set. Because bootstrap samples are selected with replacement,
each bootstrapped training set contains about two-thirds of instances in
the original training set. The 'out-of-bag' data are instances that are
\emph{not} in the bootstrapped training set. Each tree in the random forest
can make predictions for its out-of-bag data, and the out-of-bag
predictions can be aggregated to make an ensemble out-of-bag prediction.
Since the out-of-bag data are not used to grow the tree, the accuracy of
the ensemble out-of-bag predictions approximate the generalization error
of the random forest. Generalization error refers to the error of a
random forest's predictions when it is applied to predict outcomes for
data that were not used to train it, i.e., testing data.
}

\section{Missing data}{


Data passed to aorsf functions are not allowed to have missing values.
A user should impute missing values using an R package with that purpose,
such as \code{recipes} or \code{mlr3pipelines}.
}

\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(329730)

library(aorsf)
library(survival)
library(tidymodels)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## -- Attaching packages --------------------------------- tidymodels 0.2.0 --

## v broom        1.0.0     v recipes      1.0.1
## v dials        0.1.1     v rsample      0.1.1
## v dplyr        1.0.9     v tibble       3.1.7
## v ggplot2      3.3.6     v tidyr        1.2.0
## v infer        1.0.0     v tune         0.2.0
## v modeldata    0.1.1     v workflows    0.2.6
## v parsnip      1.0.0     v workflowsets 0.2.1
## v purrr        0.3.4     v yardstick    0.0.9

## -- Conflicts ------------------------------------ tidymodels_conflicts() --
## x dplyr::between()      masks aorsf::between()
## x purrr::discard()      masks scales::discard()
## x dplyr::filter()       masks stats::filter()
## x dplyr::first()        masks aorsf::first()
## x purrr::is_null()      masks testthat::is_null()
## x recipes::is_trained() masks aorsf::is_trained()
## x dplyr::lag()          masks stats::lag()
## x dplyr::last()         masks aorsf::last()
## x tidyr::matches()      masks rsample::matches(), dplyr::matches(), testthat::matches()
## x recipes::step()       masks stats::step()
## x purrr::transpose()    masks aorsf::transpose()
## * Dig deeper into tidy modeling with R at https://www.tmwr.org
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(tidyverse)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## -- Attaching packages ---------------------------------- tidyverse 1.3.1 --

## v readr   2.1.2     v forcats 0.5.1
## v stringr 1.4.0

## -- Conflicts ------------------------------------- tidyverse_conflicts() --
## x dplyr::between()       masks aorsf::between()
## x readr::col_factor()    masks scales::col_factor()
## x purrr::discard()       masks scales::discard()
## x readr::edition_get()   masks testthat::edition_get()
## x dplyr::filter()        masks stats::filter()
## x dplyr::first()         masks aorsf::first()
## x stringr::fixed()       masks recipes::fixed()
## x purrr::is_null()       masks testthat::is_null()
## x dplyr::lag()           masks stats::lag()
## x dplyr::last()          masks aorsf::last()
## x readr::local_edition() masks testthat::local_edition()
## x tidyr::matches()       masks rsample::matches(), dplyr::matches(), testthat::matches()
## x readr::spec()          masks yardstick::spec()
## x purrr::transpose()     masks aorsf::transpose()
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(randomForestSRC)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## 
##  randomForestSRC 3.0.2 
##  
##  Type rfsrc.news() to see new features, changes, and bug fixes. 
## 

## 
## Attaching package: 'randomForestSRC'

## The following object is masked from 'package:tune':
## 
##     tune

## The following object is masked from 'package:purrr':
## 
##     partial

## The following object is masked from 'package:parsnip':
## 
##     tune
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(ranger)
library(riskRegression)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## riskRegression version 2022.03.22
}\if{html}{\out{</div>}}

The entry-point into \code{aorsf} is the standard call to \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit <- orsf(pbc_orsf, Surv(time, status) ~ . - id)
}\if{html}{\out{</div>}}

printing \code{fit} provides quick descriptive summaries:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 24
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}
\subsection{Model control}{

For these examples we will make use of the \code{orsf_control_} functions to
build and compare models based on their out-of-bag predictions. We will
also standardize the out-of-bag samples using the input argument
\code{tree_seeds}
\subsection{Accelerated linear combinations}{

The accelerated ORSF ensemble is the default because it has a nice
balance of computational speed and prediction accuracy. It runs a single
iteration of Newton Raphson scoring on the Cox partial likelihood
function to find linear combinations of predictors.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_accel <- orsf(pbc_orsf, 
                  control = orsf_control_fast(),
                  formula = Surv(time, status) ~ . - id,
                  tree_seeds = 1:500)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with Cox regression}{

\code{orsf_control_cph} runs Cox regression in each non-terminal node of each
survival tree, using the regression coefficients to create linear
combinations of predictors:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_cph <- orsf(pbc_orsf, 
                control = orsf_control_cph(),
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 1:500)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with penalized cox regression}{

\code{orsf_control_net} runs penalized Cox regression in each non-terminal
node of each survival tree, using the regression coefficients to create
linear combinations of predictors. This can be really helpful if you
want to do feature selection within the node, but it is a lot slower
than the other options.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_net <- orsf(pbc_orsf, 
                # select 3 predictors out of 5 to be used in
                # each linear combination of predictors.
                control = orsf_control_net(df_target = 3),
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 1:500)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with your own function}{

Let’s make two customized functions to identify linear combinations of
predictors.
\itemize{
\item The first uses random coefficients

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_rando <- function(x_node, y_node, w_node)\{
 matrix(runif(ncol(x_node)), ncol=1) 
\}
}\if{html}{\out{</div>}}
\item The second derives coefficients from principal component analysis.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_pca <- function(x_node, y_node, w_node) \{ 

 # estimate two principal components.
 pca <- stats::prcomp(x_node, rank. = 2)
 # use the second principal component to split the node
 pca$rotation[, 2L, drop = FALSE]

\}
}\if{html}{\out{</div>}}
}

We can plug these functions into \code{orsf_control_custom()}, and then pass
the result into \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_rando <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  control = orsf_control_custom(beta_fun = f_rando),
                  tree_seeds = 1:500)

fit_pca <- orsf(pbc_orsf,
                Surv(time, status) ~ . - id,
                control = orsf_control_custom(beta_fun = f_pca),
                tree_seeds = 1:500)
}\if{html}{\out{</div>}}

So which fit seems to work best in this example? Let’s find out by
evaluating the out-of-bag survival predictions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{risk_preds <- list(
 accel = 1 - fit_accel$pred_oobag,
 cph   = 1 - fit_cph$pred_oobag,
 net   = 1 - fit_net$pred_oobag,
 rando = 1 - fit_rando$pred_oobag,
 pca   = 1 - fit_pca$pred_oobag
)

sc <- Score(object = risk_preds, 
            formula = Surv(time, status) ~ 1, 
            data = pbc_orsf, 
            summary = 'IPA',
            times = fit_accel$pred_horizon)
}\if{html}{\out{</div>}}

The AUC values, from highest to lowest:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$AUC$score[order(-AUC)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##     model times       AUC         se     lower     upper
##    <fctr> <num>     <num>      <num>     <num>     <num>
## 1:    pca  1788 0.9140542 0.01962854 0.8755830 0.9525254
## 2:  accel  1788 0.9109875 0.02174062 0.8683766 0.9535983
## 3:    net  1788 0.9081592 0.02161885 0.8657871 0.9505314
## 4:    cph  1788 0.9072690 0.02122339 0.8656719 0.9488660
## 5:  rando  1788 0.8681493 0.02418770 0.8207423 0.9155564
}\if{html}{\out{</div>}}

And the indices of prediction accuracy:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$Brier$score[order(-IPA), .(model, times, IPA)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##         model times       IPA
##        <fctr> <num>     <num>
## 1:      accel  1788 0.4893270
## 2:        net  1788 0.4842680
## 3:        cph  1788 0.4689101
## 4:        pca  1788 0.4238639
## 5:      rando  1788 0.3395780
## 6: Null model  1788 0.0000000
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item the PCA approach has the highest discrimination, showing that you
can do very well with just a two line custom function.
\item the accelerated ORSF has the highest index of prediction accuracy
\item the random coefficients generally don’t do that well.
}
}

}

\subsection{tidymodels}{

This example uses \code{tidymodels} functions but stops short of using an
official \code{tidymodels} workflow. I am working on getting \code{aorsf} pulled
into the \code{censored} package and I will update this with real workflows
if that happens!
\subsection{Comparing ORSF with other learners}{

Start with a recipe to pre-process data

\if{html}{\out{<div class="sourceCode r">}}\preformatted{imputer <- recipe(pbc_orsf, formula = time + status ~ .) \%>\% 
 step_impute_mean(all_numeric_predictors()) \%>\%
 step_impute_mode(all_nominal_predictors()) 
}\if{html}{\out{</div>}}

Next create a 10-fold cross validation object and pre-process the data:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 10-fold cross validation; make a container for the pre-processed data
analyses <- vfold_cv(data = pbc_orsf, v = 10) \%>\%
 mutate(recipe = map(splits, ~prep(imputer, training = training(.x))),
        train = map(recipe, juice),
        test = map2(splits, recipe, ~bake(.y, new_data = testing(.x))))

analyses
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## #  10-fold cross-validation 
## # A tibble: 10 x 5
##    splits           id     recipe   train               test              
##    <list>           <chr>  <list>   <list>              <list>            
##  1 <split [248/28]> Fold01 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  2 <split [248/28]> Fold02 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  3 <split [248/28]> Fold03 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  4 <split [248/28]> Fold04 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  5 <split [248/28]> Fold05 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  6 <split [248/28]> Fold06 <recipe> <tibble [248 x 20]> <tibble [28 x 20]>
##  7 <split [249/27]> Fold07 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
##  8 <split [249/27]> Fold08 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
##  9 <split [249/27]> Fold09 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
## 10 <split [249/27]> Fold10 <recipe> <tibble [249 x 20]> <tibble [27 x 20]>
}\if{html}{\out{</div>}}

Define functions for a ‘workflow’ with \code{randomForestSRC}, \code{ranger}, and
\code{aorsf}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rfsrc_wf <- function(train, test, pred_horizon)\{
 
 # rfsrc does not like tibbles, so cast input data into data.frames
 train <- as.data.frame(train)
 test <- as.data.frame(test)
 
 rfsrc(formula = Surv(time, status) ~ ., data = train) \%>\% 
  predictRisk(newdata = test, times = pred_horizon) \%>\% 
  as.numeric()
 
\}

ranger_wf <- function(train, test, pred_horizon)\{
 
 ranger(Surv(time, status) ~ ., data = train) \%>\% 
  predictRisk(newdata = test, times = pred_horizon) \%>\% 
  as.numeric()
 
\}

aorsf_wf <- function(train, test, pred_horizon)\{
 
 train \%>\% 
  orsf(Surv(time, status) ~ .,) \%>\% 
  predict(new_data = test, pred_horizon = pred_horizon) \%>\% 
  as.numeric()
 
\}
}\if{html}{\out{</div>}}

Run the ‘workflows’ on each fold:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 5 year risk prediction
ph <- 365.25 * 5

results <- analyses \%>\% 
 transmute(test, 
           pred_aorsf = map2(train, test, aorsf_wf, pred_horizon = ph),
           pred_rfsrc = map2(train, test, rfsrc_wf, pred_horizon = ph),
           pred_ranger = map2(train, test, ranger_wf, pred_horizon = ph))
}\if{html}{\out{</div>}}

Next unnest each column to get back a \code{tibble} with all of the testing
data and predictions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{results <- results \%>\% 
 unnest(everything())

glimpse(results)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## Rows: 276
## Columns: 23
## $ id          <int> 7, 36, 52, 65, 67, 72, 79, 88, 89, 100, 127, 154, 161~
## $ trt         <fct> placebo, placebo, d_penicill_main, d_penicill_main, p~
## $ age         <dbl> 55.53457, 56.41068, 50.54073, 40.20260, 51.28816, 32.~
## $ sex         <fct> f, f, m, f, f, f, f, f, f, m, f, m, f, m, f, f, f, f,~
## $ ascites     <fct> 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,~
## $ hepato      <fct> 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 1,~
## $ spiders     <fct> 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1,~
## $ edema       <fct> 0, 0, 0, 0, 0, 0, 0, 0.5, 0, 0, 0, 1, 0, 0, 0, 0, 0, ~
## $ bili        <dbl> 1.0, 0.3, 6.0, 1.2, 1.1, 0.5, 0.8, 0.6, 2.0, 2.3, 0.5~
## $ chol        <int> 322, 172, 614, 256, 466, 320, 315, 296, 408, 178, 268~
## $ albumin     <dbl> 4.09, 3.39, 3.70, 3.60, 3.91, 3.54, 4.24, 4.06, 3.65,~
## $ copper      <int> 52, 18, 158, 74, 84, 51, 13, 37, 50, 145, 9, 225, 4, ~
## $ alk.phos    <dbl> 824.0, 558.0, 5084.4, 724.0, 1787.0, 1243.0, 1637.0, ~
## $ ast         <dbl> 60.45, 71.30, 206.40, 141.05, 328.60, 122.45, 170.50,~
## $ trig        <int> 213, 96, 93, 108, 185, 80, 70, 83, 98, 122, 95, 75, 1~
## $ platelet    <int> 204, 311, 362, 430, 261, 225, 426, 442, 200, 119, 453~
## $ protime     <dbl> 9.7, 10.6, 10.6, 10.0, 10.0, 10.0, 10.9, 12.0, 11.4, ~
## $ stage       <ord> 3, 2, 1, 1, 3, 3, 3, 3, 2, 4, 2, 3, 2, 4, 4, 3, 3, 3,~
## $ time        <int> 1832, 3611, 2386, 3992, 2769, 4184, 3707, 2452, 1741,~
## $ status      <dbl> 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0,~
## $ pred_aorsf  <dbl> 0.14427839, 0.01870334, 0.51377784, 0.04334777, 0.219~
## $ pred_rfsrc  <dbl> 0.14732022, 0.03156261, 0.55773866, 0.06502143, 0.161~
## $ pred_ranger <dbl> 0.14870388, 0.01662689, 0.57519043, 0.05876898, 0.148~
}\if{html}{\out{</div>}}

And finish by aggregating the predictions and computing performance in
the testing data. Note that I am computing one statistic for all
predictions instead of computing one statistic for each fold. This
approach is fine when you have smaller testing sets and/or small event
counts.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Score(
 object = list(aorsf = results$pred_aorsf,
               rfsrc = results$pred_rfsrc,
               ranger = results$pred_ranger),
 formula = Surv(time, status) ~ 1, 
 data = results, 
 summary = 'IPA',
 times = ph
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## 
## Metric AUC:
## 
## Results by model:
## 
##     model times    AUC  lower  upper
##    <fctr> <num> <char> <char> <char>
## 1:  aorsf  1826   91.2   87.0   95.4
## 2:  rfsrc  1826   90.5   86.4   94.6
## 3: ranger  1826   90.3   86.2   94.5
## 
## Results of model comparisons:
## 
##    times  model reference delta.AUC  lower  upper     p
##    <num> <fctr>    <fctr>    <char> <char> <char> <num>
## 1:  1826  rfsrc     aorsf      -0.7   -2.3    0.8   0.4
## 2:  1826 ranger     aorsf      -0.9   -2.3    0.5   0.2
## 3:  1826 ranger     rfsrc      -0.1   -1.1    0.9   0.8

## 
## NOTE: Values are multiplied by 100 and given in \%.

## NOTE: The higher AUC the better.

## 
## Metric Brier:
## 
## Results by model:
## 
##         model   times  Brier  lower  upper    IPA
##        <fctr>   <num> <char> <char> <char> <char>
## 1: Null model 1826.25   20.5   18.1   22.9    0.0
## 2:      aorsf 1826.25   10.5    8.3   12.7   48.7
## 3:      rfsrc 1826.25   11.6    9.5   13.6   43.6
## 4:     ranger 1826.25   11.5    9.5   13.6   43.6
## 
## Results of model comparisons:
## 
##      times  model  reference delta.Brier  lower  upper            p
##      <num> <fctr>     <fctr>      <char> <char> <char>        <num>
## 1: 1826.25  aorsf Null model       -10.0  -12.6   -7.3 1.430444e-13
## 2: 1826.25  rfsrc Null model        -8.9  -11.2   -6.7 5.508455e-15
## 3: 1826.25 ranger Null model        -8.9  -11.3   -6.6 5.511591e-14
## 4: 1826.25  rfsrc      aorsf         1.1    0.2    1.9 1.141214e-02
## 5: 1826.25 ranger      aorsf         1.0    0.3    1.7 3.742797e-03
## 6: 1826.25 ranger      rfsrc        -0.0   -0.5    0.5 9.730271e-01

## 
## NOTE: Values are multiplied by 100 and given in \%.

## NOTE: The lower Brier the better, the higher IPA the better.
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item \code{aorsf} obtained slightly higher discrimination (AUC)
\item \code{aorsf} obtained higher index of prediction accuracy (IPA)
\item Way to go, \code{aorsf}
}
}

}

\subsection{mlr3 pipelines}{

this is on hold while \code{mlr3proba} goes through major updates.
}
}

\references{
Harrell FE, Califf RM, Pryor DB, Lee KL, Rosati RA. Evaluating the Yield of Medical Tests. \emph{JAMA} 1982; 247(18):2543-2546. DOI: 10.1001/jama.1982.03320430047030

Breiman L. Random forests. \emph{Machine learning} 2001 Oct; 45(1):5-32. DOI: 10.1023/A:1010933404324

Ishwaran H, Kogalur UB, Blackstone EH, Lauer MS. Random survival forests. \emph{Annals of applied statistics} 2008 Sep; 2(3):841-60. DOI: 10.1214/08-AOAS169

Jaeger BC, Long DL, Long DM, Sims M, Szychowski JM, Min YI, Mcclure LA, Howard G, Simon N. Oblique random survival forests. \emph{Annals of applied statistics} 2019 Sep; 13(3):1847-83. DOI: 10.1214/19-AOAS1261

Jaeger BC, Welden S, Lenoir K, Speiser JL, Segar MW, Pandey A, Pajewski NM. Accelerated and interpretable oblique random survival forests. \emph{arXiv e-prints} 2022 Aug; arXiv-2208. URL: https://arxiv.org/abs/2208.01129
}
