% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rope.R
\name{rope}
\alias{rope}
\alias{rope.numeric}
\alias{rope.stanreg}
\alias{rope.brmsfit}
\title{Region of Practical Equivalence (ROPE)}
\usage{
rope(x, ...)

\method{rope}{numeric}(x, range = "default", ci = 0.9,
  verbose = TRUE, ...)

\method{rope}{stanreg}(x, range = "default", ci = 0.9,
  effects = c("fixed", "random", "all"), parameters = NULL,
  verbose = TRUE, ...)

\method{rope}{brmsfit}(x, range = "default", ci = 0.9,
  effects = c("fixed", "random", "all"), component = c("conditional",
  "zi", "zero_inflated", "all"), parameters = NULL, verbose = TRUE,
  ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a \code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{range}{ROPE's lower and higher bounds. Should be a vector of length two (e.g., \code{c(-0.1, 0.1)}) or \code{"default"}. If \code{"default"}, the range is set to \code{c(-0.1, 0.1)} if input is a vector, and \code{x +- 0.1*SD(response)} if a Bayesian model is provided.}

\item{ci}{The Credible Interval (CI) probability, corresponding to the proportion of HDI, to use.}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{component}{Should results for all parameters, parameters for the conditional model
or the zero-inflated part of the model be returned? May be abbreviated. Only
applies to \pkg{brms}-models.}
}
\description{
Compute the proportion (in percentage) of the HDI (default to the 90\% HDI) of a posterior distribution that lies within a region of practical equivalence.
}
\details{
Statistically, the probability of a posterior distribution of being
  different from 0 does not make much sense (the probability of it being
  different from a single point being infinite). Therefore, the idea
  underlining ROPE is to let the user define an area around the null value
  enclosing values that are \emph{equivalent to the null} value for practical
  purposes (\cite{Kruschke 2010, 2011, 2014}).
  \cr \cr
  Kruschke (2018) suggests that such null value could be set, by default,
  to the -0.1 to 0.1 range of a standardized parameter (negligible effect
  size according to Cohen, 1988). This could be generalized: For instance,
  for linear models, the ROPE could be set as \code{0 +/- .1 * sd(y)}.
  This ROPE range can be automatically computed for models using the
  \link{rope_range} function.
  \cr \cr
  Kruschke (2010, 2011, 2014) suggests using the proportion of  the 95\%
  (or 90\%, considered more stable) \link[=hdi]{HDI} that falls within the
  ROPE as an index for "null-hypothesis" testing (as understood under the
  Bayesian framework, see \link[=equivalence_test]{equivalence_test}).
}
\examples{
library(bayestestR)

rope(x = rnorm(1000, 0, 0.01), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 0, 1), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 1, 0.01), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 1, 1), ci = c(.90, .95))
\dontrun{
library(rstanarm)
model <- rstanarm::stan_glm(mpg ~ wt + cyl, data = mtcars)
rope(model)
rope(model, ci = c(.90, .95))

library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
rope(model)
rope(model, ci = c(.90, .95))
}

}
\references{
\itemize{
\item Cohen, J. (1988). Statistical power analysis for the behavioural sciences.
\item Kruschke, J. K. (2010). What to believe: Bayesian methods for data analysis. Trends in cognitive sciences, 14(7), 293-300. \doi{10.1016/j.tics.2010.05.001}.
\item Kruschke, J. K. (2011). Bayesian assessment of null values via parameter estimation and model comparison. Perspectives on Psychological Science, 6(3), 299-312. \doi{10.1177/1745691611406925}.
\item Kruschke, J. K. (2014). Doing Bayesian data analysis: A tutorial with R, JAGS, and Stan. Academic Press. \doi{10.1177/2515245918771304}.
\item Kruschke, J. K. (2018). Rejecting or accepting parameter values in Bayesian estimation. Advances in Methods and Practices in Psychological Science, 1(2), 270-280. \doi{10.1177/2515245918771304}.
 }
}
