% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{bhomGP}
\alias{bhomGP}
\title{MCMC sampling for Homoskedastic GP}
\usage{
bhomGP(
  x = NULL,
  y = NULL,
  reps_list = NULL,
  nmcmc = 1000,
  sep = TRUE,
  inits = NULL,
  priors = NULL,
  cov = c("exp2", "matern", "ARD matern"),
  v = 2.5,
  stratergy = c("default", "flat"),
  vecchia = FALSE,
  m = min(25, length(y) - 1),
  ordering = NULL,
  reps = TRUE,
  verb = TRUE,
  cores = 4
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{reps_list}{list object from hetGP::find_reps}

\item{nmcmc}{number of MCMC iterations}

\item{sep}{logical indicating whether to fit isotropic GP (\code{sep = FALSE})
or seperable GP (\code{sep = TRUE})}

\item{inits}{set initial values for hyparameters: \code{theta_y}, \code{g}, \code{mean_y},
\code{scale_y},
Additionally, set initial conditions for tuning:
\itemize{ 
  \item \code{prof_ll}: logical; if \code{prof_ll = TRUE},
  infers tau2_y i.e., scale parameter for homGP.
  \item \code{noise}: logical; if \code{noise = TRUE}, infers nugget
  \code{g} throught M-H for latent noise process.
}}

\item{priors}{hyperparameters for priors and proposals (see details)}

\item{cov}{covariance kernel, either Matern, ARD Matern 
or squared exponential (\code{"exp2"})}

\item{v}{Matern smoothness parameter (only used if \code{cov = "matern"})}

\item{stratergy}{choose initialization stratergy; "default" uses hetGP for 
\code{vecchia = FALSE} settings and sVecchia for \code{vecchia = TRUE}. 
See details.}

\item{vecchia}{logical indicating whether to use Vecchia approximation}

\item{m}{size of Vecchia conditioning sets (only used if 
\code{vecchia = TRUE})}

\item{ordering}{optional ordering for Vecchia approximation, must correspond
to rows of \code{x}, defaults to random, is applied to \code{x}}

\item{reps}{logical; if \code{reps = TRUE} uses Woodbury inference adjusting for
replication of design points and \code{reps = FALSE} does not
use Woodbury inference}

\item{verb}{logical indicating whether to print progress}

\item{cores}{if \code{vecchia = TRUE}, user may specify the number of cores to
use for OpenMP parallelization.  Defaults to \code{min(2, maxcores - 1)} 
where \code{maxcores} is the number of detectable available cores.}
}
\value{
a list of the S3 class \code{bhomgp} or \code{bhomgp_vec} with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{Ylist}: list of all responses observed per location (x)
  \item \code{A}: number of replicates at each location
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{priors}: copy of proposal/prior settings
  \item \code{settings}: setting for predictions using \code{bhetgp} or \code{bhetgp_vec}
  object
  \item \code{g_y}: vector of MCMC samples for \code{g_y}
  \item \code{theta_y}: vector of MCMC samples for \code{theta_y} (length
        scale of mean process)
  \item \code{tau2}: vector of MAP estimates for \code{tau2} (scale 
        parameter of outer layer)
  \item \code{llik_y}: vector of MVN log likelihood of Y for reach Gibbs iteration
  \item \code{time}: computation time in seconds
  \item \code{x_approx}: conditioning set, NN and ordering for \code{vecchia = TRUE}
  \item \code{m}: copy of size of conditioning set for \code{vecchia = TRUE}      
}
}
\description{
Conducts MCMC sampling of hyperparameters for a homGP.
    Separate length scale parameters \code{theta_y} 
    govern the correlation strength of the response. \code{g} governs 
    noise for the noise. \code{tau2_y} control the amplitude of the mean process.
   In Matern covariance, \code{v} governs smoothness.
}
\details{
Maps inputs \code{x} to mean response \code{y}. Utilizes Metropolis Hastings 
    sampling of the length scale and nugget parameters with proposals and priors 
    controlled by \code{priors}. \code{g} is estimated by default but may be specified and fixed.
    When \code{vecchia = TRUE}, all calculations leverage the Vecchia approximation with 
    specified conditioning set size \code{m}. \code{tau2_y} is inferred by default but 
    may be pre-specified and fixed.
    
    NOTE on OpenMP: The Vecchia implementation relies on OpenMP parallelization
    for efficient computation.  This function will produce a warning message 
    if the package was installed without OpenMP (this is the default for 
    CRAN packages installed on Apple machines).  To set up OpenMP 
    parallelization, download the package source code and install 
    using the gcc/g++ compiler.    
    
    Proposals for \code{g} and \code{theta} follow a uniform sliding window 
    scheme, e.g. 
    
    \code{theta_star <- runif(1, l * theta_t / u, u * theta_t / l)}, 
    
    with defaults \code{l = 1} and \code{u = 2} provided in \code{priors}.
    To adjust these, set \code{priors = list(l = new_l, u = new_u)}.    
    Priors on \code{g}, \code{theta_y} follow Gamma 
    distributions with shape parameters (\code{alpha}) and rate parameters 
    (\code{beta}) controlled within the \code{priors} list object.  
    Defaults are
    \itemize{
        \item \code{priors$alpha$theta_lam <- 1.5}
        \item \code{priors$beta$theta_lam <- 4}
        \item \code{priors$alpha$theta_y <- 1.5}
        \item \code{priors$beta$theta_y <- 4}
        \item \code{priors$alpha$g <- 1.5}
        \item \code{priors$beta$g <- 4}
    }
    
    \code{tau2_y} is not sampled; rather directly inferred
    under conjugate Inverse Gamma prior with shape (\code{alpha}) and scale 
    parameters (\code{beta}) within the \code{priors} list object
    \itemize{       
        \item \code{priors$a$tau2_y <- 10}
        \item \code{priors$a$tau2_y <- 4}
    }
    These priors are designed for \code{x} scaled to 
    [0, 1] and \code{y} having mean \code{mean_y}.  These may be 
    adjusted using the \code{priors} input.
    
    Initial values for \code{theta_y}, and \code{g} may be
    specified by the user. If no initial values are specified, \code{stratergy}
    will determine the initialization method. \code{stratergy = "default"} 
    leverages mleHomGP for initial values of hyper-parameters if 
    \code{vecchia = FALSE} and scaled vecchia approach if \code{vecchia = TRUE}.
    
    For scaled Vecchia code from 
    \url{https://github.com/katzfuss-group/scaledVecchia/blob/master/vecchia_scaled.R}
    is used to fit a vecchia approximated GP to (x, y). A script is leveraged
    internally within this package that fits this method. 
    
    Optionally, choose stratergy = "flat" which will start at 
    uninformative initial values or specify initial values. 
    
    The output object of class \code{bhomgp} or \code{bhomgp_vec} is designed for 
    use with \code{trim}, \code{predict}, and \code{plot}.
}
\examples{
fx <- function(x){
result <- (6 * x - 2)^2* sin(12 * x - 4)
}

# Training data
r <- 10
xn <- seq(0, 1, length = 25)
x <- rep(xn, r)

f <- fx(x) 
y <- f + rnorm(length(x))

# Testing data
xx <- seq(0, 1, length = 100)
yy <- fx(xx)

# Example 1: Full model, no Vecchia
fit <- bhomGP(x, y, nmcmc = 100)
fit <- trim(fit, 50, 10)
fit <- predict(fit, xx)
plot(fit) # can run with trace = TRUE to view trace plots

# Example 2: Vecchia approximated model
fit <- bhomGP(x, y, nmcmc = 100, vecchia = TRUE, m = 5)
fit <- trim(fit, 50, 10)
fit <- predict(fit, xx, vecchia = TRUE)
plot(fit)

}
\references{
Binois, Mickael, Robert B. Gramacy, and Mike Ludkovski. "Practical heteroscedastic Gaussian process 
modeling for large simulation experiments." Journal of Computational and Graphical 
Statistics 27.4 (2018): 808-821.

Katzfuss, Matthias, Joseph Guinness, and Earl Lawrence. "Scaled Vecchia approximation for 
fast computer-model emulation." SIAM/ASA Journal on Uncertainty Quantification 10.2 (2022): 537-554.

Sauer, Annie Elizabeth. "Deep Gaussian process surrogates for computer experiments." (2023).
}
