% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CAR_INLA.R
\name{CAR_INLA}
\alias{CAR_INLA}
\title{Fit a (scalable) spatial Poisson mixed model to areal count data, where several CAR prior distributions can be specified for the spatial random effect.}
\usage{
CAR_INLA(
  carto = NULL,
  ID.area = NULL,
  ID.group = NULL,
  O = NULL,
  E = NULL,
  X = NULL,
  confounding = NULL,
  W = NULL,
  prior = "Leroux",
  model = "partition",
  k = 0,
  strategy = "simplified.laplace",
  PCpriors = FALSE,
  seed = NULL,
  n.sample = 1000,
  compute.intercept = FALSE,
  compute.DIC = TRUE,
  save.models = FALSE,
  plan = "sequential",
  workers = NULL,
  merge.strategy = "original"
)
}
\arguments{
\item{carto}{object of class \code{SpatialPolygonsDataFrame} or \code{sf}.
This object must contain at least the target variables of interest specified in the arguments \code{ID.area}, \code{O} and \code{E}.}

\item{ID.area}{character; name of the variable that contains the IDs of spatial areal units.}

\item{ID.group}{character; name of the variable that contains the IDs of the spatial partition (grouping variable).
Only required if \code{model="partition"}.}

\item{O}{character; name of the variable that contains the observed number of disease cases for each areal units.}

\item{E}{character; name of the variable that contains either the expected number of disease cases or the population at risk for each areal unit.}

\item{X}{a character vector containing the names of the covariates within the \code{carto} object to be included in the model as fixed effects,
or a matrix object playing the role of the fixed effects design matrix. For the latter case, the row names must match with the IDs of the spatial units defined by the \code{ID.area} variable.
If \code{X=NULL} (default), only a global intercept is included in the model as fixed effect.}

\item{confounding}{one of either \code{NULL}, \code{"restricted"} (restricted regression) or \code{"constraints"} (orthogonal constraints), which specifies the estimation method used to alleviate spatial confounding between fixed and random effects.
If only an intercept is considered in the model (\code{X=NULL}), the default value \code{confounding=NULL} will be set.
At the moment, it only works for the \emph{Global model} (specified through the \code{model="global"} argument).}

\item{W}{optional argument with the binary adjacency matrix of the spatial areal units.  If \code{NULL} (default), this object is computed from the \code{carto} argument (two areas are considered as neighbours if they share a common border).}

\item{prior}{one of either \code{"Leroux"} (default), \code{"intrinsic"}, \code{"BYM"} or \code{"BYM2"},
which specifies the prior distribution considered for the spatial random effect.}

\item{model}{one of either \code{"global"} or \code{"partition"} (default), which specifies the \emph{Global model}
or one of the scalable model proposal's (\emph{Disjoint model} and \emph{k-order neighbourhood model}, respectively).}

\item{k}{numeric value with the neighbourhood order used for the partition model. Usually k=2 or 3 is enough to get good results.
If k=0 (default) the \emph{Disjoint model} is considered. Only required if \code{model="partition"}.}

\item{strategy}{one of either \code{"gaussian"}, \code{"simplified.laplace"} (default), \code{"laplace"} or \code{"adaptive"},
which specifies the approximation strategy considered in the \code{inla} function.}

\item{PCpriors}{logical value (default \code{FALSE}); if \code{TRUE} then penalised complexity (PC) priors are used for the precision parameter of the spatial random effect.
It only works if arguments \code{prior="intrinsic"} or \code{prior="BYM2"} are specified.}

\item{seed}{numeric (default \code{NULL}); control the RNG of the \code{inla.qsample} function. See \code{help(inla.qsample)} for further information.}

\item{n.sample}{numeric; number of samples to generate from the posterior marginal distribution of the risks. Default to 1000.}

\item{compute.intercept}{logical value (default \code{FALSE}); if \code{TRUE} then the overall log-risk \eqn{\alpha} is computed.
It only works if \code{k=0} argument (\emph{Disjoint model}) is specified. CAUTION: This method might be very time consuming.}

\item{compute.DIC}{logical value; if \code{TRUE} (default) then approximate values of the Deviance Information Criterion (DIC) and Watanabe-Akaike Information Criterion (WAIC) are computed.}

\item{save.models}{logical value (default \code{FALSE}); if \code{TRUE} then a list with all the \code{inla} submodels is saved in '/temp/' folder, which can be used as input argument for the \code{\link{mergeINLA}} function.}

\item{plan}{one of either \code{"sequential"} or \code{"cluster"}, which specifies the computation strategy used for model fitting using the 'future' package.
If \code{plan="sequential"} (default) the models are fitted sequentially and in the current R session (local machine). If \code{plan="cluster"} the models are fitted in parallel on external R sessions (local machine) or distributed in remote computing nodes.}

\item{workers}{character or vector (default \code{NULL}) containing the identifications of the local or remote workers where the models are going to be processed. Only required if \code{plan="cluster"}.}

\item{merge.strategy}{one of either \code{"mixture"} or \code{"original"} (default), which specifies the merging strategy to compute posterior marginal estimates of relative risks. See \code{\link{mergeINLA}} for further details.}
}
\value{
This function returns an object of class \code{inla}. See the \code{\link{mergeINLA}} function for details.
}
\description{
Fit a spatial Poisson mixed model to areal count data. The linear predictor is modelled as \deqn{\log{r_{i}}=\alpha+\mathbf{x_i}^{'}\mathbf{\beta} + \xi_i, \quad \mbox{for} \quad i=1,\ldots,n;}
where \eqn{\alpha} is a global intercept, \eqn{\mathbf{x_i}^{'}=(x_{i1},\ldots,x_{ip})} is a p-vector of standardized covariates in the i-th area,
\eqn{\mathbf{\beta}=(\beta_1,\ldots,\beta_p)} is the p-vector of fixed effects coefficients, and \eqn{\xi_i} is a spatially structured random effect.
Several conditional autoregressive (CAR) prior distributions can be specified for the spatial random effect, such as the intrinsic CAR prior \insertCite{besag1991}{bigDM}, the convolution or BYM prior \insertCite{besag1991}{bigDM},
the CAR prior proposed by \insertCite{leroux1999estimation;textual}{bigDM}, and the reparameterization of the BYM model given by \insertCite{dean2001detecting;textual}{bigDM} named BYM2 \insertCite{riebler2016intuitive}{bigDM}.

If covariates are included in the model, two different approaches can be used to address the potential confounding issues between the fixed effects and the spatial random effects of the model: restricted regression and the use of orthogonality constraints.
At the moment, only continuous covariates can be included in the model as potential risk factors, which are automatically standardized before fitting the model. See \insertCite{adin2021alleviating;textual}{bigDM} for further details.
\cr\cr
Three main modeling approaches can be considered:
\itemize{
\item the usual model with a global spatial random effect whose dependence structure is based on the whole neighbourhood graph of the areal units (\code{model="global"} argument)
\item a Disjoint model based on a partition of the whole spatial domain where independent spatial CAR models are simultaneously fitted in each partition (\code{model="partition"} and \code{k=0} arguments)
\item a modelling approach where \emph{k}-order neighbours are added to each partition to avoid border effects in the Disjoint model (\code{model="partition"} and \code{k>0} arguments).
}
For both the Disjoint and k-order neighbour models, parallel or distributed computation strategies can be performed to speed up computations by using the 'future' package \insertCite{bengtsson2020unifying}{bigDM}.

Inference is conducted in a fully Bayesian setting using the integrated nested Laplace approximation (INLA; \insertCite{rue2009approximate;textual}{bigDM}) technique through the R-INLA package (\url{https://www.r-inla.org/}).
For the scalable model proposals \insertCite{orozco2020}{bigDM}, approximate values of the Deviance Information Criterion (DIC) and Watanabe-Akaike Information Criterion (WAIC) can also be computed.
}
\details{
For a full model specification and further details see the vignettes accompanying this package.
}
\examples{
\dontrun{

if(require("INLA", quietly=TRUE)){

  ## Load the Spain colorectal cancer mortality data ##
  data(Carto_SpainMUN)

  ## Global model with a Leroux CAR prior distribution ##
  Global <- CAR_INLA(carto=Carto_SpainMUN, ID.area="ID", O="obs", E="exp",
                     prior="Leroux", model="global", strategy="gaussian")

  summary(Global)

  ## Disjoint model with a Leroux CAR prior distribution  ##
  ## using 4 local clusters to fit the models in parallel ##
  Disjoint <- CAR_INLA(carto=Carto_SpainMUN, ID.area="ID", ID.group="region", O="obs", E="exp",
                       prior="Leroux", model="partition", k=0, strategy="gaussian",
                       plan="cluster", workers=rep("localhost",4))
  summary(Disjoint)

  ## 1st-order neighbourhood model with a Leroux CAR prior distribution ##
  ## using 4 local clusters to fit the models in parallel               ##
  order1 <- CAR_INLA(carto=Carto_SpainMUN, ID.area="ID", ID.group="region", O="obs", E="exp",
                     prior="Leroux", model="partition", k=1, strategy="gaussian",
                     plan="cluster", workers=rep("localhost",4))
  summary(order1)

  ## 2nd-order neighbourhood model with a Leroux CAR prior distribution ##
  ## using 4 local clusters to fit the models in parallel               ##
  order2 <- CAR_INLA(carto=Carto_SpainMUN, ID.area="ID", ID.group="region", O="obs", E="exp",
                     prior="Leroux", model="partition", k=2, strategy="gaussian",
                     plan="cluster", workers=rep("localhost",4))
  summary(order2)
}
}

}
\references{
\insertRef{adin2021alleviating}{bigDM}

\insertRef{bengtsson2020unifying}{bigDM}

\insertRef{besag1991}{bigDM}

\insertRef{dean2001detecting}{bigDM}

\insertRef{leroux1999estimation}{bigDM}

\insertRef{riebler2016intuitive}{bigDM}

\insertRef{rue2009approximate}{bigDM}

\insertRef{orozco2020}{bigDM}
}
