\name{plot.biwavelet}
\alias{plot.biwavelet}
\title{
Plot \code{biwavelet} objects
}
\description{
Plot \code{biwavelet} objects such as the cwt, cross-wavelet and wavelet coherence
}
\usage{
\method{plot}{biwavelet} (x, ncol = 64, xlab = "Time", ylab = "Period",
               tol = 0.95, plot.cb = FALSE, plot.phase = FALSE, 
               type=c("power.corr.norm", "power.corr", "power.norm", "power", 
               "wavelet", "phase"), 
               plot.coi = TRUE, lwd.coi=4, col.coi="white", lty.coi=2,
               plot.sig = TRUE, lwd.sig=4, col.sig="black", lty.sig=1,
               bw = FALSE, legend.loc = NULL, legend.horiz = FALSE, 
               arrow.size = 0.08, arrow.lwd = 2, arrow.cutoff = 0.9, arrow.col="black",
               xlim = NULL, ylim = NULL, xaxt = "s", yaxt = "s", 
               form='\%Y', \dots)
}
\arguments{
  \item{x}{
\code{biwavelet} object generated by \code{wt}, \code{xwt}, or \code{wtc}.
}
  \item{ncol}{
number of colors to use. Default is 64.
}
  \item{xlab}{
xlabel of the figure. Default is "Time"
}
  \item{ylab}{
ylabel of the figure. Default is "Period"
}
  \item{tol}{
tolerance level for significance contours. Sigificance contours will be drawn around all regions
of the spectrum where \code{spectrum/percentile >= tol}. Default is \code{0.95}. If strict
\code{i^{th}} percentile regions are desired, then set \code{tol=1}.
}
  \item{plot.cb}{
plot color bar if TRUE. Default is FALSE.
}
  \item{plot.phase}{
Plot phases with black arrows. Default is FALSE.
}
  \item{type}{
type of plot to create. Can be \code{power} to plot the power, \code{power.corr} to plot
the bias-corrected power, \code{power.norm} to plot the power normalized by the variance, 
\code{power.corr.norm} to plot the bias-corrected power normalized by the variance, 
\code{wavelet} to plot thewavelet coefficients, or \code{phase} to plot the phase. 
Default is \code{power.corr.norm}.
}
  \item{plot.coi}{
plot cone of influence (COI) if TRUE. Default is TRUE. Areas that fall outside 
of the COI can be affected by edge effects.
}
  \item{lwd.coi}{
Line width of COI. Default is 4.
}
  \item{col.coi}{
Color of cone of COI. Default is \code{white}.
}
  \item{lty.coi}{
Line type of COI
}
  \item{plot.sig}{
plot contours for significance if TRUE. Default is TRUE.
}
  \item{lwd.sig}{
Line width of significance contours. Default is 4.
}
  \item{col.sig}{
Color of significance contours. Default is \code{black}.
}
  \item{lty.sig}{
Line type of significance contours. Default is 1.
}
  \item{bw}{
plot in black and white if TRUE. Default is FALSE.
}
  \item{legend.loc}{
legend location coordinates as defined by \code{image.plot}. Default is \code{NULL}.
}
  \item{legend.horiz}{
plot a horizontal legend if TRUE. Default is FALSE.
}
  \item{arrow.size}{
size of the arrows. Default is 0.08.
}
  \item{arrow.lwd}{
width/thickness of arrows. Default is 2.
}
  \item{arrow.cutoff}{
cutoff value for plotting arrows. z-values that fall below the \code{arrow.cutoff}
quantile will not be plotted. Default is 0.9.
}
  \item{arrow.col}{
Color of arrows. Default is \code{black}.
}
 \item{xlim}{ 
 the x limits. The default is \code{NULL}.
}
 \item{ylim}{ 
 the y limits. The default is \code{NULL}.
}
 \item{xaxt}{ 
 Add x-axis? The default is \code{s}; use \code{n} for none.
}
 \item{yaxt}{ 
 Add y-axis? The default is \code{s}; use \code{n} for none.
}

\item{form}{
format to use to display dates on the x-axis. Default is '\%Y' for 4-digit year.
See \code{?Date} for other valid formats.
}
  \item{\dots}{
other parameters.
}
}
\details{
Arrows pointing to the right mean that \code{x} and \code{y} are in phase.

Arrows pointing to the left mean that \code{x} and \code{y} are in anti-phase.

Arrows pointing up mean that \code{y} leads \code{x} by \eqn{\pi/2}.

Arrows pointing down mean that \code{x} leads \code{y} by \eqn{\pi/2}.
}

\references{
Cazelles, B., M. Chavez, D. Berteaux, F. Menard, J. O. Vik, S. Jenouvrier, and 
N. C. Stenseth. 2008. Wavelet analysis of ecological time series. 
\emph{Oecologia} 156:287-304.

Grinsted, A., J. C. Moore, and S. Jevrejeva. 2004. Application of the cross 
wavelet transform and wavelet coherence to geophysical time series. 
\emph{Nonlinear Processes in Geophysics} 11:561-566.

Torrence, C., and G. P. Compo. 1998. A Practical Guide to Wavelet Analysis. 
\emph{Bulletin of the American Meteorological Society} 79:61-78.

Liu, Y., X. San Liang, and R. H. Weisberg. 2007. Rectification of the Bias in 
the Wavelet Power Spectrum. \emph{Journal of Atmospheric and Oceanic Technology} 
24:2093-2102.
}
\author{
Tarik C. Gouhier (tarik.gouhier@gmail.com)

Code based on WTC MATLAB package written by Aslak Grinsted.
}
\seealso{
\code{\link{image.plot}}
}

\examples{
t1=cbind(1:100, rnorm(100))
t2=cbind(1:100, rnorm(100))
## Continuous wavelet transform
wt.t1=wt(t1)
## Plot power
## Make room to the right for the color bar
par(oma=c(0, 0, 0, 1), mar=c(5, 4, 4, 5) + 0.1)
plot(wt.t1, plot.cb=TRUE, plot.phase=FALSE)

## Cross-wavelet transform
xwt.t1t2=xwt(t1, t2)
## Plot cross-wavelet
par(oma=c(0, 0, 0, 1), mar=c(5, 4, 4, 5) + 0.1)
plot(xwt.t1t2, plot.cb=TRUE)

## Example of bias-correction
t1=sin(seq(from=0, to=2*5*pi, length=1000))
t2=sin(seq(from=0, to=2*15*pi, length=1000))
t3=sin(seq(from=0, to=2*40*pi, length=1000))
## This aggregate time series should have the same power at three distinct periods
s=t1+t2+t3
## Compare plots to see bias-effect on CWT: biased power spectrum artificially
## reduces the power of higher-frequency fluctuations.
wt1=wt(cbind(1:1000, s))
par(mfrow=c(1,2))
plot(wt1, type="power.corr.norm", main="Bias-corrected")
plot(wt1, type="power.norm", main="Biased")

## Compare plots to see bias-effect on XWT: biased power spectrum artificially
## reduces the power of higher-frequency fluctuations.
x1=xwt(cbind(1:1000, s), cbind(1:1000, s))
par(mfrow=c(1,2))
plot(x1, type="power.corr.norm", main="Bias-corrected")
plot(x1, type="power.norm", main="Biased")
}
