% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forecasting_gq.R
\name{forecast.bmgarch}
\alias{forecast.bmgarch}
\alias{forecast}
\title{Forecast method for bmgarch objects.}
\usage{
\method{forecast}{bmgarch}(
  object,
  ahead = 1,
  xC = NULL,
  newdata = NULL,
  CrI = c(0.025, 0.975),
  seed = NA,
  digits = 2,
  weights = NULL,
  L = NA,
  method = "stacking",
  inc_samples = FALSE,
  ...
)
}
\arguments{
\item{object}{bmgarch object.}

\item{ahead}{Integer (Default: 1). Periods to be forecasted ahead.}

\item{xC}{Numeric vector or matrix. Covariates(s) for the constant variance terms in C, or c. Used in a log-linear model on the constant variance terms. If vector, then it acts as a covariate for all constant variance terms. If matrix, must have columns equal to number of time series, and each column acts as a covariate for the respective time series (e.g., column 1 predicts constant variance for time series 1).}

\item{newdata}{Future datapoints for LFO-CV computation}

\item{CrI}{Numeric vector (Default: \code{c(.025, .975)}). Lower and upper bound of predictive credible interval.}

\item{seed}{Integer (Optional). Specify seed for \code{\link[rstan]{sampling}}.}

\item{digits}{Integer (Default: 2, optional). Number of digits to round to when printing.}

\item{weights}{Takes weights from model_weight function. Defaults to 1 -- this parameter is not typically set by user.}

\item{L}{Minimal length of time series before engaging in lfocv}

\item{method}{Ensemble methods, 'stacking' (default) or 'pseudobma'}

\item{inc_samples}{Logical (Default: FALSE). Whether to return the MCMC samples for the fitted values.}

\item{...}{Not used}
}
\value{
forecast.bmgarch object. List containing \code{forecast}, \code{backcast}, and \code{meta}data.
See \code{\link{fitted.bmgarch}} for information on \code{backcast}.
\code{forecast} is a list of four components:
\describe{
  \item{mean}{\code{[N, 7, TS]} array of mean forecasts, where N is the timeseries length, and TS is the number of time series. E.g., \code{fc$forecast$mean[3,,"tsA"]} is the 3-ahead mean forecast for time series "tsA".}
  \item{var}{\code{[N, 7, TS]} array of variance forecasts, where N is the timeseries length, and TS is the number of time series. E.g., \code{fc$forecast$var[3,,"tsA"]} is the 3-ahead variance forecast for time series "tsA".}
  \item{cor}{\code{[N, 7, TS(TS - 1)/2]} array of correlation forecasts, where N is the timeseries length, and \code{TS(TS - 1)/2} is the number of correlations. E.g., \code{fc$forecast$cor[3,, "tsB_tsA"]} is the 3-ahead forecast for the correlation between "tsB" and "tsA". Lower triangular correlations are saved.}
  \item{meta}{Meta-data specific to the forecast. I.e., TS_length (number ahead) and xC.}
  \item{samples}{List}. If inc_samples is \code{TRUE}, then a list of arrays of MCMC samples for means, vars, and cors. Each array is [Iteration, Period, ..., ...].
}
}
\description{
Estimates (weighted) forecasted means, variances, and correlations from a fitted bmgarch model.
}
\examples{
\dontrun{
data(panas)
# Fit DCC(2,2) with constant mean structure.
fit <- bmgarch(panas, parameterization = "DCC", P = 2, Q = 2, meanstructure = "constant")

# Forecast 8 ahead
fit.fc <- forecast(fit, ahead = 8)

# Print forecasts
fit.fc
print(fit.fc)

# Plot variance forecasts
plot(fit.fc, type = "var")

# Plot correlation forecasts
plot(fit.fc, type = "cor")

# Save backcasted and forecasted values as data frame.
fit.fc.df <- as.data.frame(fit.fc)

# Save only forecasted values as data frame.
fit.fc.df <- as.data.frame(fit.fc, backcast = FALSE)

# Add another model, compute model weights and perform a model weighted forecast

# Fit a DCC(1,1) model
fit1 <- bmgarch(panas, parameterization = "DCC", P = 1, Q = 1, meanstructure = "constant")

# Compute model stacking weights based on the last 19 time points (with L = 80)
blist <- bmgarch_list( fit1, fit )
mw <- model_weights(blist, L = 80)

# Weighted forecasts:
w.fc <- forecast(object = blist, ahead = 8, weights = mw)
}
}
