% Generated by roxygen2 (4.0.2): do not edit by hand
\name{boolprep}
\alias{boolprep}
\title{Specify Boolean Model}
\usage{
boolprep(..., data, subset, weights, na.action, offset,
  family = list(binomial(link = "logit")))
}
\arguments{
\item{...}{formula specification for boolean model. See the details and
examples sections below.}

\item{data}{\code{data.frame} containing the data to be used in the model.}

\item{subset}{select subset of data. See \code{\link{lm}} for details.}

\item{weights}{specify model weights (not implemented).}

\item{na.action}{set \code{na.action}. See \code{\link{lm}} for details.}

\item{offset}{specify an offset. Offsets are not implemented and this
parameter is simply ignored.}

\item{family}{a model \code{family} to use for estimation. This can be
comprised of a list of link functions when the desire is to have model
components with different links. \code{binomial} is the only family
supported at this time. See \code{\link{family}} for more details.}
}
\value{
\code{boolprep} returns a \code{\link{boolean-class}} object
containing the model components needed to estimate a boolean model.
}
\description{
Construct a boolean object that can then be fit with \code{boolean}.
}
\details{
\code{boolprep} sets up a boolean model object that can then be fit with
\code{\link{boolean}}. A properly specified model (contained in \code{...})
will contain at least three components. The first component must specify the
boolean logic to be employed. For instance, the \code{y ~ (a | b)} formula
would indicate a logical \code{or} between the \code{a} and \code{b}
submodels, while \code{y ~ (a & b)} would indicate a logical
\code{and}. \code{y} is the name of the response variable of
interest. Logical operators can be nested; e.g., \code{y ~ (a | (b & c))} is
valid. The second and third components are submodels and are specified as
usual: \code{a ~ x1 + x2} and \code{b ~ x3 + x4 + x5}, where are the
\code{x}-variables are covariates. \code{a}, \code{b}, and \code{c} are
labels indicating the submodel position in the boolean specification.
}
\examples{
## Generate some fake data
require(mvtnorm)
set.seed(12345)
N  <- 2000
Df <- cbind(1, rmvnorm(N, mean=rep(0, 5)))

## Set coefficients
beta.a <- c(-2.00, 0.33, 0.66, 1.00)
beta.b <- c(0.00, 1.50, -0.25)

## Generate path probabilities following a normal model.
y.a <- as.vector(pnorm(tcrossprod(beta.a, Df[, 1:4])))
y.b <- as.vector(pnorm(tcrossprod(beta.b, Df[, c(1, 5, 6)])))

## AND and OR-model
or <- function(x, y) { x + y - x * y }
and <- function(x, y) { x * y }
y.star.OR  <- or(y.a, y.b)
y.star.AND <- and(y.a, y.b)

## Observed responses
y.OR <- rbinom(N, 1, y.star.OR)
y.AND <- rbinom(N, 1, y.star.AND)

## Set up data.frame for estimation
Df <- cbind(1, Df)
Df <- as.data.frame(Df)
Df[,1] <- y.OR
Df[,2] <- y.AND
names(Df) <- c("y.OR", "y.AND", "x1", "x2", "x3", "x4", "x5")

## Before estimating, boolean models need to be specified using the
## boolprep function.

## OR model, specified to use a probit link for each causal path. This
## matches the data generating process above.
mod.OR <- boolprep(y.OR ~ (a | b), a ~ x1 + x2 + x3, b ~ x4 + x5,
                   data = Df, family=list(binomial("probit")))
}
\author{
Jason W. Morgan (\email{morgan.746@osu.edu})
}
\references{
Braumoeller, Bear F. (2003) ``Causal Complexity and the Study
of Politics.'' \emph{Political Analysis} 11(3): 209--233.
}
\seealso{
See \code{\link{boolean}} for model estimation.
}

