params <-
list(EVAL = TRUE)

## ---- SETTINGS-knitr, include=FALSE-----------------------------------------------------
stopifnot(require(knitr))
options(width = 90)
opts_chunk$set(
  comment = NA,
  message = FALSE,
  warning = FALSE,
  eval = if (isTRUE(exists("params"))) params$EVAL else FALSE,
  dev = "png",
  dpi = 150,
  fig.asp = 0.8,
  fig.width = 5,
  out.width = "60%",
  fig.align = "center"
)
library(brms)
ggplot2::theme_set(theme_default())

## ----cbpp-------------------------------------------------------------------------------
data("cbpp", package = "lme4")
head(cbpp)

## ----fit1, results='hide'---------------------------------------------------------------
fit1 <- brm(incidence | trials(size) ~ period + (1|herd), 
            data = cbpp, family = binomial())

## ----fit1_summary-----------------------------------------------------------------------
summary(fit1)

## ----beta_binomial2---------------------------------------------------------------------
beta_binomial2 <- custom_family(
  "beta_binomial2", dpars = c("mu", "phi"),
  links = c("logit", "log"), lb = c(NA, 0),
  type = "int", vars = "vint1[n]"
)

## ----stan_funs--------------------------------------------------------------------------
stan_funs <- "
  real beta_binomial2_lpmf(int y, real mu, real phi, int T) {
    return beta_binomial_lpmf(y | T, mu * phi, (1 - mu) * phi);
  }
  int beta_binomial2_rng(real mu, real phi, int T) {
    return beta_binomial_rng(T, mu * phi, (1 - mu) * phi);
  }
"

## ----stanvars---------------------------------------------------------------------------
stanvars <- stanvar(scode = stan_funs, block = "functions")

## ----fit2, results='hide'---------------------------------------------------------------
fit2 <- brm(
  incidence | vint(size) ~ period + (1|herd), data = cbpp, 
  family = beta_binomial2, stanvars = stanvars
)

## ----summary_fit2-----------------------------------------------------------------------
summary(fit2)

## ---------------------------------------------------------------------------------------
expose_functions(fit2, vectorize = TRUE)

## ----log_lik----------------------------------------------------------------------------
log_lik_beta_binomial2 <- function(i, draws) {
  mu <- draws$dpars$mu[, i]
  phi <- draws$dpars$phi
  trials <- draws$data$vint1[i]
  y <- draws$data$Y[i]
  beta_binomial2_lpmf(y, mu, phi, trials)
}

## ----loo--------------------------------------------------------------------------------
loo(fit1, fit2)

## ----predict----------------------------------------------------------------------------
predict_beta_binomial2 <- function(i, draws, ...) {
  mu <- draws$dpars$mu[, i]
  phi <- draws$dpars$phi
  trials <- draws$data$vint1[i]
  beta_binomial2_rng(mu, phi, trials)
}

## ----pp_check---------------------------------------------------------------------------
pp_check(fit2)

## ----fitted-----------------------------------------------------------------------------
fitted_beta_binomial2 <- function(draws) {
  mu <- draws$dpars$mu
  trials <- draws$data$vint1
  trials <- matrix(trials, nrow = nrow(mu), ncol = ncol(mu), byrow = TRUE)
  mu * trials
}

## ----marginal_effects-------------------------------------------------------------------
marginal_effects(fit2, conditions = data.frame(size = 1))

