% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival_tidiers.R
\name{survfit_tidiers}
\alias{glance.survfit}
\alias{survfit_tidiers}
\alias{tidy.survfit}
\title{tidy survival curve fits}
\usage{
\method{tidy}{survfit}(x, ...)

\method{glance}{survfit}(x, ...)
}
\arguments{
\item{x}{"survfit" object}

\item{...}{extra arguments, not used}
}
\value{
All tidying methods return a data.frame without rownames, whose
structure depends on the method chosen.

\code{tidy} returns a row for each time point, with columns
  \item{time}{timepoint}
  \item{n.risk}{number of subjects at risk at time t0}
  \item{n.event}{number of events at time t}
  \item{n.censor}{number of censored events}
  \item{estimate}{estimate of survival}
  \item{std.error}{standard error of estimate}
  \item{conf.high}{upper end of confidence interval}
  \item{conf.low}{lower end of confidence interval}

\code{glance} returns one-row data.frame with the columns
displayed by \code{\link{print.survfit}}
  \item{records}{number of observations}
  \item{n.max}{n.max}
  \item{n.start}{n.start}
  \item{events}{number of events}
  \item{rmean}{Restricted mean (see \link[survival]{print.survfit})}
  \item{rmean.std.error}{Restricted mean standard error}
  \item{median}{median survival}
  \item{conf.low}{lower end of confidence interval on median}
  \item{conf.high}{upper end of confidence interval on median}
}
\description{
Construct tidied data frames showing survival curves over time.
}
\details{
\code{glance} does not work on multi-state survival curves,
since the values \code{glance} outputs would be calculated for each state.
\code{tidy} does work for multi-state survival objects, and includes a
\code{state} column to distinguish between them.
}
\examples{

if (require("survival", quietly = TRUE)) {
    cfit <- coxph(Surv(time, status) ~ age + sex, lung)
    sfit <- survfit(cfit)
    
    head(tidy(sfit))
    glance(sfit)
    
    library(ggplot2)
    ggplot(tidy(sfit), aes(time, estimate)) + geom_line() +
        geom_ribbon(aes(ymin=conf.low, ymax=conf.high), alpha=.25)
    
    # multi-state
    fitCI <- survfit(Surv(stop, status * as.numeric(event), type = "mstate") ~ 1,
                  data = mgus1, subset = (start == 0))
    td_multi <- tidy(fitCI)
    head(td_multi)
    tail(td_multi)
    ggplot(td_multi, aes(time, estimate, group = state)) +
        geom_line(aes(color = state)) +
        geom_ribbon(aes(ymin = conf.low, ymax = conf.high), alpha = .25)
 
    # perform simple bootstrapping
    library(dplyr)
    bootstraps <- lung \%>\% bootstrap(100) \%>\%
        do(tidy(survfit(coxph(Surv(time, status) ~ age + sex, .))))
    
    ggplot(bootstraps, aes(time, estimate, group = replicate)) +
        geom_line(alpha = .25)
    
    bootstraps_bytime <- bootstraps \%>\% group_by(time) \%>\%
        summarize(median = median(estimate),
                  low = quantile(estimate, .025),
                  high = quantile(estimate, .975))
    
    ggplot(bootstraps_bytime, aes(x = time, y = median)) + geom_line() +
        geom_ribbon(aes(ymin = low, ymax = high), alpha = .25)
 
    # bootstrap for median survival
    glances <- lung \%>\%
        bootstrap(100) \%>\%
        do(glance(survfit(coxph(Surv(time, status) ~ age + sex, .))))
    
    glances
    
    qplot(glances$median, binwidth = 15)
    quantile(glances$median, c(.025, .975))
}

}

