
# last revision: 23 Jan 2012

#@include DriftNoiseModelClassMethods.R
#' @include ChemosensorsClassMethods.R
NULL

#----------------------------
# Support Class defintion
#----------------------------


#----------------------------
# Class defintion
#----------------------------

validDriftNoiseModel <- function(object)
{
  return(TRUE)
}

#' Class DriftNoiseModel.
#'
#' Class \code{DriftNoiseModel} generates the drift noise in a multi-variate manner
#' in several steps.
#'
#' The primary question arising in drift modeling is related to the way of one defines the drift
#' phenomena for gas sensor arrays. We propose to evaluate a drift subspace via common principal component analysis.
#' The hypothesis of common principal component analysis states that exists an orthogonal matrix \code{V}
#' such that the covariance matrices of \code{K} groups have the diagonal form simultaneously.
#' The resulted eigenvectors (columns of the matrix \code{V}) define the subspace common for the groups
#' and orthogonal across the components.
#'
#' A preliminary step involves quantification of drift-related data 
#' presented in the long-term UNIMAN dataset. These results are stored
#' in \code{\link{UNIMANdnoise}} dataset.
#'
#' On the next step the drift is injected in the sensor array data 
#' by generating the noise by multi-dimensional random walk 
#' based on the multivariate normal distribution with zero-mean and diagonal covariance matrix
#' - in the sub-space defined by the matrix \code{V}.
#' The relative proportion along the diagonal elements in the covariance matrix is specified by the importance of drift
#' components in terms of of projected variance.
#'
#' On the final step the component correction operation 
#' is recalled to induce the generated noise from the random walk 
#' back into the complete multivariate space of the sensor array data.
#'
#' Slots of the class:
#' \tabular{rl}{
#'   \code{num} \tab Sensor number (\code{1:17}), which drift profile is used. The default value is \code{c(1, 2)}. \cr
#'   \code{dsd} \tab Parameter of standard deviation used to generate the drift noise. The deault value is 0.1. \cr
#'   \code{ndcomp} \tab The number of components spanning the drift sub-space. The default number is 2. \cr
#'   \code{ndvar} \tab The importance values of drift components. The default values are \code{\link{UNIMANdnoise}} dataset. \cr
#'   \code{driftModel} \tab Drift model of class \code{DriftCommonModel}. \cr
#' }
#' Methods of the class:
#' \tabular{rl}{
#'   \code{predict} \tab Generates multi-variate noise injeted to an input sensor array data. \cr
#'   \code{dsd} \tab Gets the noise level. \cr
#'   \code{dsd<-} \tab Sets the noise level. \cr
#' }
#' 
#' The \code{plot} method has three types (parameter \code{y}):
#' \tabular{rl}{
#'   \code{noise} \tab (default) Depicts the drift noise generated by the model with a linechart. \cr
#'   \code{pc} \tab  Shows the drift components in a PCA scoreplot of an input sensor array data (parameter \code{X}.\cr
#' }
#'
#' @note
#' In the case \code{num} is different from value \code{c(1:17)}, 
#' the number of components is not the same as in \code{V} matrix.
#' First, the colums in \code{V} matrix are selected according to numbers pointed in \code{num}.
#' Second, QR-decomposition of the resulted matrix is performed to orthogonolize the component vectors. 
#'
#' @name DriftNoiseModel-class
#' @rdname DriftNoiseModel-class
#' @example inst/examples/DriftNoiseModel-class.R
#' @seealso \code{\link{UNIMANdnoise}}, \code{\link{SensorArray}}
#' @exportClass DriftNoiseModel
setClass(Class="DriftNoiseModel", 
  representation=representation(
    num = "numeric",
    dsd = "numeric", ndcomp = "numeric", ndvar = "numeric",
    driftModel = "DriftCommonModel"),  
  validity=validDriftNoiseModel
)

#----------------------------
# Print/Show Methods
#----------------------------

#' @rdname class-methods
#' @aliases print,DriftNoiseModel-method
setMethod ("print","DriftNoiseModel", function(x, ...)
{
  cat(" Drift Noise Model\n")
  cat(" - num", numStr(x), "\n")
  print(x@driftModel)
})

setMethod ("show","DriftNoiseModel", function(object)
{
  cat(" Drift Noise Model (dsd ", dsd(object), "), ", "common model '", modelName(object), "'", "\n", sep='')
})

#----------------------------
# Get/Set Methods
#----------------------------

#' @rdname get-methods
#' @aliases modelName,DriftNoiseModel-method
setMethod("modelName", "DriftNoiseModel", function(x) x@driftModel$method)

#' @rdname get-methods
#' @aliases type,DriftNoiseModel-method
setMethod("type", "DriftNoiseModel", function(x) x@driftModel$method)

#' @rdname noise-methods
#' @aliases dsd,DriftNoiseModel-method
setMethod("dsd", "DriftNoiseModel", function(x) x@dsd)

#' @rdname DriftNoiseModel-class
#' @aliases dspace,DriftNoiseModel-method
setMethod("dspace", "DriftNoiseModel", function(x) x@driftModel$dspace[, 1:x@driftModel$ndcomp])

#' @rdname DriftNoiseModel-class
#' @aliases ndcomp,DriftNoiseModel-method
setMethod("ndcomp", "DriftNoiseModel", function(x) x@ndcomp)

#' @rdname DriftNoiseModel-class
#' @aliases ndvar,DriftNoiseModel-method
setMethod("ndvar", "DriftNoiseModel", function(x) x@ndvar)

#' @name dsd<-
#' @aliases dsd<-,DriftNoiseModel-method
#' @rdname noise-methods
setReplaceMethod("dsd", "DriftNoiseModel", function(object, value) 
{
  object@dsd <- value
  validObject(object)
  return (object)
})

#----------------------------
# Plot Methods
#----------------------------


#----------------------------
# Model Methods
#----------------------------

#----------------------------
# Predict methods
#----------------------------

