#' List, download or convert to chillR format data from the Chilean Agromet
#' database
#' 
#' This function can do three things related to the Chilean Agromet database
#' ("Agromet"): 1. it can list stations that are close to a specified position
#' (geographic coordinates) (not yet) 2. it can retrieve weather data for a
#' named weather station (not yet) 3. it can 'clean' downloaded data, so that
#' they can easily be used in chillR Which of these functions is carried out
#' depends on the action argument.
#' 
#' This function can run independently, but it is also called by the
#' get_weather and weather2chillR functions, which some users might find a bit
#' easier to handle.
#' 
#' the Chile Agromet dataset is described here:
#' http://agromet.inia.cl/index.php
#' 
#' under the 'list_stations' mode, several formats are possible for specifying
#' the location vector, which can consist of either two or three coordinates
#' (it can include elevation). Possible formats include c(1,2,3), c(1,2),
#' c(x=1,y=2,z=3), c(lat=2,long=1,elev=3). If elements of the vector are not
#' names, they are interpreted as c(Longitude, Latitude, Elevation).
#' 
#' The 'chillRCode' is generated by this function, when it is run with
#' geographic coordinates as location inputs. In the list of nearby stations
#' that is returned then, the chillRCode is provided and can then be used as
#' input for running the function in 'downloading' mode. For downloading the
#' data, use the same call as before but replace the location argument with the
#' chillRCode.
#' 
#' @param action if this is the character string "list_stations", the function
#' will return a list of the weather stations from the database that are
#' closest to the geographic coordinates specified by location.  if this is the
#' character string "download_weather", the function will attempt to download
#' weather data from the database for the station named by the location
#' argument, which should then be a character string corresponding to the
#' chillRcode of the station (which you can get by running this function in
#' 'list_stations mode) if this is a downloaded weather file (downloaded by
#' running this function in 'download weather' mode), the function cleans the
#' file and makes it ready for use in chillR. If the input is just a dataframe
#' (not a list, as produced with this function), you have to specify the
#' database name with the database argument
#' @param location either a vector of geographic coordinates (for the
#' 'list_stations' mode), or the 'chillRcode' of a weather station in the
#' specified database (for the 'download_weather' mode. When running this
#' function for data cleaning only, this is not needed.
#' @param time_interval numeric vector with two elements, specifying the start
#' and end date of the period of interest. Only required when running in
#' 'list_stations' or 'download weather' mode
#' @param station_list if the list of weather stations has already been
#' downloaded, the list can be passed to the function through this argument.
#' This can save a bit of time, since it can take a bit of time to download the
#' list, which can have several MB.
#' @param stations_to_choose_from if the location is specified by geographic
#' coordinates, this argument determines the number of nearby stations in the
#' list that is returned.
#' @param drop_most boolean variable indicating if most columns should be
#' dropped from the file. If set to TRUE (default), only essential columns for
#' running chillR functions are retained.
#' @param end_at_present boolean variable indicating whether the interval of
#' interest should end on the present day, rather than extending until the end
#' of the year specified under time_interval[2] (if time_interval[2] is the
#' current year).
#' @return The output depends on the action argument. If it is 'list_stations',
#' the function returns a list of station_to_choose_from weather stations that
#' are close to the specified location. This list also contains information
#' about how far away these stations are (in km), how much the elevation
#' difference is (if elevation is specified; in m) and how much overlap there
#' is between the data contained in the database and the time period specified
#' by time_interval. If action is 'download_weather' the output is a list of
#' two elements: 1. database="Agromet" 2. the downloaded weather record,
#' extended to the full duration of the specified time interval. If action is a
#' weather data.frame or a weather record downloaded with this function (in
#' 'download_weather' mode), the output is the same data in a format that is
#' easy to use in chillR. If drop_most was set to TRUE, most columns are
#' dropped.
#' @note Many databases have data quality flags, which may sometimes indicate
#' that data aren't reliable. These are not considered by this function!
#' 
#' This is work in progress, and for now the download can only be done
#' manually.
#' @author Eike Luedeling
#' @references The chillR package:
#' 
#' Luedeling E, Kunz A and Blanke M, 2013. Identification of chilling and heat
#' requirements of cherry trees - a statistical approach. International Journal
#' of Biometeorology 57,679-689.
#' @keywords utilities
#' @examples
#' 
#' 
#' #work in progress, but should work for processing daily weather files dowloaded from
#' #http://agromet.inia.cl/index.php
#' #weather<-handle_chile_agromet(filename)
#' 
#' @export handle_chile_agromet
handle_chile_agromet<-function(action,location=NA,time_interval=NA,station_list=NULL,stations_to_choose_from=25,drop_most=TRUE,
                               end_at_present=TRUE)
{

  remove_missing<-function(tab,column,missing)
  {tab[which(tab[,column]==missing),column]<-NA
  return(tab)}

  if(is.character(action))  if(action=="list_stations")
        {
    if(!is.null(names(location)))
    {lat<-unlist(sapply(names(location),function(x) max(c(length(grep("lat", x, ignore.case = TRUE)),length(grep("y", x, ignore.case = TRUE))))))
    if(sum(lat)==1) lat<-as.numeric(location[which(lat==1)])
    long<-unlist(sapply(names(location),function(x) max(c(length(grep("lon", x, ignore.case = TRUE)),length(grep("x", x, ignore.case = TRUE))))))
    if(sum(long)==1) long<-as.numeric(location[which(long==1)])
    elev<-unlist(sapply(names(location),function(x) max(c(length(grep("ele", x, ignore.case = TRUE)),length(grep("alt", x, ignore.case = TRUE)),
                                                          length(grep("z", x, ignore.case = TRUE))))))
    if(sum(elev)==1) elev<-as.numeric(location[which(elev==1)]) else elev<-NA
    } else {long<-location[1]
    lat<-location[2]
    if(length(location)==3) elev<-location[3] else elev<-NA}


    if(is.null(station_list)) station_list<-read.table("https://raw.githubusercontent.com/jpatokal/openflights/master/data/airports.dat",sep=",")
    colnames(station_list)<-c("Airport ID","Name","City","Country","IATA",
                              "ICAO","Latitude","Longitude","Altitude","Timezone","DST","Tz database time zone")
    station_list[,"Altitude"]<-station_list[,"Altitude"]*0.3048
    stat_list<-station_list
    colnames(stat_list)[which(colnames(stat_list) %in% c("Longitude","Long"))]<-"Long"
    colnames(stat_list)[which(colnames(stat_list) %in% c("Latitude","Lat"))]<-"Lat"
    colnames(stat_list)[which(colnames(stat_list) %in% c("Altitude","Elev"))]<-"Elev"
    stat_list<-stat_list[which(!is.na(stat_list$Lat)&(!is.na(stat_list$Long))),]
    myPoint<-c(long,lat)
    stat_list[,"distance"]<-round(spDistsN1(as.matrix(stat_list[,c("Long","Lat")]), myPoint, longlat=TRUE),2)
    sorted_list<-stat_list[order(stat_list$distance),]
    if(!is.na(elev)) sorted_list[,"elevation_diff"]<-elev-sorted_list$Elev
    interval_end<-YEARMODA2Date(time_interval[2]*10000+1231)
    if(end_at_present) interval_end<-min(interval_end,ISOdate(format(Sys.Date(),"%Y"),format(Sys.Date(),"%m"),
                                                              format(Sys.Date(),"%d")))
    interval_start<-YEARMODA2Date(time_interval[1]*10000+0101)
    sorted_list<-sorted_list[which(!sorted_list$ICAO=="\\N"),]
    sorted_list<-sorted_list[1:max(stations_to_choose_from,500),]
    #sorted_list[,"Overlap_years"]<-
    #  apply(sorted_list,1,function (x) (as.numeric(difftime(
    #    sort(c(YEARMODA2Date(as.numeric(x["END"])),interval_end))[1],
    #    sort(c(YEARMODA2Date(as.numeric(x["BEGIN"])),interval_start))[2])+1)/(365+length(which(sapply(time_interval[1]:time_interval[2],leap_year)))/(time_interval[2]-time_interval[1]+1)    )))
    #sorted_list[which(sorted_list[,"Overlap_years"]<0),"Overlap_years"]<-0
    #sorted_list[,"Perc_interval_covered"]<-sorted_list[,"Overlap_years"]/(time_interval[2]-time_interval[1])*100
    sorted_list[,"chillR_code"]<-sorted_list$ICAO
    if(!is.na(elev))  sorted_list<-sorted_list[,c("chillR_code","Name","City","Country","Lat","Long","Elev",
                                                  "distance","elevation_diff")] else
                                                    sorted_list<-sorted_list[,c("chillR_code","Name","City","Country","Lat","Long",
                                                                                "distance")]
    return(sorted_list[1:stations_to_choose_from,])}



 if(is.character(action)) if(action=="download_weather")
 {
   if(is.null(station_list)) station_list<-read.table("https://raw.githubusercontent.com/jpatokal/openflights/master/data/airports.dat",sep=",")
   colnames(station_list)<-c("Airport ID","Name","City","Country","IATA",
                             "ICAO","Latitude","Longitude","Altitude","Timezone","DST","Tz database time zone")
   station_list[,"Altitude"]<-station_list[,"Altitude"]*0.3048
   stat_list<-station_list
   colnames(stat_list)[which(colnames(stat_list) %in% c("Longitude","Long"))]<-"Long"
   colnames(stat_list)[which(colnames(stat_list) %in% c("Latitude","Lat"))]<-"Lat"
   colnames(stat_list)[which(colnames(stat_list) %in% c("Altitude","Elev"))]<-"Elev"
   stat_list<-stat_list[which(!is.na(stat_list$Lat)&(!is.na(stat_list$Long))),]
   if (!"chillR_code" %in% colnames(stat_list)) stat_list[,"chillR_code"]<-stat_list$ICAO
   if(location %in% stat_list$chillR_code)
   {
   suppressWarnings(dir.create("chillRtempdirectory"))
   filepath <- "chillRtempdirectory/wunder.csv"
   startlisting=TRUE
   y=2005
   record<-NULL
   for(y in time_interval[1]:time_interval[2])
     for (m in 1:12)
     {
       Url <- paste("http://www.wunderground.com/history/airport/",location,"/",
                    y,"/",m,"/1/MonthlyHistory.html?HideSpecis=0&theprefset=SHOWMETAR&theprefvalue=1&format=1",sep="")
       ff<-suppressWarnings(try(download.file(Url,filepath)))
       if(ff==0)
         {rcsv<-read.csv("chillRtempdirectory/wunder.csv")
          colnames(rcsv)[1]<-"Date"
          if(startlisting) {record<-rcsv
          startlisting=FALSE} else record<-rbind(record,rcsv)}
     }
   closeAllConnections()
   file.remove("chillRtempdirectory/wunder.csv")
   if(is.null(record)) {record<-NA; message("no data found")}
   if(!is.null(record)) if(nrow(record)==0) {record<-NA; message("no data found")}
   return(list(database="Wunderground",weather=record))} else
   {warning("location does not match a record in the database. No records retrieved.")}
 }





  if(is.list(action)) if(names(action)[1]=="database") # then we assume that this is a downloaded file to be cleaned
      {dw<-action$weather
      colnames(dw)[which(colnames(dw)=="Mean.TemperatureC")]<-"Tmean"
      colnames(dw)[which(colnames(dw)=="Min.TemperatureC")]<-"Tmin"
      colnames(dw)[which(colnames(dw)=="Max.TemperatureC")]<-"Tmax"
      colnames(dw)[which(colnames(dw)=="Precipitationmm")]<-"Prec"
      dw[,"Year"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][1])
      dw[,"Month"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][2])
      dw[,"Day"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][3])
      if(drop_most) dw<-dw[,c("Year","Month","Day","Tmin","Tmax","Tmean","Prec")]
      for (cc in c("Year","Month","Day","Tmin","Tmax","Tmean","Prec"))
        dw[,cc]<-as.numeric(dw[,cc])
      return(list(database="GSOD",weather=dw))}
      if(is.data.frame(action)) # then we assume that this is a downloaded file to be cleaned
      {dw<-action
      colnames(dw)[which(colnames(dw)=="Mean.TemperatureC")]<-"Tmean"
      colnames(dw)[which(colnames(dw)=="Min.TemperatureC")]<-"Tmin"
      colnames(dw)[which(colnames(dw)=="Max.TemperatureC")]<-"Tmax"
      colnames(dw)[which(colnames(dw)=="Precipitationmm")]<-"Prec"
      dw[,"Year"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][1])
      dw[,"Month"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][2])
      dw[,"Day"]<-sapply(dw[,"Date"],function(x) strsplit(as.character(x),"-")[[1]][3])
      if(drop_most) dw<-dw[,c("Year","Month","Day","Tmin","Tmax","Tmean","Prec")]
      for (cc in c("Year","Month","Day","Tmin","Tmax","Tmean","Prec"))
        dw[,cc]<-as.numeric(dw[,cc])
      return(dw)}

}

