#' Plot reference signature and comparison signature based on the results of CMPS algorithm
#' 
#' This function aligns two signatures and shows which basis segments find the congruent 
#' registration position.
#'
#' @param cmps.result a list generated by `extract_feature_cmps`. `cmps.result` is required to have
#' the following names: `parameters`, `congruent.seg`, `congruent.pos`, `segments`, `ccp.list`. So
#' `include = "full_result"` is recommended when computing `cmps.result`
#' @param add_background boolean; whether or not to add zebra-striped background under each basis 
#' segment; default is TRUE
#'
#' @return a list
#' * `segment_shift_plot`: a plot object generated by ggplot2. In this plot only basis segments that are 
#' congruent matching profile segments (CMPS) are plotted along with the comparison profile; each basis 
#' segment is shifted to the position where it obtains either a consistent correlation peak or a 
#' cross-correlation peak closest to the congruent registration position
#' * `signature_shift_plot`:  a plot object generated by ggplot2. In this plot both the reference 
#' signature and the comparison signature are plotted, and CMPS are highlighted. The alignment of the 
#' two signatures is achieved by shifting the reference signature to the congruent registration position. 
#' * `seg_shift`: a data.frame. This data frame shows which basis segments are plotted (are CMPS) and 
#' the units by which each segment shifted when plotting `segment_shift_plot`
#' * `sig_shift`: a numeric value. The number of units by which the reference signature shifted
#' when plotting `signature_shift_plot`
#' 
#' @export
#' @importFrom assertthat has_name
#' @import ggplot2
#' @import dplyr
#'
#' @examples
#' library(tidyverse)
#' library(cmpsR)
#' 
#' data("bullets")
#' land2_3 <- bullets$sigs[bullets$bulletland == "2-3"][[1]]
#' land1_2 <- bullets$sigs[bullets$bulletland == "1-2"][[1]]
#' 
#' # compute cmps
#' 
#' # algorithm with multi-peak insepction at three different segment scales
#' cmps_with_multi_scale <- extract_feature_cmps(land2_3$sig, land1_2$sig, include = "full_result" )
#' 
#' # generate plots using cmps_signature_plot
#' sig.plot <- cmps_signature_plot(cmps_with_multi_scale)
#' 
cmps_signature_plot <- function(cmps.result, add_background = TRUE) {
  
  has_name(cmps.result, 
           c("parameters", "congruent.seg", "congruent.pos",
             "segments", "ccp.list"))
  
  sig1 <- cmps.result$parameters$x
  sig2 <- cmps.result$parameters$y
  
  congruent.seg <- cmps.result$congruent.seg
  congruent.pos <- cmps.result$congruent.pos
  
  # get the segments
  segs <- cmps.result$segments
  
  # create a group for each segment
  gp <- lapply(names(segs$segs[congruent.seg]), function(nn){
    return(rep(nn, length(segs$segs[congruent.seg][[nn]])))
  }) %>% unlist() 
  
  # generate df for plotting    
  
  # find shift for each segment
  seg.pos <- sapply(cmps.result$ccp.list[congruent.seg], function(ccp.list.element, Tx, con.pos) {
    
    pp <- which.min(abs(ccp.list.element - con.pos))
    
    if(abs(ccp.list.element[pp] - con.pos) > Tx) {
      stop("something wrong with seg.pos")
    }
    
    ccp.list.element[pp]
    
  }, cmps.result$parameters$Tx, congruent.pos)
  
  
  names(seg.pos) <- names(segs$index[congruent.seg])
  
  
  # cat(names(seg.pos))
  
  # repeat the shifts many times to prepare for a data.frame used for plotting
  seg_shift <- lapply(names(segs$segs[congruent.seg]), function(nn){
    return(rep(seg.pos[nn], length(segs$segs[congruent.seg][[nn]])))
  }) %>% unlist()
  
  # for the signature level, use the median of 
  # all segment shifts as the signature shift
  sig_shift <- median(seg.pos)
  
  # data frame for segment shifts
  plot.df.seg_shift <- data.frame(x=segs$index[congruent.seg] %>% unlist() + seg_shift,
                                  sig=segs$segs[congruent.seg] %>% unlist(),
                                  gp = gp)
  
  # data frame for signature shift
  plot.df.sig_shift <- data.frame(x=segs$index[congruent.seg] %>% unlist() + sig_shift,
                                  sig=segs$segs[congruent.seg] %>% unlist(),
                                  gp = gp)
  
  # data frame for plotting comparison profile
  plot.df.comp <- data.frame(x=1:length(sig2), sig = sig2, gp = 0)
  
  # data frame for plotting reference profile
  plot.df.ref <- data.frame(x=1:length(sig1) + sig_shift, sig=sig1, gp=0)
  
  # segment shifts plot
  p1 <- plot.df.seg_shift %>% ggplot() +
    geom_line(aes(x=.data$x, y=.data$sig, group = .data$gp), color = "red", size = 1.2) +
    geom_line(aes(x=.data$x, y=.data$sig, group = .data$gp), 
              data = plot.df.comp, color = "black") + 
    theme_bw()
  
  # signature shift plot
  p2 <- plot.df.sig_shift %>% ggplot() +
    geom_line(aes(x=.data$x, y=.data$sig, group = gp), color = "red", size = 1.2) +
    geom_line(data = plot.df.comp, 
              # aes(x=.data$x, y=.data$sig, group = gp), color = "grey44") + 
              aes(x=.data$x, y=.data$sig, group = gp), color = "black") + 
    geom_line(data = plot.df.ref,
              aes(x=.data$x, y=.data$sig, group = gp), color = "red", linetype = "longdash") + 
    theme_bw()
  if(add_background){
    tmp <- lapply(segs$index, function(idx, sig_shift) {c(min(idx) + sig_shift, max(idx) + sig_shift + 1)}, 
                  sig_shift)
    
    rect.df <- do.call(rbind, tmp) %>% as.data.frame()
    colnames(rect.df) <- c("xminn", "xmaxx")
    
    rect.df$idx <- 1:length(segs$index)
    rect.df$fill <- if_else(rect.df$idx %% 2 == 1, "dark", "light")
    
    p2 <- p2 +
      geom_rect(data = rect.df, 
                aes(xmin = .data$xminn, xmax = .data$xmaxx, 
                    ymin = -Inf, ymax = Inf, fill = .data$fill),
                alpha = 0.15, show.legend = FALSE) +
      scale_fill_grey(start = 0.8, end = 0.3)
    
    # for seg plot
    plot.df.seg_shift$gp <- as.numeric(plot.df.seg_shift$gp)
    tmp2 <- plot.df.seg_shift %>% group_by(gp) %>% 
      summarize(.groups = "drop_last",
                t_xmin = min(.data$x, na.rm = TRUE),
                t_xmax = max(.data$x, na.rm = TRUE),
                t_ymin = min(.data$sig, na.rm = TRUE),
                t_ymax = max(.data$sig, na.rm = TRUE)
      )
    
    tmp2$idx <- 1:nrow(tmp2)
    tmp2$fill <- if_else(tmp2$idx %% 2 == 1, "dark", "light")
    
    p1 <- p1 + geom_rect(data = tmp2, 
                         aes(xmin = .data$t_xmin, xmax = .data$t_xmax, 
                             ymin = .data$t_ymin, ymax = .data$t_ymax, fill = "a"),
                         alpha = 0.4, show.legend = FALSE)+
      scale_fill_grey(start = 0.8, end = 0.3)
    
  }
  
  return(list(
    segment_shift_plot = p1, signature_shift_plot = p2,
    seg_shift = data.frame(seg.idx = names(seg.pos), seg.shift = seg.pos),
    sig_shift = sig_shift))
  
}

#' Plot the selected basis segment and its cross-correlation curve at all scales based on the 
#' results of CMPS algorithm
#' 
#' This function plots the selected basis segment with the comparison signature. One can visualize the 
#' scaled segment and its corresponding cross-correlation curve. The number of marked correlation peaks
#' at each segment scale is determined by `npeaks.set` of `extract_feature_cmps`. The red vertical dashed 
#' line indicates the congruent registration position for all segments; the green vertical dashed line 
#' indicates the position of the consistent correlation peak (if any); the blue vertical dashed line 
#' indicates the tolerance zone (determined by `Tx`)
#'
#' @param cmps.result a list generated by `extract_feature_cmps`. `cmps.result` is required to have
#' the following names: `parameters`, `congruent.pos`, `segments`, `nseg`, i.e. one should at least have  
#' `include = c("parameters", "congruent.pos",. "segments", "nseg")` when computing `cmps.result`. 
#' However, `include = "full_result` is still recommended.
#' @param seg.idx an integer. The index of a basis segment that we want to plot for. 
#'
#' @return a list of n elements, where n is the length of `npeaks.set`, i.e. the number of scales for 
#' each basis segment. And each one of these n elements is also a list, a list of two plots:
#' * `segment_plot`: The basis segment of current scale is plotted at different positions where the 
#' segment obtains correlation peak. The comparison signature is also plotted. 
#' * `scale_ccf_plot`: This is the plot of the cross-correlation curve between the comparison signature
#' and the segment of the current scale. 
#' 
#' @export
#' @importFrom assertthat has_name assert_that
#' @importFrom rlang .data
#' @import ggplot2
#'
#' @examples
#' library(tidyverse)
#' library(cmpsR)
#' library(ggpubr)
#' 
#' data("bullets")
#' land2_3 <- bullets$sigs[bullets$bulletland == "2-3"][[1]]
#' land1_2 <- bullets$sigs[bullets$bulletland == "1-2"][[1]]
#' 
#' # compute cmps
#' 
#' # algorithm with multi-peak insepction at three different segment scales
#' cmps_with_multi_scale <- extract_feature_cmps(land2_3$sig, land1_2$sig, include = "full_result" )
#' 
#' # generate plots using cmps_signature_plot
#' seg.plot <- cmps_segment_plot(cmps_with_multi_scale, seg.idx = 3)
#' 
#' pp <- ggarrange(plotlist = unlist(seg.plot, recursive = FALSE), nrow = 3, ncol = 2)
#' 
cmps_segment_plot <- function(cmps.result, seg.idx = 1){
  
  has_name(cmps.result, 
           c("parameters", "congruent.pos", "segments", "nseg"))
  
  assert_that(is.numeric(seg.idx))
  
  if(seg.idx < 1 | seg.idx > cmps.result$nseg) {
    stop(paste("seg.idx is invalid. Please choose an integer between 1 and", cmps.result$nseg))
  }
  
  
  # obtain results from cmps.result
  segments <- cmps.result$segments
  npeaks.set <- cmps.result$parameters$npeaks.set
  seg_levels <- length(npeaks.set)
  outlength_ <- cmps.result$parameters$outlength
  
  # compute the cross-correlation for each segment scale
  ccr.list <- lapply(1:seg_levels, function(seg_level) {
    
    get_ccr_peaks(cmps.result$parameters$y, segments, seg_outlength = outlength_[seg_level], 
                  nseg = seg.idx, npeaks = npeaks.set[seg_level])
    
    
  })
  
  # compute segments in different scales
  seg.scale.list <- lapply(1:seg_levels, function(seg_level) {
    
    get_seg_scale(segments, seg.idx, outlength_[seg_level])
    
  })
  
  # data frame for plotting the comparison profile
  plot.df.comp <- data.frame(
    x=1:length(cmps.result$parameters$y), 
    sig = cmps.result$parameters$y, 
    gp = 0
  )
  
  # a list of plots
  pp.list <- lapply(1:seg_levels, function(level.idx) {
    
    # plot the selected segment in all positions where
    # it obtains cross-correlation peaks
    tp.df1 <- seg.scale.list[[level.idx]] %>% as.data.frame()

    p1 <- ggplot(tp.df1) +
      geom_line(aes(x=.data$aug_idx, y=.data$aug_seg), color="black", size=1) +
      geom_line(data=plot.df.comp, aes(x=.data$x, y=.data$sig), color="grey44") +
      xlab("Position") +
      ylab("Profile Height") +
      ggtitle(paste("Plotting for scale level", level.idx, "of segment", seg.idx)) +
      theme_bw()
    
    p.df <- lapply(length(ccr.list[[level.idx]]$peaks.pos):1, function(i) {
      data.frame(x = seg.scale.list[[level.idx]]$aug_idx + ccr.list[[level.idx]]$peaks.pos[i],
                 y = seg.scale.list[[level.idx]]$aug_seg,
                 color = i)
    }) 
    
    p.df <- do.call(rbind, p.df)
    
    p1 <- p1 +
      geom_line(data = p.df,
                aes(x=.data$x, y=.data$y, color = .data$color, group = .data$color),
                size = 1, alpha = 1, show.legend = FALSE, linetype = "dashed")
    
    
    if(length(ccr.list[[level.idx]]$peaks.pos) == 1) {
      p1 <- p1 + scale_color_gradient(low="red", high="red")
    } else {
      p1 <- p1 + scale_color_gradient(low="red", high="blue") 
    }
    
    # plot the cross-correlation curve with peaks highlighted
    p2 <- data.frame(x=ccr.list[[level.idx]]$adj.pos, ccf=ccr.list[[level.idx]]$ccr$ccf) %>% 
      ggplot(aes(x=.data$x, y=.data$ccf))+
      geom_line() + 
      geom_point(
        data = data.frame(
          x = ccr.list[[level.idx]]$peaks.pos, 
          ccf = ccr.list[[level.idx]]$peaks.heights,
          size = seq(2, 1.2, length.out = length(ccr.list[[level.idx]]$peaks.pos))
        ),
        aes(colour=.data$size), show.legend = FALSE
      ) +
      xlim(min(ccr.list[[1]]$adj.pos), max(ccr.list[[1]]$adj.pos)) +
      geom_vline(xintercept = cmps.result$congruent.pos,
                 color = "red", linetype = "longdash") +
      scale_color_gradient(low="blue", high="red") + 
      xlab("Shift Position") + 
      ylab("Correlation Coefficient") + 
      ggtitle(paste("Plotting for scale level", level.idx, "of segment", seg.idx)) +
      theme_bw()
    
    if(length(cmps.result$parameters$npeaks.set) > 1) {
      p2 <- p2 +
        geom_vline(xintercept = ccr.list[[seg_levels]]$peaks.pos,
                   color = "darkgreen", linetype = "twodash") + 
        geom_vline(xintercept = c(ccr.list[[seg_levels]]$peaks.pos - cmps.result$parameters$Tx,
                                  ccr.list[[seg_levels]]$peaks.pos + cmps.result$parameters$Tx),
                   color = "blue", linetype = "dashed")
    }
    
    
    return(list(segment_plot = p1, scale_ccf_plot = p2))
    
  })
  
  return(pp.list)
}





























