% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/countrycode.R
\name{countrycode}
\alias{countrycode}
\title{Convert Country Codes}
\usage{
countrycode(
  sourcevar,
  origin,
  destination,
  warn = TRUE,
  nomatch = NA,
  custom_dict = NULL,
  custom_match = NULL,
  origin_regex = NULL
)
}
\arguments{
\item{sourcevar}{Vector which contains the codes or country names to be
converted (character or factor)}

\item{origin}{A string which identifies the coding scheme of origin (e.g., \code{"iso3c"}). See
\code{\link{codelist}} for a list of available codes.}

\item{destination}{A string or vector of strings which identify the coding
scheme of destination (e.g., \code{"iso3c"} or \code{c("cowc", "iso3c")}). See
\code{\link{codelist}} for a list of available codes. When users supply a
vector of destination codes, they are used sequentially to fill in
missing values not covered by the previous destination code in the
vector.}

\item{warn}{Prints unique elements from sourcevar for which no match was found}

\item{nomatch}{When countrycode fails to find a match for the code of
origin, it fills-in the destination vector with \code{nomatch}. The default
behavior is to fill non-matching codes with \code{NA}. If \code{nomatch = NULL},
countrycode tries to use the origin vector to fill-in missing values in the
destination vector. \code{nomatch} must be either \code{NULL}, of length 1, or of the same
length as \code{sourcevar}.}

\item{custom_dict}{A data frame which supplies a new dictionary to
replace the built-in country code dictionary. Each column
contains a different code and must include no duplicates. The
data frame format should resemble \code{\link{codelist}}. Users
can pre-assign attributes to this custom dictionary to affect
behavior (see Examples section):
\itemize{
\item "origin.regex" attribute: a character vector with the names
of columns containing regular expressions.
\item "origin.valid" attribute: a character vector with the names
of columns that are accepted as valid origin codes.
}}

\item{custom_match}{A named vector which supplies custom origin and
destination matches that will supercede any matching default result. The name
of each element will be used as the origin code, and the value of each
element will be used as the destination code.}

\item{origin_regex}{NULL or Logical: When using a custom
dictionary, if TRUE then the origin codes will be matched as
regex, if FALSE they will be matched exactly. When NULL,
\code{countrycode} will behave as TRUE if the origin name is in the
\code{custom_dictionary}'s \code{origin_regex} attribute, and FALSE
otherwise. See examples section below.}
}
\description{
Converts long country names into one of many different coding schemes.
Translates from one scheme to another. Converts country name or coding
scheme to the official short English country name. Creates a new variable
with the name of the continent or region to which each country belongs.
}
\note{
For a complete description of available country codes and languages,
please see the documentation for the \code{\link{codelist}} conversion
dictionary.

Panel data (i.e., country-year) can pose particular problems when
converting codes. For instance, some countries like Vietnam or Serbia go
through political transitions that justify changing codes over time. This
can pose problems when using codes from organizations like CoW or Polity IV,
which produce codes in country-year format. Instead of converting codes
using \code{countrycode()}, we recommend that users use the
\code{\link{codelist_panel}} data.frame as a base into which they can
merge their other data. This data.frame includes most relevant code, and is
already "reconciled" to ensure that each political unit is only represented
by one row in any given year. From there, it is just a matter of using \code{\link[=merge]{merge()}}
to combine different datasets which use different codes.
}
\examples{
library(countrycode)

# ISO to Correlates of War
countrycode(c('USA', 'DZA'), origin = 'iso3c', destination = 'cown')

# English to ISO
countrycode('Albania', origin = 'country.name', destination = 'iso3c')

# German to French
countrycode('Albanien', origin = 'country.name.de', destination = 'iso.name.fr')

# Using custom_match to supercede default codes
countrycode(c('United States', 'Algeria'), 'country.name', 'iso3c')
countrycode(c('United States', 'Algeria'), 'country.name', 'iso3c',
            custom_match = c('Algeria' = 'ALG'))

\dontrun{
 # Download the dictionary of US states from Github
 state_dict <- "https://bit.ly/2ToSrFv"
 state_dict <- read.csv(state_dict)

 # The "state.regex" column includes regular expressions, so we set an attribute.
 attr(state_dict, "origin_regex") <- "state.regex"
 countrycode(c('AL', 'AK'), 'abbreviation', 'state',
             custom_dict = state_dict)
 countrycode(c('Alabama', 'North Dakota'), 'state.regex', 'state',
             custom_dict = state_dict)
}
}
\keyword{countrycode}
