context("SystemMetadata")
test_that("datapack library loads", {
	library(datapack)
})
test_that("SystemMetadata constructors", {
    library(datapack)
    sysmeta <- new("SystemMetadata")
    expect_that(sysmeta@serialVersion, equals(1))
    expect_that(is.na(sysmeta@identifier), is_true())
    sysmeta <- new("SystemMetadata", identifier="TestId", formatId="text/csv")
    expect_that(sysmeta@identifier, equals("TestId"))
    expect_that(sysmeta@formatId, equals("text/csv"))
})
test_that("XML SystemMetadata parsing works", {
  library(datapack)
  library(XML)
  testid <- "doi:10.xxyy/AA/tesdoc123456789"
  sysmeta <- new("SystemMetadata")
  expect_that(sysmeta@serialVersion, equals(1))
  doc <- xmlParseDoc("../testfiles/sysmeta.xml", asText=FALSE)
  expect_that(xmlValue(xmlRoot(doc)[["identifier"]]), matches(testid))
  xml <- xmlRoot(doc)
  #getEncoding(doc)
  sysmeta <- parseSystemMetadata(sysmeta, xmlRoot(xml))
  expect_that(sysmeta@identifier, matches(testid))
  expect_that(nrow(sysmeta@accessPolicy), equals(5))
  expect_that(as.character(sysmeta@accessPolicy$permission[[1]]), matches("read"))
  expect_that(sysmeta@archived, is_true())
  csattrs <- xmlAttrs(xml[["checksum"]])
  expect_that(sysmeta@checksumAlgorithm, matches(csattrs[[1]]))
  expect_that(grep("urn:node:KNB", sysmeta@preferredNodes) > 0, is_true())
  expect_that(grep("urn:node:mnUNM1", sysmeta@preferredNodes) > 0, is_true())
  expect_that(grep("urn:node:BADNODE", sysmeta@blockedNodes) > 0, is_true())
  rm(sysmeta) 
  rm(xml)
  rm(doc)
  rm(csattrs)
  # Parse v2.0 system metadata
  testid <- "0007f892-0d8f-4451-94e9-94d02ba5dd0d_0"
  sysmeta <- new("SystemMetadata")
  expect_that(sysmeta@serialVersion, equals(1))
  doc <- xmlParseDoc("../testfiles/sysmeta-v2.xml", asText=FALSE)
  expect_that(xmlValue(xmlRoot(doc)[["identifier"]]), matches(testid))
  xml <- xmlRoot(doc)
  sysmeta <- parseSystemMetadata(sysmeta, xmlRoot(xml))
  expect_that(sysmeta@identifier, matches(testid))
  expect_that(nrow(sysmeta@accessPolicy), equals(1))
  expect_that(as.character(sysmeta@accessPolicy$permission[[1]]), matches("read"))
  expect_false(sysmeta@archived)
  csattrs <- xmlAttrs(xml[["checksum"]])
  expect_that(sysmeta@checksumAlgorithm, matches(csattrs[[1]]))
  expect_equal(sysmeta@seriesId, "3")
  expect_equal(sysmeta@mediaType, "text/csv")
  expect_equal(sysmeta@fileName, "testData.csv")
  
})

test_that("XML SystemMetadata serialization works", {
    library(datapack)
    library(XML)
    testid <- "doi:10.xxyy/AA/tesdoc123456789"
    sysmeta <- new("SystemMetadata")
    expect_that(sysmeta@serialVersion, equals(1))
    xml <- xmlParseDoc("../testfiles/sysmeta.xml", asText=FALSE)
    expect_that(xmlValue(xmlRoot(xml)[["identifier"]]), matches(testid))
    sysmeta <- parseSystemMetadata(sysmeta, xmlRoot(xml))
    expect_that(sysmeta@identifier, matches(testid))
    expect_that(sysmeta@archived, is_true())
    # Check if the access policy is serialized grouped by subjects
    sysmeta <- addAccessRule(sysmeta, "bob", "read")
    sysmeta <- addAccessRule(sysmeta, "alice", "read")
    sysmeta <- addAccessRule(sysmeta, "bob", "write")
    sysmeta <- addAccessRule(sysmeta, "alice", "write")
    # Add an existing rule, to ensure that rules aren't duplicated in the serialized sysmeta
    sysmeta <- addAccessRule(sysmeta, "CN=Subject2,O=Google,C=US,DC=cilogon,DC=org", "write")
    xml <- serializeSystemMetadata(sysmeta)
    # Compare the updated, serialized sysmeta with a reference
    xmlRef <- xmlParseDoc("../testfiles/sysmeta-updated.xml", asText=FALSE)
    sysmetaRef <- new("SystemMetadata")
    sysmetaUpdated <- parseSystemMetadata(sysmetaRef, xmlRoot(xmlRef))
    xmlRef <- serializeSystemMetadata(sysmetaUpdated)
    expect_equal(xml, xmlRef)
    #cat(xml)
    # Search for specific, expected items in the serialized sysmeta
    expect_that(xml, matches("<d1:systemMetadata"))
    expect_that(xml, matches("<blockedMemberNode>urn:node:BADNODE</blockedMemberNode>"))
    expect_that(xml, matches("<preferredMemberNode>urn:node:KNB</preferredMemberNode>"))
    expect_that(xml, matches("<subject>public</subject>"))
    expect_that(xml, matches("<permission>read</permission>"))
    expect_that(xml, matches("<subject>CN=Subject2,O=Google,C=US,DC=cilogon,DC=org</subject>"))
    expect_that(xml, matches("<permission>changePermission</permission>"))
    sysmeta@obsoletes <- ""
    sysmeta <- new("SystemMetadata")
    xml <- serializeSystemMetadata(sysmeta)
    foundObsoletes <- grep("<obsoletes>", xml, invert=TRUE)
    expect_that(as.logical(foundObsoletes), is_true())
    # TODO: check tree equivalence with original XML document
})
test_that("SystemMetadata XML constructor works", {
    library(datapack)
    testid <- "doi:10.xxyy/AA/tesdoc123456789"
    doc <- xmlParseDoc("../testfiles/sysmeta.xml", asText=FALSE)
    expect_that(xmlValue(xmlRoot(doc)[["identifier"]]), matches(testid))
    xml <- xmlRoot(doc)
    sysmeta <- SystemMetadata(xmlRoot(xml))
    expect_that(sysmeta@identifier, matches(testid))
    expect_that(nrow(sysmeta@accessPolicy), equals(5))
    expect_that(as.character(sysmeta@accessPolicy$permission[[1]]), matches("read"))
    expect_that(sysmeta@archived, is_true())
    csattrs <- xmlAttrs(xml[["checksum"]])
    expect_that(sysmeta@checksumAlgorithm, matches(csattrs[[1]]))
    expect_that(grep("urn:node:KNB", sysmeta@preferredNodes) > 0, is_true())
    expect_that(grep("urn:node:mnUNM1", sysmeta@preferredNodes) > 0, is_true())
    expect_that(grep("urn:node:BADNODE", sysmeta@blockedNodes) > 0, is_true())
})
test_that("SystemMetadata validation works", {
    library(datapack)
    sysmeta <- new("SystemMetadata", identifier="foo", formatId="text/csv", size=59, checksum="jdhdjhfd", rightsHolder="ff")
    isValid <- validate(sysmeta)
    expect_that(isValid, is_true())
    isValid <- validObject(sysmeta)
    expect_that(isValid, is_true())
    sysmeta <- new("SystemMetadata", identifier="foo", checksum="jdhdjhfd", rightsHolder="ff")
    errors <- validate(sysmeta)
    expect_that(length(errors), equals(2))
})

test_that("SystemMetadata accessPolicy can be constructed and changed", {
    apolicy=data.frame(list("public", "read"))
    colnames(apolicy) <- c("subject", "permission")
    sysmeta <- new("SystemMetadata", identifier="foo", formatId="text/csv", size=59, checksum="jdhdjhfd", rightsHolder="ff", accessPolicy=apolicy)
    expect_that(sysmeta@serialVersion, equals(1))
    expect_that(nrow(sysmeta@accessPolicy), equals(1))
    expect_that(as.character(sysmeta@accessPolicy$permission[[1]]), matches("read"))
    
    sysmeta <- addAccessRule(sysmeta, "foo", "write")
    expect_that(nrow(sysmeta@accessPolicy), equals(2))
    # Try to add same rule again and make sure it didn't get duplicated
    sysmeta <- addAccessRule(sysmeta, "foo", "write")
    expect_that(nrow(sysmeta@accessPolicy), equals(2))
    expect_that(as.character(sysmeta@accessPolicy$permission[[2]]), matches("write"))
    expect_true(hasAccessRule(sysmeta, "foo", "write"))
    apolicy=data.frame(subject=c("bar", "baz"), permission= c("changePermission", "write"))
    sysmeta <- addAccessRule(sysmeta, apolicy)
    # Check that specific rules were added (also testing hasAccessRule method)
    expect_true(hasAccessRule(sysmeta, "foo", "write"))
    expect_true(hasAccessRule(sysmeta, "bar", "changePermission"))
    expect_true(hasAccessRule(sysmeta, "baz", "write"))
    expect_true(!hasAccessRule(sysmeta, "baz", "changePermission"))
    expect_that(nrow(sysmeta@accessPolicy), equals(4))
    expect_that(as.character(sysmeta@accessPolicy$permission[[4]]), matches("write"))
    expect_that(as.character(sysmeta@accessPolicy$subject[[4]]), matches("baz"))    
})

