% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distcov_function.R
\name{distcov}
\alias{distcov}
\title{Calculates the distance covariance \insertCite{szekely2007,szekely2009brownian}{dcortools}.}
\usage{
distcov(
  X,
  Y,
  affine = FALSE,
  standardize = FALSE,
  bias.corr = FALSE,
  type.X = "sample",
  type.Y = "sample",
  metr.X = "euclidean",
  metr.Y = "euclidean",
  use = "all",
  algorithm = "auto"
)
}
\arguments{
\item{X}{contains either the first  sample or its corresponding distance matrix.

In the first case, X can be provided either as a vector (if one-dimensional), a matrix or a data.frame (if two-dimensional or higher). 

In the second case, the input must be a distance matrix corresponding to the sample of interest.

If X is a sample, type.X must be specified as "sample". If X is a distance matrix, type.X must be specified as "distance".}

\item{Y}{see X.}

\item{affine}{logical; specifies if the affinely invariant distance covariance \insertCite{dueck2014affinely}{dcortools} should be calculated or not.}

\item{standardize}{logical; specifies if X and Y should be standardized dividing each component by its standard deviations. No effect when affine = TRUE.}

\item{bias.corr}{logical; specifies if the bias corrected version of the sample distance covariance \insertCite{huo2016fast}{dcortools} should be calculated.}

\item{type.X}{For "distance", X is interpreted as a distance matrix. For "sample", X is interpreted as a sample.}

\item{type.Y}{see type.X.}

\item{metr.X}{specifies the metric which should be used to compute the distance matrix for X (ignored when type.X = "distance").

 Options are "euclidean", "discrete", "alpha", "minkowski", "gaussian", "gaussauto", "boundsq" or user-specified metrics (see examples).
 
 For "alpha", "minkowski", "gaussian", "gaussauto" and "boundsq", the corresponding parameters are specified via "c(metric, parameter)", e.g. c("gaussian", 3) for a Gaussian metric with bandwidth parameter 3; the default parameter is 2 for "minkowski" and "1" for all other metrics.
 
 See \insertCite{lyons2013distance,sejdinovic2013equivalence,bottcher2017detecting;textual}{dcortools} for details.}

\item{metr.Y}{see metr.X.}

\item{use}{specifies how to treat missing values. "complete.obs" excludes observations containing NAs, "all" uses all observations.}

\item{algorithm}{specifies the algorithm used for calculating the distance covariance. 

"fast" uses an O(n log n) algorithm if the observations are one-dimensional and metr.X and metr.Y are either "euclidean" or "discrete", see also \insertCite{huo2016fast;textual}{dcortools}. 

"memsave" uses a memory saving version of the standard algorithm with computational complexity O(n^2) but requiring only O(n) memory. 

"standard" uses the classical algorithm. User-specified metrics always use the classical algorithm.

"auto" chooses the best algorithm for the specific setting using a rule of thumb.}
}
\value{
numeric; the distance covariance between samples X and Y.
}
\description{
Calculates the distance covariance \insertCite{szekely2007,szekely2009brownian}{dcortools}.
}
\examples{
X <- rnorm(100)
Y <- X + 3 * rnorm(100)
distcov(X, Y) # standard distance covariance

distcov(X, Y, metr.X = "gaussauto", metr.Y = "gaussauto") 
# Gaussian distance with bandwidth choice based on median heuristic

distcov(X, Y, metr.X = c("alpha", 0.5), metr.Y = c("alpha", 0.5)) 
# alpha distance covariance with alpha = 0.5.


#Define a user-specified (slow) version of the alpha metric

alpha_user <- function(X, prm = 1, kernel = FALSE) {
    as.matrix(dist(X)) ^ prm
}

distcov(X, Y, metr.X = c("alpha", 0.5), metr.Y = c("alpha", 0.5)) 
# Gives the same result as before.
   

#User-specified Gaussian kernel function  
     
gauss_kernel <- function(X, prm = 1, kernel = TRUE)  {
    exp(as.matrix(dist(X)) ^ 2 / 2 / prm ^ 2)
}  

distcov(X, Y, metr.X = c("gauss_kernel", 2), metr.Y = c("gauss_kernel", 2)) 
# calculates the distance covariance using the corresponding kernel-induced metric

distcov(X, Y, metr.X = c("gaussian", 2), metr.Y = c("gaussian", 2)) 
# same result

Y <- matrix(nrow = 100, ncol = 2)
X <- rnorm(300)
dim(X) <- c(100, 3)
Z <- rnorm(100)
Y <- matrix(nrow = 100, ncol = 2)
Y[, 1] <- X[, 1] + Z
Y[, 2] <- 3 * Z

distcov(X, Y) 

distcov(X, Y, affine = TRUE) 
# affinely invariant distance covariance

distcov(X, Y, standardize = TRUE) 
## distance covariance standardizing the components of X and Y

}
\references{
\insertRef{bottcher2017detecting}{dcortools}

\insertRef{dueck2014affinely}{dcortools}

\insertRef{huo2016fast}{dcortools}

\insertRef{lyons2013distance}{dcortools}

\insertRef{sejdinovic2013equivalence}{dcortools}

\insertRef{szekely2007}{dcortools}

\insertRef{szekely2009brownian}{dcortools}
}
